#ifndef CONFIG_H_
#define CONFIG_H_

#include "mbed.h"
#include "utils.h"

#define CMD_CHAR_HELP '?'

namespace config
{   
    struct CommandResponder
    {
        virtual ~CommandResponder() { }
        virtual bool parse(const char& c)=0;
        virtual bool writeSettings()=0;
        virtual void echoback()=0;
    };

    void addCommand(const char& command, CommandResponder* resp);
    void removeCommand(const char& command);
    
    void handleSerial();
    
    void writeSettingsToSerial();
}

template<typename V>
struct Property: public config::CommandResponder
{
    explicit Property(const char& command, const V& defaultValue):
        command(command), value(defaultValue)
    {
        config::addCommand(command, this);
    }
    
    Property(); // not allowed
    
    virtual ~Property(){
        config::removeCommand(command);
    }
    
    /**
     *  returns: if this property accepts this command
     */
    virtual bool parse(const char& c)
    {
        if (c != command) {
            return false;
        }
        
        value = parseUnsignedFromSerial<V>(value); // TODO: deal with signedness?
        return true;
    }
    
    virtual bool writeSettings()
    {
        IO::write("%c%u",command,value); // TODO: deal with signedness?
        return true;
    }
    
    virtual void echoback()
    {
        IO::write(IO::CONFIG_HEADER);
        writeSettings();
        IO::write("\r\n");
    }
    
    const char  command;
    V           value;
};

class Action: public config::CommandResponder
{
public:
    explicit Action(const char& command, Callback<void()> f);
    
    virtual ~Action();
    
    virtual bool parse(const char& c);
    
    virtual bool writeSettings();
    
    virtual void echoback();

private:
    const char          command_;
    Callback<void()>    handler_;
};

#endif