/* Copyright (c) 2010-2011 mbed.org, MIT License
*
* Permission is hereby granted, free of charge, to any person obtaining a copy of this software
* and associated documentation files (the "Software"), to deal in the Software without
* restriction, including without limitation the rights to use, copy, modify, merge, publish,
* distribute, sublicense, and/or sell copies of the Software, and to permit persons to whom the
* Software is furnished to do so, subject to the following conditions:
*
* The above copyright notice and this permission notice shall be included in all copies or
* substantial portions of the Software.
*
* THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED, INCLUDING
* BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
* NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM,
* DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
* OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
*/

#ifndef USBWEBUSBSERIAL_H
#define USBWEBUSBSERIAL_H

#include <string>
#include "USBDevice.h"
#include "CircBuffer.h"
#include "USBEndpoints.h"
#include "USBDescriptor.h"
#include "USBDevice_Types.h"

#ifndef USBWEBUSBSERIAL_INPUT_BUFFER_SIZE
#define USBWEBUSBSERIAL_INPUT_BUFFER_SIZE 128
#endif

#ifndef USBWEBUSBSERIAL_OUTPUT_BUFFER_SIZE
#define USBWEBUSBSERIAL_OUTPUT_BUFFER_SIZE 512
#endif

/**
* WebUSB/USBSerial composite device class
* Handles outputting to standard CDC or WebUSB endpoint depending on mode
*
*/
class USBWebUSBSerial: public USBDevice
{
public:

    /**
    *   Constructor
    *
    * @param vendor_id Your vendor_id
    * @param product_id Your product_id
    * @param product_release Your product_release (default: 0x0001)
    * @param connect_blocking define if the connection must be blocked if USB not plugged in
    *
    */
    USBWebUSBSerial(uint16_t vendor_id, uint16_t product_id, uint16_t product_release =  0x0001, bool connect_blocking = true);
    virtual ~USBWebUSBSerial();

    const std::string& manufacturerName() const { return m_manufacturerName; }
    void setManufacturerName(const std::string &manufacturerName);

    const std::string& productName() const { return m_productName; }
    void setProductName(const std::string &productName);

    const std::string& serialNumber() const { return m_serialNumber; }
    void setSerialNumber(const std::string &serialNumber);

    /**
    * Read up to size bytes from input buffer
    *
    * @returns the number of bytes read
    */

    int read(uint8_t *buf, uint16_t size);

    /**
    * Check the number of bytes available.
    *
    * @returns the number of bytes available
    */
    uint8_t available();

    /** Determine if there is a character available to read
     *
     *  @returns
     *    1 if there is a character available to read,
     *    0 otherwise
     */
    int readable() { return available() ? 1 : 0; }

    /** Determine if there is space available to write a character
     *
     *  @returns
     *    1 if there is space to write a character,
     *    0 otherwise
     */
    int writeable() { return 1; } // Always return 1 since we use a circular buffer

    /**
    * Write data asynchronously using internal output buffer
    *
    * @param buf pointer to data 
    * @param size size of the buffer. The maximum size of data is limited by the size of the internal buffer
    *
    * @returns true if successfull
    */
    bool writeBuffered(const uint8_t * buf, uint16_t size);

    /**
     *  Attach a member function to call when a packet is received.
     *
     *  @param tptr pointer to the object to call the member function on
     *  @param mptr pointer to the member function to be called
     */
    template<typename T>
    void attach(T* tptr, void (T::*mptr)(void))
    {
        if (mptr && tptr)
        {
            m_rx.attach(tptr, mptr);
        }
    }

    /**
     * Attach a callback called when a packet is received
     *
     * @param fptr function pointer
     */
    void attach(void (*fptr)(void))
    {
        if (fptr)
        {
            m_rx.attach(fptr);
        }
    }

    /**
     * Attach a callback to call when serial's settings are changed.
     *
     * @param fptr function pointer
     */
    void attach(void (*fptr)(int baud, int bits, int parity, int stop))
    {
        m_settingsChangedCallback = fptr;
    }

protected:

    /*
    * Get device descriptor. Warning: this method has to store the length of the report descriptor in reportLength.
    *
    * @returns pointer to the device descriptor
    */
    virtual uint8_t * deviceDesc();

    /*
    * Get string manufacturer descriptor
    *
    * @returns pointer to the string manufacturer descriptor
    */
    virtual uint8_t * stringImanufacturerDesc();

    /*
    * Get string product descriptor
    *
    * @returns pointer to the string product descriptor
    */
    virtual uint8_t * stringIproductDesc();

    /*
    * Get string serial descriptor
    *
    * @returns pointer to the string serial descriptor
    */
    virtual uint8_t * stringIserialDesc();

    /*
    * Get configuration descriptor
    *
    * @returns pointer to the configuration descriptor
    */
    virtual uint8_t * configurationDesc();

    /*
    * Get BOS descriptor
    *
    * @returns pointer to the BOS descriptor
    */    
    virtual uint8_t * bosDesc();

    /*
    * Write from internal buffer to active endpoint
    */
    void writeToActiveEndpoint();

    /*
    * Read from the active endpoint into the internal buffer (non-blocking)
    *
    * @returns true if successful
    */
    bool readActiveEP();

    /*
    * Called by USBCallback_requestCompleted when CDC line coding is changed
    * Warning: Called in ISR
    *
    * @param baud The baud rate
    * @param bits The number of bits in a word (5-8)
    * @param parity The parity
    * @param stop The number of stop bits (1 or 2)
    */
    virtual void lineCodingChanged(int baud, int bits, int parity, int stop)
    {
        if (m_settingsChangedCallback)
        {
            m_settingsChangedCallback(baud, bits, parity, stop);
        }
    }

    virtual bool USBCallback_request();
    virtual void USBCallback_requestCompleted(uint8_t *buf, uint32_t length);
    virtual bool USBCallback_setConfiguration(uint8_t configuration);
    virtual void USBCallback_busReset(void);

    virtual bool EPCDC_OUT_callback();
    virtual bool EPWEBUSB_OUT_callback();
    virtual bool EPOUTCallbackHandler(uint8_t endpoint);

    virtual bool EPCDC_IN_callback();
    virtual bool EPWEBUSB_IN_callback();

private:
    uint8_t activeOutEndpoint() const { return m_webUSBMode ? EPWEBUSB_OUT : EPCDC_OUT; }
    uint8_t activeInEndpoint() const { return m_webUSBMode ? EPWEBUSB_IN : EPCDC_IN; }
    void createDynamicDescriptors();
    uint8_t *createStringDescriptor(const std::string &string) const;
    void setWebUSBMode(bool webUSBMode);
    uint32_t timeSinceWrite() const;

    static uint8_t s_MSOS2Descriptor[];
    volatile bool m_terminalConnected;
    FunctionPointer m_rx;
    CircBuffer<uint8_t,USBWEBUSBSERIAL_INPUT_BUFFER_SIZE> m_inputBuffer;
    CircBuffer<uint8_t,USBWEBUSBSERIAL_OUTPUT_BUFFER_SIZE> m_outputBuffer;
    void (*m_settingsChangedCallback)(int baud, int bits, int parity, int stop) = nullptr;

    bool m_pendingWrite = false;
    uint32_t m_lastWriteTime = 0;
    std::string m_manufacturerName = "mbed.org";
    uint8_t *m_manufacturerStringDesc = nullptr;
    std::string m_productName = "CDC/WebUSB Device";
    uint8_t *m_productStringDesc = nullptr;
    std::string m_serialNumber = "0123456789";
    uint8_t *m_serialStringDesc = nullptr;

    bool m_webUSBMode = false;
};

#endif
