/* Pmod Interface Library
 *
 */
#ifndef PMODINTERFACE_H
#define PMODINTERFACE_H

#include "mbed.h"
#include "PmodInterface.h"
#include "MAX14661.h"

/** RAPC Library, Provides utilities for remotely accessing peripherals
 *
 * Example:
 * @code
 * // Configure board to pass UART signals to peripheral connector.
 *
 * #include "PmodInterface.h"
 *
 * PmodInterface pInt(PTD0, PTD2, PTD3, PTD1,
 *          PTA4, PTA5, PTC8, PTC9,
 *          PTA12, PTE0, PTE1
 *          );
 *
 * int main() {
 *    char ibuf[256];
 *    char obuf[256];
 *    pInt.init();
 *    while(1) {
 *        scanf("%s", ibuf);
 *        pInt.call(ibuf, obuf);
 *        printf("%s=", ibuf);
 *        printf("%s\n", obuf);
 * }
 * @endcode
 */
class PmodInterface
{
public:

    /** Create a PmodInterface interface
     *
     */
    PmodInterface();

    ~PmodInterface();

    /** Name the DIO bits
    */
    enum PINTdioBits {
        DB_OUT =  0x01,   /**< Pin Output State Bit */
        DB_OWD =  0x02,   /**< Output Write Disable, set to block writing the output state */
        DB_DIR =  0x04,   /**< Pin Direction, 0 = Input, 1 = Output */
        DB_DWE =  0x08,   /**< Direction Write Enable, set this bit to write the direction  */
        DB_PU  =  0x10,   /**< Pull Up State, 0 = PullNone, 1 = PullUp */
        DB_PWE =  0x20,   /**< Pull Up Write Enable, set this bit to write the pull up state  */
    };

    /** Name the I2C arguments
    */
    enum PINTi2cArgs {
        IA_CNT = 0, /**< Argument Count */
        IA_ADD,     /**< Device Address */
        IA_DATA,    /**< Data, Read = # bytes to read, Write = first data byte */
        IA_RDDA     /**< Read Data, data to write prior to read */
    };

    /** Initialize the digital pins and PWM
     *
     */
    void init(DigitalInOut *dio[], I2C* i2c, MAX14661* mux, const int* mux_a, const int* mux_p);

    /** Process Remote Arduino Peripheral Module Command
     *
     *  @param input a pointer to the string containing the command
     *  @param output a pointer to the string to write the result
     */
    void call(char* input, char* output);

private:

    // Internal Functions

    /** Process Digital I/O Command
     *  
     *  @param resp a pointer to the string to write the result
     */
    void fnc_dio(char* resp);

    /** Process I2C Command
     *  
     *  @param resp a pointer to the string to write the result
     */
    void fnc_i2c(char* resp);

    /** Process Multiplexer Command
     *  
     *  @param resp a pointer to the string to write the result
     */
    void fnc_mux(char* resp);


    /** Process SPI Command
     *  
     *  @param resp a pointer to the string to write the result
     */
    void fnc_spi(char* resp);

    // Internal Resources
    I2C *_i2c;
    MAX14661 *_mux;
    const int *_mux_a;  // mux micro map LUT
    const int *_mux_p;  // mux pmod map LUT

    // Array of pointers to the DIO pins
    DigitalInOut **_pmd;

    // Internal Buffers
    int _args[32];
    char _dbuf[64];

};

#endif