/* MAX14661 Driver Library
 *
 */

#ifndef MAX14661_H
#define MAX14661_H

#include "mbed.h"

/** MAX14661 Library, Provides utilities for configuring the MAX14661 over I2C
 *
 * Example:
 * @code
 * // Enable only switch B3 and read back switch state.
 *
 * #include "MAX14661.h"
 *
 * MAX14661 mux(p28, p27);
 *
 * int main() {
 *     mux.setAB(0x0000, MAX14661::SW03);
 *     printf("mux = 0x%08X\n", mux.read());
 *     mux.clearAll();
 *     printf("mux = 0x%08X\n", mux.read());
 * }
 * @endcode
 */
class MAX14661
{
public:

    /** Create a MAX14661 interface
     *
     * @param sda I2C data line pin
     * @param scl I2C clock line pin
     * @param addr MAX14661 I2C address
     */
    MAX14661(PinName sda, PinName scl, int addr = 0x98);

    ~MAX14661();

    /** Name the register addresses
    */
    enum MAX14661regs {
        REG_DIR0 = 0x00,  /**< 8A-1A Direct Access */
        REG_DIR1,         /**< 16A-9A Direct Access */
        REG_DIR2,         /**< 8B-1B Direct Access */
        REG_DIR3,         /**< 16B-9B Direct Access */
        REG_SHDW0 = 0x10, /**< 8A-1A Shadow */
        REG_SHDW1,        /**< 16A-9A Shadow */
        REG_SHDW2,        /**< 8B-1B Shadow */
        REG_SHSW3,        /**< 16B-9B Shadow */
        REG_CMD_A,        /**< Command A */
        REG_CMD_B         /**< Command A */
    };

    /** Name the command codes
    */
    enum MAX14661cmds {
        CMD_EN01 = 0x00,  /**< Enable switch 1 */
        CMD_EN02,         /**< Enable switch 2 */
        CMD_EN03,         /**< Enable switch 3 */
        CMD_EN04,         /**< Enable switch 4 */
        CMD_EN05,         /**< Enable switch 5 */
        CMD_EN06,         /**< Enable switch 6 */
        CMD_EN07,         /**< Enable switch 7 */
        CMD_EN08,         /**< Enable switch 8 */
        CMD_EN09,         /**< Enable switch 9 */
        CMD_EN10,         /**< Enable switch 10 */
        CMD_EN11,         /**< Enable switch 11 */
        CMD_EN12,         /**< Enable switch 12 */
        CMD_EN13,         /**< Enable switch 13 */
        CMD_EN14,         /**< Enable switch 14 */
        CMD_EN15,         /**< Enable switch 15 */
        CMD_EN16,         /**< Enable switch 16 */
        CMD_DIS,          /**< Disable switches  */
        CMD_COPY,         /**< Copy shadow registers to switches */
        CMD_NOOP = 0x1F   /**< Keep current state, no changes */
    };

    /** Name the switch bits
    */
    enum MAX14661sws {
        SW01 = (1 << 0),  /**< Bit mask for switch 1 */
        SW02 = (1 << 1),  /**< Bit mask for switch 2 */
        SW03 = (1 << 2),  /**< Bit mask for switch 3 */
        SW04 = (1 << 3),  /**< Bit mask for switch 4 */
        SW05 = (1 << 4),  /**< Bit mask for switch 5 */
        SW06 = (1 << 5),  /**< Bit mask for switch 6 */
        SW07 = (1 << 6),  /**< Bit mask for switch 7 */
        SW08 = (1 << 7),  /**< Bit mask for switch 8 */
        SW09 = (1 << 8),  /**< Bit mask for switch 9 */
        SW10 = (1 << 9),  /**< Bit mask for switch 10 */
        SW11 = (1 << 10), /**< Bit mask for switch 11 */
        SW12 = (1 << 11), /**< Bit mask for switch 12 */
        SW13 = (1 << 12), /**< Bit mask for switch 13 */
        SW14 = (1 << 13), /**< Bit mask for switch 14 */
        SW15 = (1 << 14), /**< Bit mask for switch 15 */
        SW16 = (1 << 15)  /**< Bit mask for switch 16 */
    };

    /** Clears all bits to opens all 32 switches
     */
    void clearAll();

    /** Set all 32 switches simultaneously
     *
     *  @param swA the desired state of switches [A16 - A01]
     *  @param swB the desired state of switches [B16 - B01]
     */
    void setAB(int swA, int swB);

    /** Read the status of all 32 switches concatenated into a single int
     *
     *  @returns
     *    the switch states [B16-B01,A16-A1]
     */
    int read();

private:
    I2C _i2c;
    int _addr;
};

#endif