/* ARD2PMD Board Driver Library
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy of this software
 * and associated documentation files (the "Software"), to deal in the Software without restriction,
 * including without limitation the rights to use, copy, modify, merge, publish, distribute,
 * sublicense, and/or sell copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all copies or
 * substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED, INCLUDING
 * BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 * NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM,
 * DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 */

#ifndef ARD2PMD_H
#define ARD2PMD_H

#include "mbed.h"
#include "MAX14661.h"


/** ARD2PMD Library, Provides utilities for configuring the Maxim ARD2PMD Board
 *
 * Example:
 * @code
 * // Configure board to pass UART signals to peripheral connector.
 *
 * #include "ARD2PMD.h"
 *
 * ARD2PMD a2p();
 *
 * int main() {
 *     a2p.mux.setAB((ARD2PMD::RX | ARD2PMD::PA3), (ARD2PMD::TX | ARD2PMD::PA2));
 * }
 * @endcode
 */
class ARD2PMD
{
public:

    /** Create a ARD2PMD interface
     *
     * @param sda I2C data line pin
     * @param scl I2C clock line pin
     * @param addr MAX14661 I2C address
     */
    ARD2PMD();

    ~ARD2PMD();

    /** Initialize the digital pins and PWM
     *
     */
    void init();

    /** ard2pmd resources mux, digital IO
    */
    MAX14661 mux;
    DigitalInOut pa1;
    DigitalInOut pa2;
    DigitalInOut pa3;
    DigitalInOut pa4;
    DigitalInOut pb1;
    DigitalInOut pb2;
    DigitalInOut pb3;
    DigitalInOut pb4;

    /** Array of pointers to the DIO pins
    */
    DigitalInOut *pmd[8];

    /** LUT for multiplexer micro connections
    */
    static const int mux_a[17];

    /** LUT for multiplexer PMOD connections
    */
    static const int mux_p[9];

    /** Name the register addresses
    */
    enum MUXsignals {
        RX = MAX14661::SW01,  /**< UART Receive */
        TX = MAX14661::SW02,  /**< UART Transmit */
        D2 = MAX14661::SW03,  /**< Digital 2 */
        D3 = MAX14661::SW04,  /**< Digital 3 */
        PB1 = MAX14661::SW05, /**< Pmd row B pin 1 */
        PB2 = MAX14661::SW06, /**< Pmd row B pin 2 */
        PB3 = MAX14661::SW07, /**< Pmd row B pin 3 */
        PB4 = MAX14661::SW08, /**< Pmd row B pin 4 */
        D8 = MAX14661::SW16,  /**< Digital 8 */
        D9 = MAX14661::SW15,  /**< Digital 9 */
        PA1 = MAX14661::SW14, /**< Pmd row A pin 1 */
        PA2 = MAX14661::SW13, /**< Pmd row A pin 1 */
        PA3 = MAX14661::SW12, /**< Pmd row A pin 1 */
        PA4 = MAX14661::SW11, /**< Pmd row A pin 1 */
        SDA = MAX14661::SW10, /**< I2C Data */
        SCL = MAX14661::SW09  /**< I2C Clock */
    };

};

#endif