/**
 * Gainspan wi-fi module library for mbed
 * Copyright (c) 2012 gsfan
 * Released under the MIT License: http://mbed.org/license/mit
 */

/** @file
 * @brief Gainspan wi-fi module library for mbed
 * GS1011MIC, GS1011MIP, GainSpan WiFi Breakout, etc.
 * module configuration: ATB=115200
 */

#ifndef _GSWIFI_H_
#define _GSWIFI_H_

#include "mbed.h"
#include "GSwifi_net.h"
#include "RingBuffer.h"
#include "host.h"
#include "ipaddr.h"

#define GS_BAUD 115200

#define GS_UART_DIRECT
#define GS_BULK

#define GS_DNSNAME "setup.local"
#define GS_WREGDOMAIN "2" // 0:FCC, 1:ETSI, 2:TELEC

#define GS_TIMEOUT 10000 // ms
#define GS_TIMEOUT2 30000 // ms
#define GS_CMD_SIZE 100
#if defined(TARGET_LPC1768) || defined(TARGET_LPC2368)
//#define GS_DATA_SIZE 1500
#define GS_DATA_SIZE 1000
#elif defined(TARGET_LPC11U24)
#define GS_DATA_SIZE 500
#endif

#ifdef GS_UART_DIRECT
#if defined(TARGET_LPC1768) || defined(TARGET_LPC2368)
#define _gs_getc() LPC_UART1->RBR
#define _gs_putc(c) while(!(LPC_UART1->LSR & (1<<5))); LPC_UART1->THR = c
#elif defined(TARGET_LPC11U24)
#define _gs_getc() LPC_USART->RBR
#define _gs_putc(c) while(!(LPC_USART->LSR & (1<<5))); LPC_USART->THR = c
#endif
#else
#define _gs_getc() _gs.getc()
#define _gs_putc(c) _gs.putc(c)
#endif

/**
 * Wi-Fi security
 */
enum GSSECURITY {
    GSSEC_AUTO = 0,
    GSSEC_NONE = 0,
    GSSEC_OPEN = 1,
    GSSEC_WEP = 2,
    GSSEC_WPA_PSK = 4,
    GSSEC_WPA2_PSK = 8,
    GSSEC_WPA_ENT = 16,
    GSSEC_WPA2_ENT = 32,
    GSSEC_WPS_BUTTON = 64,
};

/**
 * TCP/IP protocol
 */
enum GSPROTOCOL {
    GSPROT_UDP = 0,
    GSPROT_TCP = 1,
    GSPROT_HTTPGET,
    GSPROT_HTTPPOST,
    GSPROT_HTTPD,
};

/**
 * Client/Server
 */
enum GSTYPE {
    GSTYPE_CLIENT = 0,
    GSTYPE_SERVER = 1,
};

enum GSRESPONCE {
    GSRES_NONE,
    GSRES_NORMAL,
    GSRES_CONNECT,
    GSRES_WPS,
    GSRES_MACADDRESS,
    GSRES_DHCP,
    GSRES_DNSLOOKUP,
    GSRES_HTTP,
    GSRES_RSSI,
    GSRES_TIME,
};

enum GSMODE {
    GSMODE_COMMAND,
    GSMODE_DATA_RX,
    GSMODE_DATA_RXUDP,
    GSMODE_DATA_RX_BULK,
    GSMODE_DATA_RXUDP_BULK,
    GSMODE_DATA_RXHTTP,
};

enum GSSTATUS {
    GSSTAT_READY,
    GSSTAT_STANDBY,
    GSSTAT_WAKEUP,
    GSSTAT_DEEPSLEEP,
};

/**
 * data receive callback function
 */
typedef void (*onGsReceiveFunc)(int cid, int len);

struct GS_Socket {
    GSTYPE type;
    GSPROTOCOL protocol;
    bool connect;
    Host host;
    RingBuffer *data;
    int lcid;
    int received;
    onGsReceiveFunc onGsReceive;
};

/**
 * GSwifi class
 */
class GSwifi {
public:
    /**
     * default constructor
     */
    GSwifi (PinName p_tx, PinName p_rx, int baud = GS_BAUD);
    /**
     * Default constructor (with hardware fllow controll)
     */
    GSwifi (PinName p_tx, PinName p_rx, PinName p_cts, PinName p_rts, int baud = GS_BAUD);
    /**
     * send command
     */
    int command (const char *cmd, GSRESPONCE res, int timeout = GS_TIMEOUT);
    /**
     * recv responce
     */
    int cmdResponse (GSRESPONCE res, int ms);
    /**
     * associate infrastructure
     * @param sec GSSEC_OPEN, GSSEC_WEP, GSSEC_WPA_PSK, GSSEC_WPA2_PSK, GSSEC_WPS_BUTTON
     * @param ssid SSID
     * @param pass pass phrase
     * @param dhcp 0:static ip, 1:dhcp
     * @param reconnect auto re-connect
     * @retval 0 success
     * @retval -1 failure
     */
    int connect (GSSECURITY sec, const char *ssid, const char *pass, int dhcp = 1, int reconnect = 0);
    /**
     * adhock
     * @param sec GSSEC_OPEN or GSSEC_WEP
     * @param ssid SSID
     * @param pass 10 or 26 hexadecimal digits
     * @param ipaddr my ip address
     * @param netmask subnet mask
     * @retval 0 success
     * @retval -1 failure
     */
    int adhock (GSSECURITY sec, const char *ssid, const char *pass, IpAddr ipaddr, IpAddr netmask);
    /**
     * limited AP
     * @param sec GSSEC_OPEN or GSSEC_WEP
     * @param ssid SSID
     * @param pass 10 or 26 hexadecimal digits
     * @param ipaddr my ip address (dhcp start address)
     * @param netmask subnet mask
     * @retval 0 success
     * @retval -1 failure
     * firmware: s2w-secureweb, s2w-web, s2w-wpsweb
     */
    int limitedap (GSSECURITY sec, const char *ssid, const char *pass, IpAddr ipaddr, IpAddr netmask, char *dns = NULL);
    /**
     * unassociate
     */
    int disconnect ();

    /**
     * use DHCP
     */
    int setAddress ();
    /**
     * use static ip address
     */
    int setAddress (IpAddr ipaddr, IpAddr netmask, IpAddr gateway, IpAddr nameserver);
    /**
     * get ip address
     */
    int getAddress (IpAddr &ipaddr, IpAddr &netmask, IpAddr &gateway, IpAddr &nameserver);
    /**
     * resolv hostname
     * @param name hostname
     * @param addr resolved ip address
     */
    int getHostByName (const char* name, IpAddr &addr);
    /**
     * resolv hostname
     * @param host.name hostname
     * @param host.ipaddr resolved ip address
     */
    int getHostByName (Host &host);
    /**
     * RF power
     * @param power 0(high)-7(low)
     */
    int setRFPower (int power);
    /**
     * power save mode
     * @param active rx radio 0:switched off, 1:always on
     * @param save power save 0:disable, 1:enable
     */
    int powerSave (int active, int save);
    /**
     * standby mode
     * @param msec wakeup after
     * wakeup after msec or alarm1/2
     * core off, save to RTC ram
     */
    int standby (int msec);
    /**
     * restore standby
     */
    int wakeup ();
    /**
     * deep sleep mode
     */
    int deepSleep ();
    /**
     * wifi connected
     */
    bool isConnected ();
    /**
     * status
     * @return GSSTATUS
     */
    GSSTATUS getStatus ();
    /**
     * RSSI
     * @return RSSI (dBm)
     */
    int getRssi ();
    /**
     * set NTP server
     * @param host SNTP server
     * @param sec time sync interval, 0:one time
     */
    int ntpdate (Host host, int sec = 0);
    /**
     * set system time
     * @param time date time (UTC)
     */
    int setTime (time_t time);
    /**
     * get RTC time
     * @return date time (UTC)
     */
    time_t getTime ();
    /**
     * GPIO output
     * @param port 10,11,30,31
     * @param out 0:set(high), 1:reset(low)
     */
    int gpioOut (int port, int out);

    /**
     * main polling
     */
    void poll();

    /**
     * tcp/udp client
     * @return CID, -1:failure
     */
    int open (Host &host, GSPROTOCOL pro, onGsReceiveFunc ponGsReceive = NULL);
    /**
     * tcp/udp server
     * @return CID, -1:failure
     */
    int listen (int port, GSPROTOCOL pro, onGsReceiveFunc ponGsReceive = NULL);
/*
    template<typename T>
    int listen2 (T* tptr, int port, GSPROTOCOL pro, void (T::*mptr)(int,int)) {
        if((mptr != NULL) && (tptr != NULL)) {
            return listen(port, pro, reinterpret_cast<onGsReceiveFunc>(mptr));
        }
    }
*/
    /**
     * close client/server
     */
    int close (int cid);
    /**
     * send data tcp(s/c), udp(c)
     */
    int send (int cid, const char *buf, int len);
    /**
     * send data udp(s)
     */
    int send (int cid, const char *buf, int len, Host &host);
    /**
     * recv data tcp(s/c), udp(c)
     * @return length
     */
    int recv (int cid, char *buf, int len);
    /**
     * recv data udp(s)
     * @return length
     */
    int recv (int cid, char *buf, int len, Host &host);
    /**
     * tcp/udp connected
     */
    bool isConnected (int cid);

    /**
     * http request (GET method)
     */
    int httpGet (Host &host, const char *uri, const char *user, const char *pwd, int ssl = 0, onGsReceiveFunc ponGsReceive = NULL);
    int httpGet (Host &host, const char *uri, int ssl = 0, onGsReceiveFunc ponGsReceive = NULL);
    /**
     * http request (POST method)
     */
    int httpPost (Host &host, const char *uri, const char *body, const char *user, const char *pwd, int ssl = 0, onGsReceiveFunc ponGsReceive = NULL);
    int httpPost (Host &host, const char *uri, const char *body, int ssl = 0, onGsReceiveFunc ponGsReceive = NULL);

    /**
     * certificate 
     */
    int certAdd (const char *name, const char *cert, int len);

#ifdef GS_USE_SMTP
    /**
     * send mail (smtp)
     * @param host SMTP server
     * @param to To address
     * @param from From address
     * @param subject Subject
     * @param mesg Message
     * @param user username (SMTP Auth)
     * @param pwd password (SMTP Auth)
     * @retval 0 success
     * @retval -1 failure
     */
    int mail (Host &host, const char *to, const char *from, const char *subject, const char *mesg, const char *user = NULL, const char *pwd = NULL);
#endif

#ifdef GS_USE_HTTPD
    /**
     * start http server
     * @param port
     */
    int httpd (int port = 80);
    /**
     * attach uri to dirctory handler
     */
    void send_httpd_error (int cid, int err);
    /**
     * attach uri to dirctory handler
     */
    int attach_httpd (const char *uri, const char *dir);
    /**
     * attach uri to cgi handler
     */
    int attach_httpd (const char *uri, onHttpdCgiFunc ponHttpCgi);
#ifdef GS_USE_WEBSOCKET
    int send_websocket (int cid, const char *buf, int len);
#endif
#endif

    /**
     * Web server
     */
    int provisioning (char *user, char *pass);
    /**
     * change baud rate
     */
    int setBaud (int baud);
    /**
     * base64 encode
     */
    int base64encode (char *input, int length, char *output, int len);
    /**
     * url encode
     */
    int urlencode (char *str, char *buf, int len);
    /**
     * url decode
     */
    int urldecode (char *str, char *buf, int len);

#ifdef DEBUG
    void dump ();
    void test ();
    int getc();
    void putc(char c);
    int readable();
#endif

protected:
    void poll_cmd ();
    int x2i (char c);
    char i2x (int i);
    void isr_recv ();
    void newSock (int cid, GSTYPE type, GSPROTOCOL pro, onGsReceiveFunc ponGsReceive);
    int from_hex (int ch);
    int to_hex (int code);

#ifdef GS_USE_SMTP
    int wait_smtp (int cid, int code);
#endif

#ifdef GS_USE_HTTPD
    int get_handler (char *uri);
    int httpd_request (int cid, GS_httpd *gshttpd, char *dir);
    char *mimetype (char *file);
    int strnicmp (char *p1, char *p2, int n);
#endif

private:
    Serial _gs;
    bool _rts;
    volatile bool _connect;
    volatile GSSTATUS _status;
    volatile int _gs_ok, _gs_failure, _gs_enter;
    volatile GSRESPONCE _response;
    GSMODE _gs_mode;
    int _escape;
    int _cid, _rssi;
    IpAddr _ipaddr, _netmask, _gateway, _nameserver, _resolv;
    Host _from, _to;
    char _mac[6];
    RingBuffer _buf_cmd;
    struct GS_Socket _gs_sock[16];
    time_t _time;
    char *_ssid;
    int _reconnect, _reconnect_count;
    
#ifdef GS_USE_HTTPD
    struct GS_httpd _httpd[16];
    struct GS_httpd_handler _handler[10];
    int _handler_count;

    void poll_httpd (int cid, int len);
#ifdef GS_USE_WEBSOCKET
    void poll_websocket (int cid, int len);
    void send_websocket_accept (int cid);
#endif
#endif
};

#endif
