/* Copyright (C) 2013 gsfan, MIT License
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy of this software
 * and associated documentation files (the "Software"), to deal in the Software without restriction,
 * including without limitation the rights to use, copy, modify, merge, publish, distribute,
 * sublicense, and/or sell copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all copies or
 * substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED, INCLUDING
 * BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 * NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM,
 * DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 */
/** @file
 * @brief Gainspan wi-fi module library for mbed
 * GS1011MIC, GS1011MIP, GainSpan WiFi Breakout, etc.
 */

#include "dbg.h"
#include "mbed.h"
#include "GSwifi.h"


void GSwifi::newSock (int cid, GSTYPE type, GSPROTOCOL pro) {
    DBG("newSock %d\r\n", cid);
    _gs_sock[cid].type = type;
    _gs_sock[cid].protocol = pro;
    _gs_sock[cid].connect = true;
    if (_gs_sock[cid].data == NULL) {
        _gs_sock[cid].data = new CircBuffer<char>(GS_DATA_SIZE);
    } else {
        _gs_sock[cid].data->flush();
    }
    _gs_sock[cid].lcid = 0;
    _gs_sock[cid].received = false;
    _gs_sock[cid].onGsReceive.detach();
}

int GSwifi::open (Host &host, GSPROTOCOL pro, int port) {
    char cmd[GS_CMD_SIZE];

    if (! _connect || _status != GSSTAT_READY) return -1;
    if (host.getIp().isNull()) {
        if (getHostByName(host)) {
            if (getHostByName(host)) return -1;
        }
    }
    if (host.getPort() == 0) {
        return -1;
    }

    if (pro == GSPROT_UDP) {
        sprintf(cmd, "AT+NCUDP=%d.%d.%d.%d,%d", host.getIp()[0], host.getIp()[1], host.getIp()[2], host.getIp()[3], host.getPort());
        if (port) {
            sprintf(&cmd[strlen(cmd)], ",%d", port);
        }
    } else {
        sprintf(cmd, "AT+NCTCP=%d.%d.%d.%d,%d", host.getIp()[0], host.getIp()[1], host.getIp()[2], host.getIp()[3], host.getPort());
    }
    if (command(cmd, GSRES_CONNECT)) return -1;

    newSock(_cid, GSTYPE_CLIENT, pro);
    return _cid;
}

int GSwifi::listen (int port, GSPROTOCOL pro) {
    char cmd[GS_CMD_SIZE];

    if (! _connect || _status != GSSTAT_READY) return -1;
    if (port == 0) {
        return -1;
    }

    if (pro == GSPROT_UDP) {
        sprintf(cmd, "AT+NSUDP=%d", port);
    } else {
        sprintf(cmd, "AT+NSTCP=%d", port);
    }
    if (command(cmd, GSRES_CONNECT)) return -1;

    newSock(_cid, GSTYPE_SERVER, pro);
    return _cid;
}

int GSwifi::close (int cid) {
    char cmd[GS_CMD_SIZE];

    if (! _gs_sock[cid].connect) return -1;

    _gs_sock[cid].connect = false;
//    delete _gs_sock[cid].data;
//    _gs_sock[cid].data = NULL;
    sprintf(cmd, "AT+NCLOSE=%X", cid);
    return command(cmd, GSRES_NORMAL);    
}

int GSwifi::send (int cid, const char *buf, int len) {
    int i;
    char cmd[GS_CMD_SIZE];

    if ((! _gs_sock[cid].connect) || acquireUart()) return -1;

    if ((_gs_sock[cid].protocol == GSPROT_TCP) ||
      (_gs_sock[cid].protocol == GSPROT_UDP && _gs_sock[cid].type == GSTYPE_CLIENT) ||
      (_gs_sock[cid].protocol == GSPROT_HTTPD)) {
        // TCP Client, TCP Server, UDP Client
        resetResponse(GSRES_NONE);
#ifdef GS_BULK
        sprintf(cmd, "\x1bZ%X%04d", cid, len);
        _gs_puts(cmd);
        for (i = 0; i < len; i ++) {
            _gs_putc(buf[i]);
#ifdef DEBUG_VIEW
            DBG("%c", buf[i]);
#endif
        }
        releaseUart();
#else // GS_BULK
        sprintf(cmd, "\x1bS%X", cid);
        _gs_puts(cmd);
        for (i = 0; i < len; i ++) {
            if (buf[i] >= 0x20 && buf[i] < 0x7f) {
                _gs_putc(buf[i]);
#ifdef DEBUG_VIEW
                DBG("%c", buf[i]);
#endif
            }
        }
        _gs_putc(0x1b);
        _gs_putc('E');
        releaseUart();
#endif
    } else {
        releaseUart();
        return -1;
    }
    
    return waitResponse(GS_TIMEOUT);
}

int GSwifi::send (int cid, const char *buf, int len, Host &host) {
    char cmd[GS_CMD_SIZE];
    int i;

    if ((! _gs_sock[cid].connect) || acquireUart()) return -1;

    if ((_gs_sock[cid].protocol == GSPROT_UDP && _gs_sock[cid].type == GSTYPE_SERVER)) {
        // UDP Server
        resetResponse(GSRES_NONE);
#ifdef GS_BULK
        sprintf(cmd, "\x1bY%X%d.%d.%d.%d:%d:%04d", cid, host.getIp()[0], host.getIp()[1], host.getIp()[2], host.getIp()[3], host.getPort(), len);
        _gs_puts(cmd);
        for (i = 0; i < len; i ++) {
            _gs_putc(buf[i]);
#ifdef DEBUG_VIEW
            DBG("%c", buf[i]);
#endif
        }
        releaseUart();
#else // GS_BULK
        sprintf(cmd, "\x1bU%X%d.%d.%d.%d:%d:", cid, host.getIp()[0], host.getIp()[1], host.getIp()[2], host.getIp()[3], host.getPort());
        _gs_puts(cmd);
        for (i = 0; i < len; i ++) {
            if (buf[i] >= 0x20 && buf[i] < 0x7f) {
                _gs_putc(buf[i]);
#ifdef DEBUG_VIEW
                DBG("%c", buf[i]);
#endif
            }
        }
        _gs_putc(0x1b);
        _gs_putc('E');
        releaseUart();
#endif
    } else {
        releaseUart();
        return -1;
    }

    return waitResponse(GS_TIMEOUT);
}

int GSwifi::recv (int cid, char *buf, int len) {
    int i;
    Timer timeout;
    
    if (_gs_sock[cid].data == NULL) return 0;

    timeout.start();
    while (_gs_sock[cid].data->isEmpty()) {
        if (timeout.read_ms() > GS_TIMEOUT) return 0;
    }
    timeout.stop();

    for (i = 0; i < len; i ++) {
        if (_gs_sock[cid].data->dequeue(&buf[i]) == false) break;
    }
    return i;
}

int GSwifi::recv (int cid, char *buf, int len, Host &host) {
    int i;
    Timer timeout;
    
    if (_gs_sock[cid].data == NULL) return 0;

    timeout.start();
    while (_gs_sock[cid].data->isEmpty()) {
        if (timeout.read_ms() > GS_TIMEOUT) return 0;
    }
    timeout.stop();

    for (i = 0; i < len; i ++) {
        if (_gs_sock[cid].data->dequeue(&buf[i]) == false) break;
    }
    host = _from;
    return i;
}

bool GSwifi::isConnected (int cid) {
    return _gs_sock[cid].connect;
}
