/**
 ******************************************************************************
 * @file    main.cpp
 * @author  APG Mass Market
 * @version V1.0.1
 * @date    16-Nov-2015
 * @brief   Example application for using the X_NUCLEO_IPS01A1 
 *          Intelligent Power Switch Nucleo expansion board.
 ******************************************************************************
 * @attention
 *
 * <h2><center>&copy; COPYRIGHT(c) 2015 STMicroelectronics</center></h2>
 *
 * Redistribution and use in source and binary forms, with or without modification,
 * are permitted provided that the following conditions are met:
 *   1. Redistributions of source code must retain the above copyright notice,
 *      this list of conditions and the following disclaimer.
 *   2. Redistributions in binary form must reproduce the above copyright notice,
 *      this list of conditions and the following disclaimer in the documentation
 *      and/or other materials provided with the distribution.
 *   3. Neither the name of STMicroelectronics nor the names of its contributors
 *      may be used to endorse or promote products derived from this software
 *      without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 *  SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
 * CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 ******************************************************************************
*/ 

/**
 * @mainpage X_NUCLEO_IPS01A1 Intelligent Power Switch Nucleo Expansion Board Firmware Package
 *
 * <b>Introduction</b>
 *
 * This firmware package includes Components Device Drivers, Board Support Package
 * and example application for STMicroelectronics X_NUCLEO_IPS01A1 Intelligent Power Switch
 * Nucleo Expansion Board
 * 
 * <b>Example Application</b>
 *
 */
/*** Includes ----------------------------------------------------------------- ***/
#include "mbed.h"
#include "assert.h"
#include "x_nucleo_ips01a1.h"

/*** Static variables --------------------------------------------------------- ***/
#ifdef DBG_MCU
#include "DbgMCU.h"
static DbgMCU enable_dbg;
#endif // DBG_MCU

/* HW settings */
/* Pay attention before changing HW settings, they must be coherent with you HW design */
/* Power Switch Connection to Arduino connectors */
#define IPS01A1_PIN_IN_0          (D5)
#define IPS01A1_PIN_IN_1          (D6)
#define IPS01A1_PIN_SEL_0         (D14)
#define IPS01A1_PIN_SEL_1         (D15)
#define IPS01A1_PIN_S_EN          (D2)
#define IPS01A1_PIN_FAULT_RST     (D4)
#define IPS01A1_PIN_MULTISENSE    (A2)
#define IPS01A1_PIN_V_GND         (A4)

/* V-Ref */
#define V_REF 3.3
/* Rsense Value */
#define R_SENSE 1e3
/* R_D1 */
#define R_D1 15e3
/* R_D2 */
#define R_D2 30e3

/* End of HW settings */

static X_NUCLEO_IPS01A1 &ips_expansion_board = X_NUCLEO_IPS01A1::Instance(IPS01A1_PIN_IN_0,
                                                                          IPS01A1_PIN_IN_1,
                                                                          IPS01A1_PIN_SEL_0, 
                                                                          IPS01A1_PIN_SEL_1,
                                                                          IPS01A1_PIN_S_EN,
                                                                          IPS01A1_PIN_FAULT_RST,
                                                                          IPS01A1_PIN_MULTISENSE,
                                                                          IPS01A1_PIN_V_GND, 
                                                                          V_REF,
                                                                          R_SENSE,
                                                                          R_D1,
                                                                          R_D2);



static Ticker ticker;
DigitalOut UserLed(LED1);    

/*** Main function ------------------------------------------------------------- ***/
/* Generic main function/loop, interrupt based cyclic execution
*/

float Multisense_Signal = 0;    // Multisense pin - signal level
float V_GND_Signal = 0;         // GND pin - signal level
float Temp_Signal = 0;          // Device temperature (info from Device)
float Vcc_Level = 0;            // Vcc signal level (info from device)
bool ButtonPressed = 0;         // User Button
int TestSequence = 0;           // Test sequence counter

//------------------------------------
// Hyperterminal configuration
// 9600 bauds, 8-bit data, no parity
//------------------------------------

InterruptIn UserButton(USER_BUTTON);  // B1 is the User Button
void B1_pressed (void);
void LedBlink (int TestSequence);
void Write_Serial (void);


int main(void){
    UserButton.fall(&B1_pressed);       //interrupt User Button
    printf("Start test Procedure \n\n\n");
    
    while (true) {
        // wait for User button is pressed 
        while (!ButtonPressed) {
        }
           
        ButtonPressed = 0;
        
        LedBlink(TestSequence);
     
        switch (TestSequence) {
          case (1): {
            printf("Test 1: StandBy\n\r");
            ips_expansion_board.vps220h1.In_0= 0;
            ips_expansion_board.vps220h1.In_1 = 0;
            ips_expansion_board.vps220h1.Sel_0 =0;
            ips_expansion_board.vps220h1.Sel_1 =0;
            ips_expansion_board.vps220h1.S_En = 0;
            ips_expansion_board.vps220h1.FaultRST = 0;
            wait (0.1);
            Multisense_Signal= ips_expansion_board.GetCurrent(0);
            Write_Serial();
            printf("Current Ch1 (stand by)= %2.3fA \n\n\r", Multisense_Signal);
           }
           break;
          case(2):{
            printf("Test 2: Temperature\n\r");
            ips_expansion_board.vps220h1.In_0= 0;
            ips_expansion_board.vps220h1.In_1 = 0;
            ips_expansion_board.vps220h1.Sel_0 =0;
            ips_expansion_board.vps220h1.Sel_1 =1;
            ips_expansion_board.vps220h1.S_En = 1;
            ips_expansion_board.vps220h1.FaultRST = 0;
            wait (0.1); 
      
            Temp_Signal= ips_expansion_board.GetTemperature();
            Write_Serial();
            printf("Temperature (equiv)= %2.3fC \n\n\r", Temp_Signal);
             
          }
          break;
          case(3):{
            printf("Test 3: Vcc Level\n\r");
            ips_expansion_board.vps220h1.In_0= 0;
            ips_expansion_board.vps220h1.In_1 = 0;
            ips_expansion_board.vps220h1.Sel_0 =1;
            ips_expansion_board.vps220h1.Sel_1 =1;
            ips_expansion_board.vps220h1.S_En = 1;
            ips_expansion_board.vps220h1.FaultRST = 0;
            wait (0.1); 
            Vcc_Level= ips_expansion_board.GetVcc();
            Write_Serial();
            printf("Vcc= %2.3fV \n\n\r", Vcc_Level);
          }
          break;
          case(4):{
            printf("Test 4: Ch1=ON, CH2=OFF\n\r");
            ips_expansion_board.vps220h1.In_0= 1;
            ips_expansion_board.vps220h1.In_1 = 0;
            ips_expansion_board.vps220h1.Sel_0 =0;
            ips_expansion_board.vps220h1.Sel_1 =0;
            ips_expansion_board.vps220h1.S_En = 1;
            ips_expansion_board.vps220h1.FaultRST = 0;
            wait (0.1);
            Write_Serial();
            printf("Current Ch1= %2.3fA \n\n\r",ips_expansion_board.GetCurrent(1) );
            wait (.5);
            ips_expansion_board.vps220h1.In_0= 0;
            ips_expansion_board.vps220h1.In_1 = 0;
          }
          break;
          case(5):{
            printf("Test 5: Ch1=OFF, CH2=ON\n\r");
            ips_expansion_board.vps220h1.In_0= 0;
            ips_expansion_board.vps220h1.In_1 = 1;
            ips_expansion_board.vps220h1.Sel_0 =1;
            ips_expansion_board.vps220h1.Sel_1 =0;
            ips_expansion_board.vps220h1.S_En = 1;
            ips_expansion_board.vps220h1.FaultRST = 0;
            wait (0.1);
            Write_Serial();
            printf("Current Ch1= %2.3fA \n\n\r",ips_expansion_board.GetCurrent(0));
            wait (.5);
            ips_expansion_board.vps220h1.In_0= 0;
            ips_expansion_board.vps220h1.In_1 = 0;
           }
          break;
          case(6):{
            printf("Test 6: Ch1=ON, CH2=ON -- Current Ch1\n\r");
            ips_expansion_board.vps220h1.In_0= 1;
            ips_expansion_board.vps220h1.In_1 = 1;
            ips_expansion_board.vps220h1.Sel_0 =0;
            ips_expansion_board.vps220h1.Sel_1 =0;
            ips_expansion_board.vps220h1.S_En = 1;
            ips_expansion_board.vps220h1.FaultRST = 0;
            wait (0.1);
            Write_Serial();
            printf("Current Ch1= %2.3fA \n\n\r", ips_expansion_board.GetCurrent(0) );
            wait (.5);
            ips_expansion_board.vps220h1.In_0= 0;
            ips_expansion_board.vps220h1.In_1 = 0;
              
          }
          break;
          case(7):{
            printf("Test 7: Ch1=ON, CH2=ON -- Current Ch2\n\r");
            ips_expansion_board.vps220h1.In_0= 1;
            ips_expansion_board.vps220h1.In_1 = 1;
            ips_expansion_board.vps220h1.Sel_0 =1;
            ips_expansion_board.vps220h1.Sel_1 =0;
            ips_expansion_board.vps220h1.S_En = 1;
            ips_expansion_board.vps220h1.FaultRST = 0;
            wait (0.1);
            Write_Serial();
            printf("Current Ch2= %2.3fA \n\n\r", ips_expansion_board.GetCurrent(1));
            wait (.5);
            ips_expansion_board.vps220h1.In_0= 0;
            ips_expansion_board.vps220h1.In_1 = 0;
             
          }
          break;
          default: {
            printf("End Test Procedure\n\n\n\r");  
            TestSequence = 0;
            ips_expansion_board.vps220h1.In_0= 0;
            ips_expansion_board.vps220h1.In_1 = 0;
            ips_expansion_board.vps220h1.Sel_0 =0;
            ips_expansion_board.vps220h1.Sel_1 =0;
            ips_expansion_board.vps220h1.S_En = 0;
            ips_expansion_board.vps220h1.FaultRST = 0;
           }
          break;
        }      
    }
    
 }


void B1_pressed (){
  // Interrupt procedure - User button is pressed
    TestSequence ++;

    UserLed = 1; // LED is ON
    wait(0.05); // 50 ms
    UserLed = 0; // LED is OFF

    ButtonPressed = 1;

}
void LedBlink (int TestSequence){
// Option feedback by usingUser LED
    for (int TestCounter =0; TestCounter<TestSequence; TestCounter++) {
        UserLed = 1; // LED is ON
        wait(0.05); // 50 ms
        UserLed = 0; // LED is OFF
        wait(0.05); // 50 msec
    }
    wait(1-(TestSequence*2*0.05));
}

void Write_Serial (){
// This code send messages and data to the serial port
    // send info to serial port
    printf("Input 0= %d\t", ips_expansion_board.vps220h1.In_0.read());
    printf("Input 1= %d\t", ips_expansion_board.vps220h1.In_1.read());
    printf("SEL 0= %d\t", ips_expansion_board.vps220h1.Sel_0.read());
    printf("SEL 1= %d\t", ips_expansion_board.vps220h1.Sel_1.read());
    printf("SEL_EN= %d\t", ips_expansion_board.vps220h1.S_En.read());
    printf("FaultRST= %d\t", ips_expansion_board.vps220h1.FaultRST.read());
 }



