#ifndef MOTION_H
#define MOTION_H

#include <vector>
#include <string>
#include "HomeMonUtils.h"

/** Simple structure to hold a motion vector - x, y, and z motion
*/
struct motion_vec {
    double x;
    double y;
    double z;
};

/** Motion class stores historical motion alert data and thresholding for alert generation
*/
class Motion {
    private: 
      motion_vec min_motion;
	  int max_samples_limit;
      int max_samples;
	  int sample_ptr;
      bool wrapped_once;
      std::vector<std::string> motion_samples;

    public:
    /**
    * Instantiate a motion class.  Initialization hard-coded to 
    * known good values.  Use member functions to modify.
    */
	  Motion();
	  /**
	  * Allows host to retrieve motion threshold as a vector.
	  *
	  * @param return - Returns a motion vector structure with x, y, and z threholds for motion.
	  */
      motion_vec get_motion_thresh(void);
      /**
      * Allows host to set the motion threshold
      *
      * @param motion_thresh - Motion vector threshold setting
      * @param return - pass/faill for set
      */
	  bool set_motion_thresh(motion_vec motion_thresh);
	  /**
	  * Adds a sample to the motion database.  Units are time in seconds since device booted.
	  *
	  * @param motion_sample - represents seconds since device booted that a motion alert occurred.
	  */
      void add_sample(double motion_sample);
      /**
      * Returns all motion samples in the motion database.
      *
      * @param return - vector containing strings for all motion alerts (seconds since device booted)
      */
      const std::vector<std::string> &get_samples();
      /**
      * Allows the host to change the motion database size.
      * WARNING: This may cause sample to be lost due to vector resize operation
      *
      * @param num_samples - new size of motion alert DB (max allowed currently 100)
      * @param return - pass/fail of operation
      */
	  bool change_max_samples(int num_samples);
	  /**
	  * Returns current database size
	  *
	  * @param return - size in samples of motion alert DB
	  */
	  int get_max_samples();

};

#endif