#ifndef TEMPERATURE_H
#define TEMPERATURE_H

#include <vector>
#include <string>

/** Temperature class stores historical temperature data and tresholds for temperature alerts
* 
*/
class Temperature {
    private: 
	  double max_temp_limit, min_temp_limit, period_limit, max_samples_limit;
      double min_temp;
      double max_temp;
      double temp_sample_period;
      int max_samples;
      int sample_ptr;
      bool wrapped_once;
      std::vector<std::string> temp_samples;

    public:
    /**
    * Creates a new temperature object.  All parameters
    * hard coded at first and methods used to set appropriately
    * based on user's desires.
    */
	  Temperature();
	  /**
	  * Returns minimum temperature threshold
	  *
	  * @param return - mininum temperature threshold
	  */
      double get_min();
      /**
      * Returns maximum temperature threshold
      *
      * @param return - maximum temperature threshold
      */
      double get_max();
      /**
      * Returns temperature sampling period in seconds
      *
      * @param return - temperature sampling period in seconds
      */
      double get_period();
      /**
      * Sets the minimum temperature threshold for alerts.
      *
      * @param min - new minimum temperature threshold for alerts
      * @param return - indicates success or failure
      */
      bool set_min(double min);
      /**
      * Sets the maximum temperature threshold for alerts.
      *
      * @param max - new maximum temperature threshold for alerts
      * @param return - indicates success or failure
      */
      bool set_max(double max);
      /**
      * Sets the temperature period in seconds to store temperature samples in DB
      *
      * @param period - termperature sampling period in seconds
      * @param return - pass or fail of attempt
      */
      bool set_period(double period);
      /**
      * Method to add a temperature sample to the database
      *
      * @param temp_sample - temperature sample to be stored to dB
      */
      void add_sample(double temp_sample);
      /**
      * Changes number of samples in temperature database.
      * Current maximum is 100 samples
      * WARNING: Changing this may trash samples since vector is resized.
      *
      * @param num_samples - Number of samples in DB
      * @param return - pass fail of attemp 
      */
      bool change_max_samples(int num_samples);
      /**
      * Returns database size in temperature DB
      *
      * @param return - current size of temp DB in samples
      */
	  int get_max_samples();
	  /**
	  * Returns all temperature samples as strings in the database.
	  * number returned should match max_samples.
	  *
	  * @param return - Reference to vector containing temperature DB as strings
	  */
      const std::vector<std::string> &get_samples();

};

#endif