/*
 * mbed library program 
 *      Stepping Motor
 *
 * Copyright (c) 2014 Kenji Arai / JH1PJL
 *  http://www.page.sannet.ne.jp/kenjia/index.html
 *  http://mbed.org/users/kenjiArai/
 *      Created: August    20th, 2014
 *      Revised: August    23rd, 2014
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED,
 * INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE
 * AND NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM,
 * DAMAGES OR OTHER  LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 */

#ifndef        MBED_STEPPER
#define        MBED_STEPPER

#include "mbed.h"

#define MT_SLOP_STEP            10
#define MT_MIN_STEP             (MT_SLOP_STEP+MT_SLOP_STEP)
#define MT_PLS_WIDTH_MIN        2      // 2mS

/** Unipolar Type Stepping Motor Driver using 4 Outputs(DigitalOut) with timer interrupt(Ticker)
 *
 * Driver circuit: Low side driver (4ch for one Stepper) e.g. TD62003AP
 *  CAUTION: This is only for Unipolar Type Stepping Motor!
 *           Cannot use for Bipolar Type
 *      Plese refer http://en.wikipedia.org/wiki/Stepper_motor
 *
 * @code
 * #include "mbed.h"
 * #include "stepper.h"
 *
 * #define TIMEBASE  15000      // 15mS
 *
 * STEPPER sm(D5, D4, D3, D2);
 *
 * uint8_t pls_width[10] = {5, 4, 3, 2, 1, 1, 1, 1, 1, 1 };
 *
 * int main() {
 *   sm_r.set_max_speed(TIMEBASE);
 *   sm.move(+100);
 *   while (sm.status){ ;}
 *   sm.move(-1000);
 *   wait(10);
 *   sm.move(0);
 *   while(true){;}
 * }
 * @endcode
 */

class STEPPER {
public:
    // Please copy following pls_width[] part in to your main.cpp
    /** pulse width definition -> Use for start and stop phase
      *  = data * TIMEBASE -> e.g following data = 5 then 5*15000/1000 = 75mS
      */
    //uint8_t pls_width[MT_SLOP_STEP] = {5, 4, 3, 2, 1, 1, 1, 1, 1, 1 };

    /** Motor Status */
    enum MOTOR_STATE { M_STOP = 0, M_UP, M_CONTINUE, M_DOWN, M_CHANGE, M_UNKOWN = 0xff};
    
    /** Configure data pin
      * @param data SDA and SCL pins
      */
    STEPPER (PinName xp, PinName xn, PinName yp, PinName yn);

    /** Move steps
      * @param number of steps
      * @return none
      */
    void move (int32_t steps);

    /** Set time period (max speed of stepper)
      * @param time: e.g. 10mS(=10000) -> 100 PPS(Puls per Sec)
      * @return none
      */
    void set_max_speed (uint32_t time_base_us);

    /** Check status
      * @param none
      * @return running(= 1), stopped(= 0)
      */
      //surely other states can be returned too eg UP, CONTINUE, CHANGE, DOWN?
    uint8_t status (void);
    
    /** De-energize coils
      * @param none
      * @return none
      */
    void stop();


protected:
    // Rotation Direction
    enum { D_CCW = -1, D_UNKOWN = 0, D_CW = 1};
    
    typedef struct{
        int8_t direction;
        uint32_t total_step;
    } Motor_Inf;
    
    typedef struct{
        uint8_t  state;
        int8_t   direction;
        uint8_t  up_cnt;
        uint8_t  up_cnt_keep;
        uint8_t  down_cnt;
        uint8_t  change_cnt;
        uint8_t  pls_width;
        uint8_t  ongoing;
        uint32_t continue_cnt;
        uint8_t motor_step; //keeps track of phase only, no need of uint32_t  
    } Motor_Control;

    DigitalOut _xp, _xn, _yp, _yn;
    Ticker _smdrv;
    
    void set4ports (void);
    void setup_mtr_drv_dt(Motor_Inf *mi, Motor_Control *mt);
    void millisec_inteval();

private:
    uint8_t busy_sm_drv;
    Motor_Inf inf;   
    Motor_Control cntl;

};
   
#endif  //  MBED_STEPPER
