/*
 * Using STM32's counter peripherals to interface rotary encoders.
 * Encoders are supported on F4xx's TIM1,2,3,4,5. TIM2 & TIM5 have 32bit count, others 16bit.
 * Beware mbed uses TIM5 for system timer, SPI needs TIM1, others used for PWM.
 * Check your platform's PeripheralPins.c & PeripheralNames.h if you need both PWM & encoders.
 *
 * Edit HAL_TIM_Encoder_MspInitFx.cpp to suit your mcu & board's available pinouts & pullups/downs.
 *
 * Thanks to:
 * http://petoknm.wordpress.com/2015/01/05/rotary-encoder-and-stm32/
 *
 * References:
 * http://www.st.com/st-web-ui/static/active/en/resource/technical/document/user_manual/DM00122015.pdf
 * http://www.st.com/st-web-ui/static/active/en/resource/technical/document/reference_manual/DM00096844.pdf
 * http://www.st.com/web/en/resource/technical/document/application_note/DM00042534.pdf
 * http://www.st.com/web/en/resource/technical/document/datasheet/DM00102166.pdf
 * 
 * David Lowe Jan 2015
 */

#include "mbed.h"
#include "Encoder.h"

//STM mbed bug: these macros are MISSING from stm32f3xx_hal_tim.h
#ifdef TARGET_STM32F3
#define __HAL_TIM_GET_COUNTER(__HANDLE__) ((__HANDLE__)->Instance->CNT)
#define __HAL_TIM_IS_TIM_COUNTING_DOWN(__HANDLE__)            (((__HANDLE__)->Instance->CR1 &(TIM_CR1_DIR)) == (TIM_CR1_DIR))
#endif

TIM_Encoder_InitTypeDef encoder1, encoder2, encoder3, encoder4;
TIM_HandleTypeDef  timer1,  timer2,  timer3,  timer4;

int main()
{
    //examples
    
    //counting on A-input only, 2 ticks per cycle, rolls over at 100
    EncoderInit(&encoder1, &timer1, TIM1, 99, TIM_ENCODERMODE_TI1);

    //counting on both A&B inputs, 4 ticks per cycle, full 32-bit count
    EncoderInit(&encoder2, &timer2, TIM2, 0xffffffff, TIM_ENCODERMODE_TI12);

    //counting on B-input only, 2 ticks per cycle, full 16-bit count
    EncoderInit(&encoder3, &timer3, TIM3, 0xffff, TIM_ENCODERMODE_TI2);
    
    //counting on both A&B inputs, 4 ticks per cycle, full 16-bit count
    EncoderInit(&encoder4, &timer4, TIM4, 0xffff, TIM_ENCODERMODE_TI12);
    
    //TIM5 is used by mbed for systick
    //EncoderInit(encoder2, timer2, TIM5, 0xffffffff, TIM_ENCODERMODE_TI12);
    
    printf("STM HAL encoder demo\n\r");
    
    while(1) {
        uint16_t count1=0, count3=0, count4=0;
        uint32_t count2=0;
        int8_t dir1, dir2, dir3, dir4;

        
        //OK 401 411 446 TICKER 030
        //count1=TIM1->CNT;
        //dir1=TIM1->CR1&TIM_CR1_DIR;
        count1=__HAL_TIM_GET_COUNTER(&timer1);
        dir1 = __HAL_TIM_IS_TIM_COUNTING_DOWN(&timer1);

        //OK 401 411 446 NOK 030
        //count2=TIM2->CNT;
        //dir2=TIM2->CR1&TIM_CR1_DIR;
        count2=__HAL_TIM_GET_COUNTER(&timer2);
        dir2 = __HAL_TIM_IS_TIM_COUNTING_DOWN(&timer2);

        //OK 401 411 446 030
        //count3=TIM3->CNT;
        //dir3=TIM3->CR1&TIM_CR1_DIR;
        count3=__HAL_TIM_GET_COUNTER(&timer3);
        dir3 = __HAL_TIM_IS_TIM_COUNTING_DOWN(&timer3);

        //OK 401 411 446 N/A 030
        //count4=TIM4->CNT;
        //dir4=TIM4->CR1&TIM_CR1_DIR;
        count4=__HAL_TIM_GET_COUNTER(&timer4);
        dir4 = __HAL_TIM_IS_TIM_COUNTING_DOWN(&timer4);

        //TICKER 401 411 446 N/A 030
//        count5=__HAL_TIM_GET_COUNTER(&timer5);
//        dir5 = __HAL_TIM_IS_TIM_COUNTING_DOWN(&timer5);
        
        printf("%d%s %d%s %d%s %d%s\r\n", count1, dir1==0 ? "+":"-",
                                             count2, dir2==0 ? "+":"-",
                                             count3, dir3==0 ? "+":"-",
                                             count4, dir4==0 ? "+":"-" );
        wait(0.5);
    }
}
