/* MCP23008_Keypad Library v1.0
 * Copyright (c) 2016 Grant Phillips
 * grant.phillips@nmmu.ac.za
 *
 * This is a modified version of Riaan Ehlers' library which was written for the 
 * Microchip PIC18 series microcontrollers.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */
 
#ifndef MCP23008_Keypad_H
#define MCP23008_Keypad_H
 
#include "mbed.h"
 
/** Class library for a 4x4 keypad connected to a MCP23008 I2C IO Expander.
 * The Keypad is assumed to be connected to the MCP23008 as follows:
 *
 * GP7 -> ROW4; GP6 -> ROW3; GP5 -> ROW2; GP4 -> ROW1; GP3 -> COL4; GP2 -> COL3; GP1 -> COL2; GP0 -> COL1
 *
 * Example:
 * @code
 * #include "mbed.h"
 * #include "MCP23008_Keypad.h"
 *
 * MCP23008_Keypad keypad(PB_9, PB_6, 0x42);
 *
 * int main() {
 *     char rcv;
 *     int released = 1;
 *
 *     while(1){
 *         rcv = keypad.ReadKey();                 //read the current key pressed
 *
 *         if(rcv == '\0')
 *             released = 1;                       //set the flag when all keys are released
 *            
 *         if((rcv != '\0') && (released == 1)) {  //if a key is pressed AND previous key was released
 *             printf("%c\n", rcv);            
 *             released = 0;                       //clear the flag to indicate that key is still pressed
 *         }
 *     }
 * }
 * @endcode
 */
 
class MCP23008_Keypad {
  public:
    /** Create a MCP23008_Keypad object.
    * @param SDApin I2C compatible pin used for connecting to the MCP23008's SDA pin
    * @param SCLpin I2C compatible pin used for connecting to the MCP23008's SCL pin
    * @param MCP23008Address The I2C address of the MCP23008 to which the Keypad is connected.
    */
    MCP23008_Keypad(PinName SDApin, PinName SCLpin, int MCP23008Address);
    
    /** Returns which key is currently is pressed.
    * @param 
    *     None
    * @return 
    *     The character representation of the key being pressed.  Returns '\0' if no key is pressed.
    */
    char ReadKey();
    
 
  private:
    I2C i2c;
    int _addr;
};
 
#endif