/* LM6029ACW Library v1.0
 * Copyright (c) 2016 Grant Phillips
 * grant.phillips@nmmu.ac.za
 *
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */
 
#ifndef LM6029ACW_H
#define LM6029ACW_H
 
#include "mbed.h"

#define lcd_delayx      0.000001    //delay for writing and reading from lcd - 1us
#define lcd_init_delay  0.001       //delay when initializing the lcd - 1ms

/** Class library for a Topway LM6029ACW graphics LCD.
 *
 * Example:
 * @code
 * #include "mbed.h"
 * #include "LM6029ACW.h"
 * 
 * LM6029ACW lcd(PD_7, PE_2, PD_6, PD_3, PD_2, PC_15, PC_14, PC_13, PC_11, PC_9, PC_8, PC_6, PC_2);
 * //            CS1   RES   RS    WR    RD    DB7    DB6    DB5    DB4    DB3   DB2   DB1   DB0
 * 
 * DigitalOut backlight(PB_7);
 * 
 * int main() { 
 *     backlight = 0;                      //turn on the lcd's backlight
 *     while(1) {
 *         lcd.GotoXY(4, 3);               //move cursor to row, col
 *         lcd.ClrScr(0);                  //clear all the pixels on the display
 *         lcd.PutStr("Hello World!", 1);  //print text in black pixels
 *         wait(2.0);
 *         
 *         lcd.GotoXY(4, 3);               //move cursor to row, col
 *         lcd.ClrScr(1);                  //fill all the pixels on the display
 *         lcd.PutStr("Hello World!", 0);  //print text in clear pixels
 *         wait(2.0);
 *     } 
 * }
 * @endcode
 */
 
class LM6029ACW {
  public:
    /** Create a LM6029ACW object for a graphics LCD connected to the specified pins. 
    * @param CS1 Chip Select pin used to connect to LM6029ACW's CS1 pin
    * @param RES Reset pin used to connect to LM6029ACW's RES pin
    * @param RS Register Select pin used to connect to LM6029ACW's RS pin
    * @param WR Write pin used to connect to LM6029ACW's WR pin
    * @param RD Read pin used to connect to LM6029ACW's RD pin
    * @param DB7-DB0 Data Bus pin useds to connect to LM6029ACW's DB7 - DB0 pins
    */
    LM6029ACW(PinName CS1, PinName RES, PinName RS, PinName WR, PinName RD, PinName DB7, PinName DB6, PinName DB5, PinName DB4, PinName DB3, PinName DB2, PinName DB1, PinName DB0);
    
    /** Clears the lcd by clearing or filling all the pixels. 
    * @param color Bit value; clear pixels(0) or fill pixels(1)
    */
    void ClrScr(unsigned char color);
    
    /** Puts the lcd in power save mode - display off. 
    */
    void PowerSave(void);
    
    /** Puts the lcd in normal power mode - display on. 
    */
    void PowerActive(void);
    
    /** Draws a pixel on the lcd at the specified xy position. 
    * @param x x position (0 - 127)
    * @param y y position (0 - 63)
    * @param color mode of the pixel; cleared(0), filled(1), or toggle(2)
    */
    void DrawPixel(unsigned char x, unsigned char y, unsigned char color);
    
    /** Draws a line on the lcd from x1,y1 to x2,y2. 
    * @param x1 x1 position (0 - 127)
    * @param y1 y1 position (0 - 63)
    * @param x2 x2 position (0 - 127)
    * @param y2 y2 position (0 - 63)
    * @param color mode of the pixel; cleared(0), filled(1), or toggle(2)
    */
    void DrawLine(int x1, int y1, int x2, int y2, unsigned int color);
    
    /** Draws a circle on the lcd at x,y with the option to fill it.  
    * @param x x position (0 - 127)
    * @param y y position (0 - 63)
    * @param radius radius of the circle (0 - 127)
    * @param fill circle must be filled (=1) or not filled (=0)
    * @param color mode of the pixel; cleared(0), filled(1)
    */
    void DrawCircle(unsigned char x, unsigned char y, unsigned char radius, unsigned char fill, unsigned char color);
    
    /** Changes the position on the lcd of the current character cursor from where the next characters will be printed.
    * @param x column position (0 - 20)
    * @param y row position (0 - 7)
    */
    void GotoXY(unsigned char x, unsigned char y);
    
    /** Prints a string on the lcd at the current character cursor position.
    * @param str string (char array) to print
    * @param color mode of the characters in the string; normal(1), inverted(0)
    */
    void PutStr(char *str, unsigned char color);
    
    /** Prints a character on the lcd at the current character cursor position.
    * @param c character to print
    * @param color mode of the character; normal(1), inverted(0)
    */
    void PutChar(unsigned char c, unsigned char color);
    
    
  private:
    DigitalOut CS1pin, RESpin, RSpin, WRpin, RDpin;
    BusOut DATApins;
    unsigned char DisplayRAM[128][8];
    unsigned char character_x, character_y;
    
        
    /* Creates a delay which is simply a code loop which is independent from any hardware timers.
    * @param del 16-bit value to represent the delay cycles
    */
    void Delay(unsigned int del);
    
    /* Writes a COMMAND to the LM6029ACW lcd module.
    * @param cmd The byte of command to write to the lcd
    */
    void WriteCommand(unsigned char cmd);

    /* Writes DATA to the LM6029ACW lcd module.
    * @param dat The byte of data to write to the lcd
    */
    void WriteData(unsigned char dat);
    
    /* Set the display RAM page address.
    * @param Page 4-bit value representing the page address (0 - 7)
    */
    void SetPage(unsigned char Page);
    
    /* Set the column address counter.
    * @param Page Byte value representing the column (0 - 127)
    */
    void SetColumn(unsigned char Col);
    
    /** Initializes the LM6029ACW lcd module. 
    */
    void Init(void);
    
    /* Used by LM6029ACW_DrawChar only updates the column and page numbers..
    */
    void WriteCharCol(unsigned char v, unsigned char x, unsigned char page, unsigned char color);
    
    /* Used by LM6029ACW_PutChar() only and draws the specified character at byte level using a set font.
    */
    void DrawChar(unsigned char c, unsigned char x, unsigned char page, unsigned char color);
};
 
#endif