/* KEYPAD_DISCO_F429ZI Library v1.0
 * Copyright (c) 2016 Grant Phillips
 * grant.phillips@nmmu.ac.za
 *
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */
 
#ifndef KEYPAD_DISCO_F429ZI_H
#define KEYPAD_DISCO_F429ZI_H
 
#include "mbed.h"
#include "TS_DISCO_F429ZI.h"
#include "LCD_DISCO_F429ZI.h"

/** Class library for a touchscreen-based 4x4 keypad for the LCD display present on the DISCO_F429ZI board. 
 * It requires the TS_DISCO_F429ZI and LCD_DISCO_F429ZI to also be imported into the program.
 *
 * Example:
 * @code
 * #include "mbed.h"
 * #include "TS_DISCO_F429ZI.h"
 * #include "LCD_DISCO_F429ZI.h"
 * #include "KEYPAD_DISCO_F429ZI.h"
 * 
 * KEYPAD_DISCO_F429ZI keypad;
 *                            
 * int main() {
 *     char buf[20];
 *     char k, key;
 *     
 *     keypad.Show(1, 1);                              //Display the keypad, enable the Message and Output Box areas
 *     keypad.WriteMsgBoxLine(0, "Press any key...");  //display a message in the
 *     keypad.WriteMsgBoxLine(1, "----------------");  //Message Box area of the keypad
 *    
 *     while(1) {
 *         k = keypad.ReadKey();                       //read the current key pressed
 *         if(k != '\0') {                             //if a key is pressed
 *             key = k;                                //store the key in a variable
 *             do                                      //read until key is released
 *                 k = keypad.ReadKey();
 *             while(k != '\0');
 *             sprintf(buf, "%c was pressed", key);    
 *             keypad.WriteOutBox(buf);                //display result in Output Box area of keypad
 *         }
 *     }
 * }
 * @endcode
 */
 
class KEYPAD_DISCO_F429ZI {
  public:
    /** Create a KEYPAD_DISCO_F429ZI object.
    */
    KEYPAD_DISCO_F429ZI();
    
    /** Draws the keys for the keypad and the Message Box and Output Box areas if enabled.
    * @param 
    *     MessageBoxEnable Enable (=1) or disable(=0 the drawing of a Message Box area on the keypad screen.
    *     OutBoxEnable Enable (=1) or disable(=0 the drawing of a Output Box area on the keypad screen.
    */
    void Show(int MessageBoxEnable, int OutBoxEnable);
    
    /** Return the current key pressed on the keypad.
    * @param 
    *     None
    * @return
    *     ASCII character of the key pressed.  Will be NULL ('/0') if no key is pressed.
    */
    char ReadKey();

    /** Writes text (21 characters max) to one of the two lines of the Message Box area if it is enabled in Show().
    * @param 
    *     LineNum The line number to to where the text must be displayed (0 or 1).
    *     text String to display.
    */
    void WriteMsgBoxLine(int LineNum, char *text);
    
    /** Writes text (14 characters max) to the Output Box area if it is enabled in Show().
    * @param 
    *     text String to display.
    */
    void WriteOutBox(char *text);
    
    /** Sets the ASCII characters for each key on the 4x4 keypad.
    * @param 
    *     keymap Pointer to a two-dimensional array (array[4][4]) with ASCII characters for each key.
    */
    void SetKeys(char (*keymap)[4]);
    
    /** Sets the foreground color for the keypad keys.
    * @param 
    *     color The color constant or code ARGB(8-8-8-8).
    */
    void SetKeyForeColor(uint32_t color);
    
    /** Sets the background color for the keypad keys.
    * @param 
    *     color The color constant or code ARGB(8-8-8-8).
    */
    void SetKeyBackColor(uint32_t color);
    
    /** Sets the background color for the keypad keys when it is pressed.
    * @param 
    *     color The color constant or code ARGB(8-8-8-8).
    */
    void SetKeyPressColor(uint32_t color);
    
    /** Sets the foreground color for the Message Box area.
    * @param 
    *     color The color constant or code ARGB(8-8-8-8).
    */
    void SetMsgBoxForeColor(uint32_t color);
    
    /** Sets the background color for the Message Box area.
    * @param 
    *     color The color constant or code ARGB(8-8-8-8).
    */
    void SetMsgBoxBackColor(uint32_t color);
    
    /** Sets the foreground color for the Output Box area.
    * @param 
    *     color The color constant or code ARGB(8-8-8-8).
    */
    void SetOutBoxForeColor(uint32_t color);
    
    /** Sets the background color for the Output Box area.
    * @param 
    *     color The color constant or code ARGB(8-8-8-8).
    */
    void SetOutBoxBackColor(uint32_t color);
    
    /** Sets the background color for the whole keypad area.
    * @param 
    *     color The color constant or code ARGB(8-8-8-8).
    */
    void SetBackColor(uint32_t color);
 
  private:
    LCD_DISCO_F429ZI lcd;
    TS_DISCO_F429ZI ts;
    uint16_t LastkeyX, LastkeyY;  
    uint8_t LastKey;
    uint8_t ReadKeyState;
    uint16_t KeysTopY;
    uint8_t row, col;
    char Keys[4][4];
    uint32_t MsgBColor, MsgFColor;
    uint32_t OutBColor, OutFColor;
    uint32_t KeyBColor, KeyFColor, KeyPressColor;
    uint32_t BackColor;
    int MessageBoxEnabledFlag, OutBoxEnabledFlag;
};
 
#endif