/* ESP8266 Library v1.0
 * Copyright (c) 2017 Grant Phillips
 * grant.phillips@nmmu.ac.za
 * 
 * This library was adapted from the WeeESP8266 version by ITEAD STUDIO.
 * (https://os.mbed.com/users/itead/code/WeeESP8266/)
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */
 
#ifndef ESP8266_H
#define ESP8266_H
 
#include "mbed.h"
#include <string>
 
/** Class library for using the ESP8266 wifi module.
 */
 
class ESP8266 {
  public:
    /** Create a ESP8266 object connected to the specified pins. 
    * @param Tx Pin used to connect to ESP8266's Tx pin
    * @param Rx Pin used to connect to ESP8266's Rx pin
    */
    ESP8266(PinName Tx, PinName Rx, PinName Rst);
    
    /** Starts the ESP8266 in a specified mode.
    * @param mode The WiFi mode the ESP8266 must be set to (1=Station, 2=SoftAP, 3=Station+SoftAP)
    *
    * @retval true - succesfull
    * @retval false - not succesfull
    */
    bool startup(uint8_t mode);
    
    /** Get the version of AT Command Set from the ESP8266.
    * 
    * @param version String variable where version will be stored. 
    */
    void getVersion(char* version);
    
    /** Search available AP list and return it.
    * 
    * @param list String variable where AP list will be stored.
    * @note This method will occupy a lot of memeory(hundreds of Bytes to a couple of KBytes). 
    *  Do not call this method unless you must and ensure that your board has enough memory left.
    */
    void getAPList(char* list);
    
    /** Set SoftAP parameters. 
     * 
     * @param ssid SSID of SoftAP. 
     * @param pwd PASSWORD of SoftAP. 
     * @param chl the channel (1 - 13, default: 7). 
     * @param ecn the way of encrypstion (0 - OPEN, 1 - WEP, 
     *  2 - WPA_PSK, 3 - WPA2_PSK, 4 - WPA_WPA2_PSK, default: 4). 
     * @note This method should not be called when station mode. 
     */
    bool setSoftAPParam(string ssid, string pwd, uint8_t chl = 7, uint8_t ecn = 4);
    
    /** Enable/Disable DHCP
    *
    * @param enabled DHCP enabled when true
    * @param mode mode of DHCP 0-softAP, 1-station, 2-both
    * @return true only if ESP8266 enables/disables DHCP successfully
    */
    bool DHCP(bool enabled, int mode);
    
    /** Join in AP. 
    *
    * @param ssid SSID of AP to join in. 
    * @param pwd Password of AP to join in. 
    * @retval true - success.
    * @retval false - failure.
    * @note This method will take a couple of seconds. 
    */
    bool joinAP(string ssid, string pwd);
    
    /** Get the local IP address of ESP8266. 
    *
    * @param version String variable where version will be stored. 
    */
    void getLocalIP(char* LocalIP);
    
    /** Sets the IP Address of the ESP8266 Station. 
    *
    * @param LocalIP The IP Address of the ESP8266 Station
    * @retval true - success.
    * @retval false - failure.
    */
    bool setLocalIP(string LocalIP);
    
    /** Sets the IP Address of the ESP8266 SoftAP. 
    *
    * @param LocalIP The IP Address of the ESP8266 SoftAP
    * @retval true - success.
    * @retval false - failure.
    */
    bool setSoftAPIP(string LocalIP);
    
    /** Start TCP Server(Only in multiple mode). 
    * 
    * After started, only methods TCPPacketSend, TCPPacketRead, and TCPPacketArrived can be used. 
    * Once TCPServerStop is called, all other methods can be used again.
    *
    * @param port The port number to listen on (default: 333).
    * @retval true - success.
    * @retval false - failure.
    */
    bool TCPServerStart(uint32_t port = 333);
    
    /** Indicates whether a new packet has arrived from a connected client. 
    * 
    * @retval true - A new packet has arrived.
    * @retval false - No new packets have arrived.
    */
    bool TCPPacketReceived(void);
    
    /** Reads the current received TCP packet
    *
    * @param ID Pointer for the ID of client for the new received packet.
    * @param LEN Pointer for the length of the new received packet.
    * @param DATA String pointer for the string of the new received packet.
    */
    void TCPPacketRead(uint8_t *ID, uint32_t *LEN, char DATA[]);
    
    /** Send a data packet using TCP to a specified client.
    * 
    * @param ID The identifier of client to send the packet to (available value: 0 - 4). 
    * @param LEN The length of packet to send. 
    * @param DATA The buffer of data to send. 
    * @retval true - success.
    * @retval false - failure.
    */
    bool TCPPacketSend(uint8_t ID, uint32_t LEN, char DATA[]);
    
    
  private:
    Serial esp;             //Serial object for connecting to ESP8266
    DigitalOut _rst;
    bool mDEBUG;   
    Timer t;
    
    char packetData[1000];
    int packetID;
    int packetLen;
    char rxtemp[1000];
    int rxtemp_idx;
    bool new_rxmsg;     //flag to indicate if a new complete message was received
    bool rxpass1, rxpass2, rxpass3;
    void RxInterrupt(void); //Interrupt to receive data when TCP server is set
    
    bool eAT(void);
    bool eATRST(void);
    bool eATGMR(string &version);
    bool sATCWMODE(uint8_t mode);
    bool sATCIPMUX(uint8_t mode);
    bool eATCWLAP(string &list);
    bool sATCWJAP(string ssid, string pwd);
    bool eATCIFSR(string &list);
    bool sATCIPSERVER(uint8_t mode, uint32_t port = 333);
    bool sATCIPSENDMultiple(uint8_t mux_id, char buffer[], uint32_t len);
    bool sATCWSAP(string ssid, string pwd, uint8_t chl, uint8_t ecn);
    bool sATCWDHCP(bool enabled, int mode);
    bool sATCIPSTA(string ip);
    bool sATCIPAP(string ip);
/*

bool qATCWMODE(uint8_t *mode);



bool eATCWQAP(void);
bool sATCWSAP(string ssid, string pwd, uint8_t chl, uint8_t ecn);
bool eATCWLIF(string &list);

bool eATCIPSTATUS(string &list);
bool eATCIFSR(string &list);

    */
    
    
    
    
    /** Verify whether ESP8266 is connected correctly or not.
    * @param None
    * 
    * @retval true - connected.
    * @retval false - not connected or found.
    */
    bool kick(void);
    
    /** Restart ESP8266 by "AT+RST". This method will take 3 seconds or more. 
    *
    * @retval true - success.
    * @retval false - failure.
    */
    bool reset(void);
    
    /** Sets the ESP8266's WiFi mode.
    * @param mode The WiFi mode the ESP8266 must be set to (1=Station, 2=SoftAP, 3=Station+SoftAP)
    *
    * @retval true - success.
    * @retval false - failure.
    */
    bool setWiFiMode(uint8_t mode);
    
    /** Enable of disable multiple connections on the ESP8266
    * @param onoff Value to Enable(=true) or Disable(=false) multiple connections.
    *
    * @retval true - success.
    * @retval false - failure.
    */
    bool setMux(bool onoff);
    
    /* Receive data from uart and search first target. Return true if target found, false for timeout. */
    bool recvFind(string target, uint32_t timeout = 1000);
    
    /* Receive data from uart. Return all received data if target found or timeout. */
    string recvString(string target, uint32_t timeout = 1000);
    
    /* Receive data from uart. Return all received data if one of target1 and target2 found or timeout. */
    string recvString(string target1, string target2, uint32_t timeout = 1000);

    /* Receive data from uart. Return all received data if one of target1, target2 and target3 found or timeout. */
    string recvString(string target1, string target2, string target3, uint32_t timeout = 1000);
    
    /* Receive data from uart and search first target and cut out the substring between begin and end(excluding begin and end self). 
    * Return true if target found, false for timeout. */
    bool recvFindAndFilter(string target, string begin, string end, string &data, uint32_t timeout = 1000);
    
    
};
 
#endif