/* DISCOF469SerialLCD Library v1.0
 * Copyright (c) 2018 Grant Phillips
 * grant.phillips@mandela.ac.za
 *
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */
 
#ifndef DISCOF469SerialLCD_H
#define DISCOF469SerialLCD_H
 
#include "mbed.h"
#include "BufferedSerial.h"
#include <inttypes.h>

#define LCD_WHITE   0xffffffff
#define LCD_SILVER  0xffC0C0C0
#define LCD_GRAY    0xff808080
#define LCD_BLACK   0xff000000
#define LCD_RED     0xffff0000
#define LCD_MAROON  0xff800000
#define LCD_YELLOW  0xffffff00
#define LCD_OLIVE   0xff808000
#define LCD_LIME    0xff00ff00
#define LCD_GREEN   0xff008000
#define LCD_AQUA    0xff00ffff
#define LCD_TEAL    0xff008080
#define LCD_BLUE    0xff0000ff
#define LCD_NAVY    0xff000080
#define LCD_FUCHSIA 0xffff00ff
#define LCD_PURPLE  0xff800080

/** Class library for a serial lcd implemented on a DISCO-F469NI Development board running
 * specific firmware for this purpose.
 *
 * Example:
 * @code
 * #include "mbed.h"
 * #include "DISCOF469SerialLCD.h"
 * 
 * DISCOF469SerialLCD lcd(PA_0, PA_1); //tx, rx
 * 
 * void DisplayPattern(void);
 * 
 * int main() {
 *     char buf[40];               
 *     int numtouches;             
 *     bool cleared = false;
 *     uint16_t x1, y1, x2, y2;
 * 
 *     DisplayPattern();
 *     lcd.LED1(1);
 *     wait(1.0);
 *     lcd.LED2(1);
 *     wait(1.0);
 *     lcd.LED3(1);
 *     wait(1.0);
 *     lcd.LED4(1);
 *     wait(1.0);
 *     lcd.Clear(LCD_WHITE);
 *     lcd.LED1(0);
 *     lcd.LED2(0);
 *     lcd.LED3(0);
 *     lcd.LED4(0);
 *     
 *     while(1) 
 *     {
 *         numtouches = lcd.Touches();     //read if the screen is touched
 *         if(numtouches > 0)              //has the screen been touched?
 *         {
 *             cleared = false;            //reset clear flag
 *             if(numtouches == 1)         //if one finger is on the screen
 *             {
 *                 lcd.GetTouch1(&x1, &y1);
 *                 sprintf(buf, "Touch 1: x=%3u y=%3u", x1, y1);
 *                 lcd.DrawStringAtLine(0,4,LCD_BLUE, LCD_WHITE, 0, buf);
 *                 lcd.FillCircle(x1, y1, 50, LCD_RED);
 *             }
 *             else if(numtouches == 2)    //if two fingers are on the screen
 *             {
 *                 lcd.GetTouch1(&x1, &y1);
 *                 lcd.GetTouch2(&x2, &y2);
 *                 sprintf(buf, "Touch 1: x=%3u y=%3u", x1, y1);
 *                 lcd.DrawStringAtLine(0,4,LCD_BLUE, LCD_WHITE, 0, buf);
 *                 sprintf(buf, "Touch 2: x=%3u y=%3u", x2, y2);
 *                 lcd.DrawStringAtLine(1,4,LCD_BLUE, LCD_WHITE, 0, buf);
 *                 lcd.FillCircle(x1, y1, 50, LCD_RED);
 *                 lcd.FillCircle(x2, y2, 50, LCD_BLUE);
 *             }
 *         }
 *         else
 *         {
 *             if(!cleared)                //if the screen hasn't been cleared 
 *             {
 *                 lcd.Clear(LCD_WHITE);   //clear the screen after fingers left screen
 *                 cleared = true;
 *             }
 *         }
 *     }
 * }
 * 
 * void DisplayPattern(void)
 * {
 *     lcd.FillRectangle(0, 0, 50, 479,LCD_WHITE);
 *     lcd.FillRectangle(50, 0, 50, 479,LCD_SILVER);
 *     lcd.FillRectangle(100, 0, 50, 479,LCD_GRAY);
 *     lcd.FillRectangle(150, 0, 50, 479,LCD_BLACK);
 *     lcd.FillRectangle(200, 0, 50, 479,LCD_RED);
 *     lcd.FillRectangle(250, 0, 50, 479,LCD_MAROON);
 *     lcd.FillRectangle(300, 0, 50, 479,LCD_YELLOW);
 *     lcd.FillRectangle(350, 0, 50, 479,LCD_OLIVE);
 *     lcd.FillRectangle(400, 0, 50, 479,LCD_LIME);
 *     lcd.FillRectangle(450, 0, 50, 479,LCD_GREEN);
 *     lcd.FillRectangle(500, 0, 50, 479,LCD_AQUA);
 *     lcd.FillRectangle(550, 0, 50, 479,LCD_TEAL);
 *     lcd.FillRectangle(600, 0, 50, 479,LCD_BLUE);
 *     lcd.FillRectangle(650, 0, 50, 479,LCD_NAVY);
 *     lcd.FillRectangle(700, 0, 50, 479,LCD_FUCHSIA);
 *     lcd.FillRectangle(750, 0, 50, 479,LCD_PURPLE);
 *     lcd.DrawStringAtXY(0, 0, 1, LCD_WHITE, LCD_BLACK, "White");
 *     lcd.DrawStringAtXY(50, 0, 1, LCD_WHITE, LCD_BLACK, "Silver");
 *     lcd.DrawStringAtXY(100, 0, 1, LCD_WHITE, LCD_BLACK, "Gray");
 *     lcd.DrawStringAtXY(150, 0, 1, LCD_WHITE, LCD_BLACK, "Black");
 *     lcd.DrawStringAtXY(200, 0, 1, LCD_WHITE, LCD_BLACK, "Red");
 *     lcd.DrawStringAtXY(250, 0, 1, LCD_WHITE, LCD_BLACK, "Maroon");
 *     lcd.DrawStringAtXY(300, 0, 1, LCD_WHITE, LCD_BLACK, "Yellow");
 *     lcd.DrawStringAtXY(350, 0, 1, LCD_WHITE, LCD_BLACK, "Olive");
 *     lcd.DrawStringAtXY(400, 0, 1, LCD_WHITE, LCD_BLACK, "Lime");
 *     lcd.DrawStringAtXY(450, 0, 1, LCD_WHITE, LCD_BLACK, "Green");
 *     lcd.DrawStringAtXY(500, 0, 1, LCD_WHITE, LCD_BLACK, "Aqua");
 *     lcd.DrawStringAtXY(550, 0, 1, LCD_WHITE, LCD_BLACK, "Teal");
 *     lcd.DrawStringAtXY(600, 0, 1, LCD_WHITE, LCD_BLACK, "Blue");
 *     lcd.DrawStringAtXY(650, 0, 1, LCD_WHITE, LCD_BLACK, "Navy");
 *     lcd.DrawStringAtXY(700, 0, 1, LCD_WHITE, LCD_BLACK, "Fuchsia");
 *     lcd.DrawStringAtXY(750, 0, 1, LCD_WHITE, LCD_BLACK, "Purple");
 * }
 * @endcode
 */
 
class DISCOF469SerialLCD {
  public:
    /** Create a DISCOF469SerialLCD object for a graphics LCD connected to the specified pins. 
    * @param Tx USART TX pin used to connect to Nextion LCD's RX pin
    * @param Rx USART RX pin used to connect to Nextion LCD's TX pin
    */
    DISCOF469SerialLCD(PinName Tx, PinName Rx);
    
    /** 
    * @brief Clears the LCD by filling it with the specified color pixels. 
    * @param Color Color to fill the screen with - represented in ARGB8888 color space format.
    */
    void Clear(uint32_t Color);

    /**
    * @brief  Draws a pixel on the LCD.
    * @param  Xpos X position
    * @param  Ypos Y position
    * @param  Color Pixel color in ARGB mode (8-8-8-8)
    */
    void DrawPixel(uint16_t Xpos, uint16_t Ypos, uint32_t Color);

    /**
    * @brief  Reads the color of an LCD pixel.
    * @param  Xpos X position
    * @param  Ypos Y position
    * @retval uint32_t Pixel color in ARGB mode (8-8-8-8)
    */
    uint32_t ReadPixel(uint16_t Xpos, uint16_t Ypos);
    
    /** 
    * @brief  Draws a line on the LCD.
    * @param  x1 Point 1 X position
    * @param  y1 Point 1 Y position
    * @param  x2 Point 2 X position
    * @param  y2 Point 2 Y position
    * @param Color Line color in ARGB mode (8-8-8-8)
    */
    void DrawLine(uint16_t x1, uint16_t y1, uint16_t x2, uint16_t y2, uint32_t Color);

    /**
    * @brief  Draws a rectangle on the LCD.
    * @param  Xpos X position
    * @param  Ypos Y position
    * @param  Width Rectangle width
    * @param  Height Rectangle height
    * @param  Color Rectangle color in ARGB mode (8-8-8-8)
    */
    void DrawRectangle(uint16_t Xpos, uint16_t Ypos, uint16_t Width, uint16_t Height, uint32_t Color);
    
    /**
    * @brief  Draws a filled rectangle on the LCD.
    * @param  Xpos X position
    * @param  Ypos Y position
    * @param  Width Rectangle width
    * @param  Height Rectangle height
    * @param  Color Rectangle color in ARGB mode (8-8-8-8)
    */
    void FillRectangle(uint16_t Xpos, uint16_t Ypos, uint16_t Width, uint16_t Height, uint32_t Color);

    /**
    * @brief  Draws a circle on the LCD.
    * @param  Xpos X position
    * @param  Ypos Y position
    * @param  Radius Circle radius
    * @param  Color Circle color in ARGB mode (8-8-8-8)
    */
    void DrawCircle(uint16_t Xpos, uint16_t Ypos, uint16_t Radius, uint32_t Color);
    
    /**
    * @brief  Draws a filled circle on the LCD.
    * @param  Xpos X position
    * @param  Ypos Y position
    * @param  Radius Circle radius
    * @param  Color Circle color in ARGB mode (8-8-8-8)
    */
    void FillCircle(uint16_t Xpos, uint16_t Ypos, uint16_t Radius, uint32_t Color);

    /**
    * @brief  Draws an ellipse on the LCD.
    * @param  Xpos X position
    * @param  Ypos Y position
    * @param  XRadius Ellipse X radius
    * @param  YRadius Ellipse Y radius
    * @param  Color Ellipse color in ARGB mode (8-8-8-8)
    */
    void DrawEllipse(uint16_t Xpos, uint16_t Ypos, uint16_t XRadius, uint16_t YRadius, uint32_t Color);
    
    /**
    * @brief  Draws a filled ellipse on the LCD.
    * @param  Xpos X position
    * @param  Ypos Y position
    * @param  XRadius Ellipse X radius
    * @param  YRadius Ellipse Y radius
    * @param  Color Ellipse color in ARGB mode (8-8-8-8)
    */
    void FillEllipse(uint16_t Xpos, uint16_t Ypos, uint16_t XRadius, uint16_t YRadius, uint32_t Color);
    
    /**
    * @brief  Displays a string on the LCD at Xpos,Ypos.
    * @param  Xpos X position (in pixel)
    * @param  Ypos Y position (in pixel)
    * @param  FontSize The size of the text (0 - 4)
    * @param  TextColor The text foreground color
    * @param  BackColor The text background color
    * @param  Text String to display on LCD
    */
    void DrawStringAtXY(uint16_t Xpos, uint16_t Ypos, uint8_t FontSize, uint32_t TextColor, uint32_t BackColor, char *Text);
    
    /**
    * @brief  Displays a string on the LCD at a certain line based on the font size.
    * @param  Line Line where to display the string
    * @param  FontSize The size of the text (0 - 4)
    * @param  TextColor The text foreground color
    * @param  BackColor The text background color
    * @param  Mode Display mode (0=LEFT mode; 1=CENTER mode; 2=RIGHT mode)
    * @param  Text String to display on LCD
    */
    void DrawStringAtLine(uint16_t Line, uint8_t FontSize, uint32_t TextColor, uint32_t BackColor, uint8_t Mode, char *Text);

    /** Determines if the touchscreen is being touched or not. 
    *
    * @retval uint8_t Number of simulantanous touches on screen. 0 = not touched.
    */
    uint8_t Touches(void);
           
    /**
    * @brief  Gets the x and y coordinates of the first touch on the touchscreen.
    * @param  x Pointer to the x coordinate
    * @param  y Pointer to the y coordinate
    */
    void GetTouch1(uint16_t *x, uint16_t *y);
    
    /**
    * @brief  Gets the x and y coordinates of the second touch on the touchscreen.
    * @param  x Pointer to the x coordinate
    * @param  y Pointer to the y coordinate
    */
    void GetTouch2(uint16_t *x, uint16_t *y);
    
    /**
    * @brief  Writes to the LCD's LED1.
    * @param  OnOff 1 = On; 0 = Off
    */
    void LED1(uint8_t OnOff);
    
    /**
    * @brief  Writes to the LCD's LED1.
    * @param  OnOff 1 = On; 0 = Off
    */
    void LED2(uint8_t OnOff);
    
    /**
    * @brief  Writes to the LCD's LED1.
    * @param  OnOff 1 = On; 0 = Off
    */
    void LED3(uint8_t OnOff);
    
    /**
    * @brief  Writes to the LCD's LED1.
    * @param  OnOff 1 = On; 0 = Off
    */
    void LED4(uint8_t OnOff);
    

    
  private:
    
    
    BufferedSerial lcd;             //Serial object for connecting to LCD
    //Serial pc;
    uint8_t mTouches;
    uint16_t mTouch1X, mTouch1Y,mTouch2X, mTouch2Y;
    uint32_t mReadPixelColor;
    int mcnt;
    
    char mRxMsg[40];
    int mRxIdx;
    char cmd[5];
    uint32_t data32_0;
    uint16_t data16_0, data16_1;
    uint8_t data8_0;
    void ServiceSerialRX(void); //Rx Interrupt
};
 
#endif