#include "mbed.h"
#include "TextLCD.h"

# define TEMPERATURE_TIMEOUT 10*60+5 //s

// https://github.com/andresv/OOThermometer
// mbed shows data from serial on LCD screen

TextLCD lcd(p26, p25, p24, p23, p22, p21, TextLCD::LCD16x2B); // rs, e, d4-d7
DigitalOut led1(LED1);

Serial pc(USBTX, USBRX);
Serial base(p9, p10);
Timeout syncronizator;
Timeout timeouter;

Timer in_timer;
Timer out_timer;

enum {
    SERIAL_ADDR_NODE		= 0x01,
	SERIAL_ADDR_BASE		= 0x00,
	
	MSG_THERMOMETER		    = 0x01,
	MSG_REQ_THERMOMETER		= 0x10
};

typedef struct {
    unsigned short battery;
    int light;
    int temp;
    char addr;
    char header;
} __attribute__((packed)) thermometer_msg_t;

typedef struct {
	char header;
} __attribute__((packed)) req_thermometer_msg_t;

char rx_buffer[50];
char* inserter = rx_buffer + sizeof(rx_buffer);
unsigned int bytes_received = 0;

float in_temp = 0.0;
float out_temp = 0.0;

void packet_received();
void check_timeout();
void update_out_temp();
void update_in_temp();
void request_temp();

//----------------------------------------------------------------------------
// Byte received from UART
//----------------------------------------------------------------------------
void receive() {
    led1 = !led1;
    if (base.readable()) {
        // if we have not got any data for 50 ms display data
        // because it was probably packet end
        // here it is like watchdog kick:
        syncronizator.attach(&packet_received, 0.05);
        
        *inserter = base.getc();
        inserter--;
        bytes_received++;

        if (inserter == rx_buffer)
            inserter = rx_buffer + sizeof(rx_buffer);
    }
}

//----------------------------------------------------------------------------
// Probabaly packet received from UART
//----------------------------------------------------------------------------
void packet_received() {
    thermometer_msg_t* msg = (thermometer_msg_t*)(inserter+1);
    unsigned int i = 0;

    for (i=0; i<bytes_received; i++)
        pc.printf("%02X\n", ((char*)msg)[i]);
    pc.printf("\n");
    
    if (msg->header == MSG_THERMOMETER && bytes_received == sizeof(thermometer_msg_t)) {
        if (msg->addr == SERIAL_ADDR_BASE) {
            in_temp = ((float)msg->temp)/10;
            in_timer.reset();
            update_in_temp();
        }
        else if (msg->addr == SERIAL_ADDR_NODE) {
            out_temp = ((float)msg->temp)/10;
            out_timer.reset();
            update_out_temp();
        }
    }
    
    bytes_received = 0;
    inserter = rx_buffer + sizeof(rx_buffer);
}

//----------------------------------------------------------------------------
// Check whether there is timeout condition
//----------------------------------------------------------------------------
void check_timeout() {
    pc.printf("timeout!\n");
    timeouter.attach(&check_timeout, TEMPERATURE_TIMEOUT+1);
    update_in_temp();
    update_out_temp();
}

void update_in_temp() {
    lcd.locate(2, 1);
    pc.printf("in timer %.1f\n", in_timer.read());
    if (in_timer.read() < TEMPERATURE_TIMEOUT) {
        lcd.printf("%.1f\n", in_temp);
    }
    else {
        lcd.printf("XX.X\n");
    }
}

void update_out_temp() {
    lcd.locate(10, 1);
    pc.printf("out timer %.1f\n", out_timer.read());
    if (out_timer.read() < TEMPERATURE_TIMEOUT) {
        lcd.printf("%.1f\n", out_temp);
    }
    else {
        lcd.printf("XX.X\n");
    }
}

//----------------------------------------------------------------------------
// Request temperature from base and node
//----------------------------------------------------------------------------
void request_temp() {
    req_thermometer_msg_t msg;
    unsigned int i;
    
    msg.header = MSG_REQ_THERMOMETER;
    
    for (i=0; i<sizeof(req_thermometer_msg_t); i++)
        base.putc(((char*)&msg)[i]);
}

//----------------------------------------------------------------------------
// Main
//----------------------------------------------------------------------------
int main() {
    base.baud(115200);
    pc.baud(115200);
    
    base.attach(&receive, Serial::RxIrq);
    lcd.printf("   IN     OUT\n");
    
    in_timer.start();
    out_timer.start();
    timeouter.attach(&check_timeout, TEMPERATURE_TIMEOUT+1);
    
    request_temp();
    while(1) {
    }
}
