#ifndef __LPS22HH_H__
#define __LPS22HH_H__


/* Includes ------------------------------------------------------------------*/

#include "mbed.h"
/* Defines -------------------------------------------------------------------*/
/** I2C Device Address 8 bit format  if SA0=0 -> B9 if SA0=1 -> BB **/
#define LPS22HH_I2C_ADD_H                       0xBBU
#define LPS22HH_I2C_ADD_L                       0xB9U

/** Device Identification (Who am I) **/
#define LPS22HH_ID                              0xB3U

/**
  * @}
  *
  */

/**
  * @addtogroup  LPS22HH_Sensitivity
  * @brief       These macro are maintained for back compatibility.
  *              in order to convert data into engineering units please
  *              use functions:
  *                -> _from_lsb_to_hpa(int16_t lsb)
  *                -> _from_lsb_to_celsius(int16_t lsb);
  *
  *              REMOVING the MACRO you are compliant with:
  *              MISRA-C 2012 [Dir 4.9] -> " avoid function-like macros "
  * @{
  *
  */

#define LPS22HH_FROM_LSB_TO_hPa(lsb)     (float)( lsb / 4096.0f )
#define LPS22HH_FROM_LSB_TO_degC(lsb)    (float)( lsb / 100.0f )

#define LPS22HH_WHO_AM_I                        0x0FU

#define PROPERTY_DISABLE                (0U)
#define PROPERTY_ENABLE                 (1U)


/* Typedefs ------------------------------------------------------------------*/

typedef enum
{
  LPS22HH_OK = 0,
  LPS22HH_ERROR =-1
} LPS22HHStatusTypeDef;

typedef enum {
  LPS22HH_POWER_DOWN          = 0x00,
  LPS22HH_ONE_SHOOT           = 0x08,
  LPS22HH_1_Hz                = 0x01,
  LPS22HH_10_Hz               = 0x02,
  LPS22HH_25_Hz               = 0x03,
  LPS22HH_50_Hz               = 0x04,
  LPS22HH_75_Hz               = 0x05,
  LPS22HH_1_Hz_LOW_NOISE      = 0x11,
  LPS22HH_10_Hz_LOW_NOISE     = 0x12,
  LPS22HH_25_Hz_LOW_NOISE     = 0x13,
  LPS22HH_50_Hz_LOW_NOISE     = 0x14,
  LPS22HH_75_Hz_LOW_NOISE     = 0x15,
  LPS22HH_100_Hz              = 0x06,
  LPS22HH_200_Hz              = 0x07,
} lps22hh_odr_t;

typedef int32_t (*lps22hh_write_ptr)(void *, uint8_t, uint8_t*, uint16_t);
typedef int32_t (*lps22hh_read_ptr) (void *, uint8_t, uint8_t*, uint16_t);

typedef struct {
  /** Component mandatory fields **/
  lps22hh_write_ptr  write_reg;
  lps22hh_read_ptr   read_reg;
  /** Customizable optional pointer **/
  void *handle;
} lps22hh_ctx_t;

typedef enum {
  LPS22HH_LPF_ODR_DIV_2    = 0,
  LPS22HH_LPF_ODR_DIV_9    = 2,
  LPS22HH_LPF_ODR_DIV_20   = 3,
} lps22hh_lpfp_cfg_t;

typedef union{
  int16_t i16bit[3];
  uint8_t u8bit[6];
} axis3bit16_t;

typedef union{
  int16_t i16bit;
  uint8_t u8bit[2];
} axis1bit16_t;

typedef union{
  int32_t i32bit[3];
  uint8_t u8bit[12];
} axis3bit32_t;

typedef union{
  int32_t i32bit;
  uint8_t u8bit[4];
} axis1bit32_t;

/**
  * @}
  *
  */

typedef struct{
  uint8_t bit0       : 1;
  uint8_t bit1       : 1;
  uint8_t bit2       : 1;
  uint8_t bit3       : 1;
  uint8_t bit4       : 1;
  uint8_t bit5       : 1;
  uint8_t bit6       : 1;
  uint8_t bit7       : 1;
} bitwise_t;

#define PROPERTY_DISABLE                (0U)
#define PROPERTY_ENABLE                 (1U)



/**
  * @}
  *
  */

/** @addtogroup  LPS22HH_Interfaces_Functions
  * @brief       This section provide a set of functions used to read and
  *              write a generic register of the device.
  *              MANDATORY: return 0 -> no Error.
  * @{
  *
  */

typedef int32_t (*lps22hh_write_ptr)(void *, uint8_t, uint8_t*, uint16_t);
typedef int32_t (*lps22hh_read_ptr) (void *, uint8_t, uint8_t*, uint16_t);


/**
  * @}
  *
  */

/** @defgroup LPS22HH_Infos
  * @{
  *
  */

/** I2C Device Address 8 bit format  if SA0=0 -> B9 if SA0=1 -> BB **/
#define LPS22HH_I2C_ADD_H                       0xBBU
#define LPS22HH_I2C_ADD_L                       0xB9U

/** Device Identification (Who am I) **/
#define LPS22HH_ID                              0xB3U

/**
  * @}
  *
  */

/**
  * @addtogroup  LPS22HH_Sensitivity
  * @brief       These macro are maintained for back compatibility.
  *              in order to convert data into engineering units please
  *              use functions:
  *                -> _from_lsb_to_hpa(int16_t lsb)
  *                -> _from_lsb_to_celsius(int16_t lsb);
  *
  *              REMOVING the MACRO you are compliant with:
  *              MISRA-C 2012 [Dir 4.9] -> " avoid function-like macros "
  * @{
  *
  */

#define LPS22HH_FROM_LSB_TO_hPa(lsb)     (float)( lsb / 4096.0f )
#define LPS22HH_FROM_LSB_TO_degC(lsb)    (float)( lsb / 100.0f )

/**
  * @}
  *
  */

#define LPS22HH_INTERRUPT_CFG                   0x0BU
typedef struct {
  uint8_t pe                              : 2;  /* ple + phe */
  uint8_t lir                             : 1;
  uint8_t diff_en                         : 1;
  uint8_t reset_az                        : 1;
  uint8_t autozero                        : 1;
  uint8_t reset_arp                       : 1;
  uint8_t autorefp                        : 1;
} lps22hh_interrupt_cfg_t;

#define LPS22HH_THS_P_L                         0x0CU
typedef struct {
  uint8_t ths                             : 8;
} lps22hh_ths_p_l_t;

#define LPS22HH_THS_P_H                         0x0DU
typedef struct {
  uint8_t ths                             : 7;
  uint8_t not_used_01                     : 1;
} lps22hh_ths_p_h_t;

#define LPS22HH_IF_CTRL                         0x0EU
typedef struct {
  uint8_t i2c_disable                     : 1;
  uint8_t i3c_disable                     : 1;
  uint8_t pd_dis_int1                     : 1;
  uint8_t sdo_pu_en                       : 1;
  uint8_t sda_pu_en                       : 1;
  uint8_t not_used_01                     : 2;
  uint8_t int_en_i3c                      : 1;
} lps22hh_if_ctrl_t;

#define LPS22HH_WHO_AM_I                        0x0FU
#define LPS22HH_CTRL_REG1                       0x10U
typedef struct {
  uint8_t sim                             : 1;
  uint8_t bdu                             : 1;
  uint8_t lpfp_cfg                        : 2;  /* en_lpfp + lpfp_cfg */
  uint8_t odr                             : 3;
  uint8_t not_used_01                     : 1;
} lps22hh_ctrl_reg1_t;

#define LPS22HH_CTRL_REG2                       0x11U
typedef struct {
  uint8_t one_shot                        : 1;
  uint8_t low_noise_en                    : 1;
  uint8_t swreset                         : 1;
  uint8_t not_used_01                     : 1;
  uint8_t if_add_inc                      : 1;
  uint8_t pp_od                           : 1;
  uint8_t int_h_l                         : 1;
  uint8_t boot                            : 1;
} lps22hh_ctrl_reg2_t;

#define LPS22HH_CTRL_REG3                       0x12U
typedef struct {
  uint8_t int_s                           : 2;
  uint8_t drdy                            : 1;
  uint8_t int_f_ovr                       : 1;
  uint8_t int_f_wtm                       : 1;
  uint8_t int_f_full                      : 1;
  uint8_t not_used_01                     : 2;
} lps22hh_ctrl_reg3_t;

#define LPS22HH_FIFO_CTRL                       0x13U
typedef struct {
  uint8_t f_mode                          : 3;  /* f_mode + trig_modes */
  uint8_t stop_on_wtm                     : 1;
  uint8_t not_used_01                     : 4;
} lps22hh_fifo_ctrl_t;

#define LPS22HH_FIFO_WTM                        0x14U
typedef struct {
  uint8_t wtm                             : 7;
  uint8_t not_used_01                     : 1;
} lps22hh_fifo_wtm_t;

#define LPS22HH_REF_P_XL                        0x15U
#define LPS22HH_REF_P_L                         0x16U
#define LPS22HH_RPDS_L                          0x18U
#define LPS22HH_RPDS_H                          0x19U
#define LPS22HH_INT_SOURCE                      0x24U
typedef struct {
  uint8_t ph                              : 1;
  uint8_t pl                              : 1;
  uint8_t ia                              : 1;
  uint8_t not_used_01                     : 5;
} lps22hh_int_source_t;

#define LPS22HH_FIFO_STATUS1                    0x25U
#define LPS22HH_FIFO_STATUS2                    0x26U
typedef struct {
  uint8_t not_used_01                     : 5;
  uint8_t fifo_full_ia                    : 1;
  uint8_t fifo_ovr_ia                     : 1;
  uint8_t fifo_wtm_ia                     : 1;
} lps22hh_fifo_status2_t;

#define LPS22HH_STATUS                          0x27U
typedef struct {
  uint8_t p_da                            : 1;
  uint8_t t_da                            : 1;
  uint8_t not_used_01                     : 2;
  uint8_t p_or                            : 1;
  uint8_t t_or                            : 1;
  uint8_t not_used_02                     : 2;
} lps22hh_status_t;

#define LPS22HH_PRESSURE_OUT_XL                 0x28U
#define LPS22HH_PRESSURE_OUT_L                  0x29U
#define LPS22HH_PRESSURE_OUT_H                  0x2AU
#define LPS22HH_TEMP_OUT_L                      0x2BU
#define LPS22HH_TEMP_OUT_H                      0x2CU
#define LPS22HH_FIFO_DATA_OUT_PRESS_XL          0x78U
#define LPS22HH_FIFO_DATA_OUT_PRESS_L           0x79U
#define LPS22HH_FIFO_DATA_OUT_PRESS_H           0x7AU
#define LPS22HH_FIFO_DATA_OUT_TEMP_L            0x7BU
#define LPS22HH_FIFO_DATA_OUT_TEMP_H            0x7CU

/**
  * @defgroup LPS22HH_Register_Union
  * @brief    This union group all the registers that has a bitfield
  *           description.
  *           This union is useful but not need by the driver.
  *
  *           REMOVING this union you are compliant with:
  *           MISRA-C 2012 [Rule 19.2] -> " Union are not allowed "
  *
  * @{
  *
  */
typedef union{
  lps22hh_interrupt_cfg_t        interrupt_cfg;
  lps22hh_if_ctrl_t              if_ctrl;
  lps22hh_ctrl_reg1_t            ctrl_reg1;
  lps22hh_ctrl_reg2_t            ctrl_reg2;
  lps22hh_ctrl_reg3_t            ctrl_reg3;
  lps22hh_fifo_ctrl_t            fifo_ctrl;
  lps22hh_fifo_wtm_t             fifo_wtm;
  lps22hh_int_source_t           int_source;
  lps22hh_fifo_status2_t         fifo_status2;
  lps22hh_status_t               status;
  bitwise_t                      bitwise;
  uint8_t                        byte;
} lps22hh_reg_t;

typedef enum {
  LPS22HH_BYPASS_MODE            = 0,
  LPS22HH_FIFO_MODE              = 1,
  LPS22HH_STREAM_MODE            = 2,
  LPS22HH_DYNAMIC_STREAM_MODE    = 3,
  LPS22HH_BYPASS_TO_FIFO_MODE    = 5,
  LPS22HH_BYPASS_TO_STREAM_MODE  = 6,
  LPS22HH_STREAM_TO_FIFO_MODE    = 7,
} lps22hh_f_mode_t;

/**
  * @}
  *
  */


/* Class Declaration ---------------------------------------------------------*/
   
/**
 * Abstract class of a LPS22HH pressure sensor.
 */
class LPS22HH
{
  public:
    LPS22HH(PinName sda, PinName scl);
    
    LPS22HHStatusTypeDef begin();
    LPS22HHStatusTypeDef end();
    LPS22HHStatusTypeDef ReadID(uint8_t *Id);
    LPS22HHStatusTypeDef Enable();
    LPS22HHStatusTypeDef Disable();
    LPS22HHStatusTypeDef GetOutputDataRate(float *Odr);
    LPS22HHStatusTypeDef SetOutputDataRate(float Odr);
    ////LPS22HHStatusTypeDef GetPressure(float *Value);
    float GetPressure();
    LPS22HHStatusTypeDef Get_PRESS_DRDY_Status(uint8_t *Status);
    
    LPS22HHStatusTypeDef GetTemperature(float *Value);
    LPS22HHStatusTypeDef Get_TEMP_DRDY_Status(uint8_t *Status);
    
    LPS22HHStatusTypeDef Read_Reg(uint8_t reg, uint8_t *Data);
    LPS22HHStatusTypeDef Write_Reg(uint8_t reg, uint8_t Data);
    
    LPS22HHStatusTypeDef Get_FIFO_Data(float *Press, float *Temp);
    LPS22HHStatusTypeDef Get_FIFO_FTh_Status(uint8_t *Status);
    LPS22HHStatusTypeDef Get_FIFO_Full_Status(uint8_t *Status);
    LPS22HHStatusTypeDef Get_FIFO_Ovr_Status(uint8_t *Status);
    LPS22HHStatusTypeDef Get_FIFO_Level(uint8_t *Status);
    LPS22HHStatusTypeDef Reset_FIFO_Interrupt(uint8_t interrupt);
    LPS22HHStatusTypeDef Set_FIFO_Interrupt(uint8_t interrupt);
    LPS22HHStatusTypeDef Set_FIFO_Mode(uint8_t Mode);
    LPS22HHStatusTypeDef Set_FIFO_Watermark_Level(uint8_t Watermark);
    LPS22HHStatusTypeDef Stop_FIFO_On_Watermark(uint8_t Stop);
    
    LPS22HHStatusTypeDef Set_One_Shot();
    LPS22HHStatusTypeDef Get_One_Shot_Status(uint8_t *Status);

    
    /**
     * @brief Utility function to read data.
     * @param  pBuffer: pointer to data to be read.
     * @param  RegisterAddr: specifies internal address register to be read.
     * @param  NumByteToRead: number of bytes to be read.
     * @retval 0 if ok, an error code otherwise.
     */
    uint8_t IO_Read(uint8_t* pBuffer, uint8_t RegisterAddr, uint16_t NumByteToRead)
    {   
        char tempBuff[NumByteToRead];
        uint8_t devAddr = (uint8_t)(((address) >> 1) & 0x7F);
        char regAddr = (char)RegisterAddr;
        _i2c.write((int)devAddr, &regAddr, 1);    
        //_i2c->beginTransmission(((uint8_t)(((address) >> 1) & 0x7F)));
        //dev_i2c->write(RegisterAddr);
        //dev_i2c->endTransmission(false);
        _i2c.read(devAddr,tempBuff,NumByteToRead);
        
        memcpy(pBuffer,tempBuff,NumByteToRead);
        //dev_i2c->requestFrom(((uint8_t)(((address) >> 1) & 0x7F)), (uint8_t) NumByteToRead);

        /*
        int i=0;
        while (dev_i2c->available()) {
          pBuffer[i] = dev_i2c->read();
          i++;
        }
        */
        return 0;
      

      
    }
    
    /**
     * @brief Utility function to write data.
     * @param  pBuffer: pointer to data to be written.
     * @param  RegisterAddr: specifies internal address register to be written.
     * @param  NumByteToWrite: number of bytes to write.
     * @retval 0 if ok, an error code otherwise.
     */
    uint8_t IO_Write(uint8_t* pBuffer, uint8_t RegisterAddr, uint16_t NumByteToWrite)
    {  
      uint8_t devAddr = (uint8_t)(((address) >> 1) & 0x7F);
      char tData[NumByteToWrite+1];      
      tData[0] = RegisterAddr;
      memcpy(tData+1,pBuffer,NumByteToWrite);
      
      _i2c.write(devAddr,tData,NumByteToWrite+1);

        return 0;
      
    }

  private:
    LPS22HHStatusTypeDef SetOutputDataRate_When_Enabled(float Odr);
    LPS22HHStatusTypeDef SetOutputDataRate_When_Disabled(float Odr);

    /* Helper classes. */
    I2C _i2c;
    
    /* Configuration */
    uint8_t address;
        
    lps22hh_odr_t last_odr;
    uint8_t enabled;
    
    lps22hh_ctx_t reg_ctx;
    
};

#ifdef __cplusplus
 extern "C" {
#endif
int32_t LPS22HH_io_write( void *handle, uint8_t WriteAddr, uint8_t *pBuffer, uint16_t nBytesToWrite );
int32_t LPS22HH_io_read( void *handle, uint8_t ReadAddr, uint8_t *pBuffer, uint16_t nBytesToRead );
#ifdef __cplusplus
  }
#endif

#endif