#include "LPS22HH.h"

int32_t lps22hh_write_reg(lps22hh_ctx_t* ctx, uint8_t reg, uint8_t* data, uint16_t len);
int32_t lps22hh_read_reg(lps22hh_ctx_t* ctx, uint8_t reg, uint8_t* data, uint16_t len);

int32_t lps22hh_data_rate_set(lps22hh_ctx_t *ctx, lps22hh_odr_t val);
int32_t lps22hh_data_rate_get(lps22hh_ctx_t *ctx, lps22hh_odr_t *val);

int32_t lps22hh_lp_bandwidth_set(lps22hh_ctx_t *ctx, lps22hh_lpfp_cfg_t val);
int32_t lps22hh_lp_bandwidth_get(lps22hh_ctx_t *ctx, lps22hh_lpfp_cfg_t *val);

int32_t lps22hh_block_data_update_set(lps22hh_ctx_t *ctx, uint8_t val);
int32_t lps22hh_auto_increment_set(lps22hh_ctx_t *ctx, uint8_t val);

int32_t lps22hh_pressure_raw_get(lps22hh_ctx_t *ctx, uint8_t *buff);
int32_t lps22hh_temperature_raw_get(lps22hh_ctx_t *ctx, uint8_t *buff);

int32_t lps22hh_press_flag_data_ready_get(lps22hh_ctx_t *ctx, uint8_t *val);
int32_t lps22hh_temp_flag_data_ready_get(lps22hh_ctx_t *ctx, uint8_t *val);

int32_t lps22hh_fifo_pressure_raw_get(lps22hh_ctx_t *ctx, uint8_t *buff);
int32_t lps22hh_fifo_temperature_raw_get(lps22hh_ctx_t *ctx, uint8_t *buff);
int32_t lps22hh_fifo_wtm_flag_get(lps22hh_ctx_t *ctx, uint8_t *val);
int32_t lps22hh_fifo_ovr_flag_get(lps22hh_ctx_t *ctx, uint8_t *val);
int32_t lps22hh_fifo_full_flag_get(lps22hh_ctx_t *ctx, uint8_t *val);
int32_t lps22hh_fifo_src_get(lps22hh_ctx_t *ctx, lps22hh_fifo_status2_t *val);
int32_t lps22hh_fifo_data_level_get(lps22hh_ctx_t *ctx, uint8_t *buff);
int32_t lps22hh_fifo_threshold_on_int_set(lps22hh_ctx_t *ctx, uint8_t val);
int32_t lps22hh_fifo_full_on_int_set(lps22hh_ctx_t *ctx, uint8_t val);
int32_t lps22hh_fifo_ovr_on_int_set(lps22hh_ctx_t *ctx, uint8_t val);
int32_t lps22hh_fifo_stop_on_wtm_set(lps22hh_ctx_t *ctx, uint8_t val);
int32_t lps22hh_fifo_mode_get(lps22hh_ctx_t *ctx, lps22hh_f_mode_t *val);
int32_t lps22hh_fifo_mode_set(lps22hh_ctx_t *ctx, lps22hh_f_mode_t val);
int32_t lps22hh_fifo_watermark_set(lps22hh_ctx_t *ctx, uint8_t val);

/* Class Implementation ------------------------------------------------------*/

/** Constructor
 * @param i2c object of an helper class which handles the I2C peripheral
 * @param address the address of the component's instance
 */
LPS22HH::LPS22HH(PinName sda, PinName scl) : _i2c(sda, scl)
{
  address=LPS22HH_I2C_ADD_H;
  reg_ctx.write_reg = LPS22HH_io_write;
  reg_ctx.read_reg = LPS22HH_io_read;
  reg_ctx.handle = (void *)this;
  enabled = 0;
}

/** Constructor
 * @param spi object of an helper class which handles the SPI peripheral
 * @param cs_pin the chip select pin
 * @param spi_speed the SPI speed
 
LPS22HH::LPS22HH(SPIClass *spi, int cs_pin, uint32_t spi_speed) : dev_spi(spi), cs_pin(cs_pin), spi_speed(spi_speed)
{
  reg_ctx.write_reg = LPS22HH_io_write;
  reg_ctx.read_reg = LPS22HH_io_read;
  reg_ctx.handle = (void *)this;
  dev_i2c = NULL;
  address = 0;
  enabled = 0;
}
*/

/**
 * @brief  Configure the sensor in order to be used
 * @retval 0 in case of success, an error code otherwise
 */
LPS22HHStatusTypeDef LPS22HH::begin()
{
  
  /* Power down the device, set Low Noise Enable (bit 5), clear One Shot (bit 4) */
  if (lps22hh_data_rate_set(&reg_ctx, (lps22hh_odr_t)(LPS22HH_POWER_DOWN | 0x10)) != LPS22HH_OK)
  {
    return LPS22HH_ERROR;
  }

  /* Disable low-pass filter on LPS22HH pressure data */
  if (lps22hh_lp_bandwidth_set(&reg_ctx, LPS22HH_LPF_ODR_DIV_2) != LPS22HH_OK)
  {
    return LPS22HH_ERROR;
  }

  /* Set block data update mode */
  if (lps22hh_block_data_update_set(&reg_ctx, PROPERTY_ENABLE) != LPS22HH_OK)
  {
    return LPS22HH_ERROR;
  }

  /* Set autoincrement for multi-byte read/write */
  if (lps22hh_auto_increment_set(&reg_ctx, PROPERTY_ENABLE) != LPS22HH_OK)
  {
    return LPS22HH_ERROR;
  }

  last_odr = LPS22HH_25_Hz;  
  enabled = 0;

  return LPS22HH_OK;
}

/**
 * @brief  Disable the sensor and relative resources
 * @retval 0 in case of success, an error code otherwise
 */
LPS22HHStatusTypeDef LPS22HH::end()
{
  /* Disable pressure and temperature sensor */
  if (Disable() != LPS22HH_OK)
  {
    return LPS22HH_ERROR;
  }

  
  return LPS22HH_OK;
}

/**
 * @brief  Get WHO_AM_I value
 * @param  Id the WHO_AM_I value
 * @retval 0 in case of success, an error code otherwise
 */
LPS22HHStatusTypeDef LPS22HH::ReadID(uint8_t *Id)
{
  
  //lps22hh_device_id_get(lps22hh_ctx_t *ctx, uint8_t *buff)
  int32_t ret;
  ret =  lps22hh_read_reg(&reg_ctx, LPS22HH_WHO_AM_I, Id, 1);
  
  
  if (ret != LPS22HH_OK)
  {
    return LPS22HH_ERROR;
  }

  return LPS22HH_OK;
}

/**
 * @brief  Enable the LPS22HH pressure sensor
 * @retval 0 in case of success, an error code otherwise
 */
LPS22HHStatusTypeDef LPS22HH::Enable()
{
  /* Check if the component is already enabled */
  if (enabled == 1U)
  {
    return LPS22HH_OK;
  }

  /* Output data rate selection. */
  if (lps22hh_data_rate_set(&reg_ctx, last_odr) != LPS22HH_OK)
  {
    return LPS22HH_ERROR;
  }

  enabled = 1;

  return LPS22HH_OK;
}

/**
 * @brief  Disable the LPS22HH pressure sensor
 * @retval 0 in case of success, an error code otherwise
 */
LPS22HHStatusTypeDef LPS22HH::Disable()
{
  /* Check if the component is already disabled */
  if (enabled == 0U)
  {
    return LPS22HH_OK;
  }

  lps22hh_ctrl_reg1_t ctrl_reg1;
  lps22hh_ctrl_reg2_t ctrl_reg2;
  int32_t ret;
  
  //lps22hh_data_rate_get(&reg_ctx, &last_odr)

  ret = lps22hh_read_reg(&reg_ctx, LPS22HH_CTRL_REG1, (uint8_t*)&ctrl_reg1, 1);
  if (ret == 0) {
    ret = lps22hh_read_reg(&reg_ctx, LPS22HH_CTRL_REG2, (uint8_t*)&ctrl_reg2, 1);
  }
  if (ret == 0) {
    ret = lps22hh_read_reg(&reg_ctx, LPS22HH_CTRL_REG2, (uint8_t*)&ctrl_reg2, 1);
    switch (((ctrl_reg2.low_noise_en << 4) + (ctrl_reg2.one_shot << 3) +
            ctrl_reg1.odr )) {
      case LPS22HH_POWER_DOWN:
        last_odr = LPS22HH_POWER_DOWN;
        break;
      case LPS22HH_ONE_SHOOT:
        last_odr = LPS22HH_ONE_SHOOT;
        break;
      case LPS22HH_1_Hz:
        last_odr = LPS22HH_1_Hz;
        break;
      case LPS22HH_10_Hz:
        last_odr = LPS22HH_10_Hz;
        break;
      case LPS22HH_25_Hz:
        last_odr = LPS22HH_25_Hz;
        break;
      case LPS22HH_50_Hz:
        last_odr = LPS22HH_50_Hz;
        break;
      case LPS22HH_75_Hz:
        last_odr = LPS22HH_75_Hz;
        break;
      case LPS22HH_1_Hz_LOW_NOISE:
        last_odr = LPS22HH_1_Hz_LOW_NOISE;
        break;
      case LPS22HH_10_Hz_LOW_NOISE:
        last_odr = LPS22HH_10_Hz_LOW_NOISE;
        break;
      case LPS22HH_25_Hz_LOW_NOISE:
        last_odr = LPS22HH_25_Hz_LOW_NOISE;
        break;
      case LPS22HH_50_Hz_LOW_NOISE:
        last_odr = LPS22HH_50_Hz_LOW_NOISE;
        break;
      case LPS22HH_75_Hz_LOW_NOISE:
        last_odr = LPS22HH_75_Hz_LOW_NOISE;
        break;
      case LPS22HH_100_Hz:
        last_odr = LPS22HH_100_Hz;
        break;
      case LPS22HH_200_Hz:
        last_odr = LPS22HH_200_Hz;
        break;
      default:
        last_odr = LPS22HH_POWER_DOWN;
        break;
    }
  }
  

  /* Get current output data rate. */
  if (ret != LPS22HH_OK)
  {
    return LPS22HH_ERROR;
  }
   /* Output data rate selection - power down. */
  if (lps22hh_data_rate_set(&reg_ctx, LPS22HH_POWER_DOWN) != LPS22HH_OK)
  {
    return LPS22HH_ERROR;
  }


  enabled = 0;

  return LPS22HH_OK;
}


/**
 * @brief  Get output data rate
 * @param  Odr the output data rate value
 * @retval 0 in case of success, an error code otherwise
 */
LPS22HHStatusTypeDef LPS22HH::GetOutputDataRate(float *Odr)
{
  LPS22HHStatusTypeDef result = LPS22HH_OK;
  lps22hh_odr_t odr_low_level;

lps22hh_ctrl_reg1_t ctrl_reg1;
  lps22hh_ctrl_reg2_t ctrl_reg2;
  int32_t ret;

  ret = lps22hh_read_reg(&reg_ctx, LPS22HH_CTRL_REG1, (uint8_t*)&ctrl_reg1, 1);
  if (ret == 0) {
    ret = lps22hh_read_reg(&reg_ctx, LPS22HH_CTRL_REG2, (uint8_t*)&ctrl_reg2, 1);
  }
  if (ret == 0) {
    ret = lps22hh_read_reg(&reg_ctx, LPS22HH_CTRL_REG2, (uint8_t*)&ctrl_reg2, 1);
    switch (((ctrl_reg2.low_noise_en << 4) + (ctrl_reg2.one_shot << 3) +
            ctrl_reg1.odr )) {
      case LPS22HH_POWER_DOWN:
        odr_low_level = LPS22HH_POWER_DOWN;
        break;
      case LPS22HH_ONE_SHOOT:
        odr_low_level = LPS22HH_ONE_SHOOT;
        break;
      case LPS22HH_1_Hz:
        odr_low_level = LPS22HH_1_Hz;
        break;
      case LPS22HH_10_Hz:
        odr_low_level = LPS22HH_10_Hz;
        break;
      case LPS22HH_25_Hz:
        odr_low_level = LPS22HH_25_Hz;
        break;
      case LPS22HH_50_Hz:
        odr_low_level = LPS22HH_50_Hz;
        break;
      case LPS22HH_75_Hz:
        odr_low_level = LPS22HH_75_Hz;
        break;
      case LPS22HH_1_Hz_LOW_NOISE:
        odr_low_level = LPS22HH_1_Hz_LOW_NOISE;
        break;
      case LPS22HH_10_Hz_LOW_NOISE:
        odr_low_level = LPS22HH_10_Hz_LOW_NOISE;
        break;
      case LPS22HH_25_Hz_LOW_NOISE:
        odr_low_level = LPS22HH_25_Hz_LOW_NOISE;
        break;
      case LPS22HH_50_Hz_LOW_NOISE:
        odr_low_level = LPS22HH_50_Hz_LOW_NOISE;
        break;
      case LPS22HH_75_Hz_LOW_NOISE:
        odr_low_level = LPS22HH_75_Hz_LOW_NOISE;
        break;
      case LPS22HH_100_Hz:
        odr_low_level = LPS22HH_100_Hz;
        break;
      case LPS22HH_200_Hz:
        odr_low_level = LPS22HH_200_Hz;
        break;
      default:
        odr_low_level = LPS22HH_POWER_DOWN;
        break;
    }
  }

  if (ret != LPS22HH_OK)
  {
    return LPS22HH_ERROR;
  }

  switch (odr_low_level)
  {
    case LPS22HH_POWER_DOWN:
      *Odr = 0.0f;
      break;

    case LPS22HH_1_Hz:
      *Odr = 1.0f;
      break;

    case LPS22HH_10_Hz:
      *Odr = 10.0f;
      break;

    case LPS22HH_25_Hz:
      *Odr = 25.0f;
      break;

    case LPS22HH_50_Hz:
      *Odr = 50.0f;
      break;

    case LPS22HH_75_Hz:
      *Odr = 75.0f;
      break;

    case LPS22HH_100_Hz:
      *Odr = 100.0f;
      break;

    case LPS22HH_200_Hz:
      *Odr = 200.0f;
      break;

    default:
      result = LPS22HH_ERROR;
      break;
  }

  return result;
}

/**
 * @brief  Set the LPS22HH pressure sensor output data rate
 * @param  Odr the output data rate value to be set
 * @retval 0 in case of success, an error code otherwise
 */
LPS22HHStatusTypeDef LPS22HH::SetOutputDataRate(float Odr)
{
  /* Check if the component is enabled */
  if (enabled == 1U)
  {
    return SetOutputDataRate_When_Enabled(Odr);
  }
  else
  {
    return SetOutputDataRate_When_Disabled(Odr);
  }
}


/**
 * @brief  Set output data rate
 * @param  Odr the output data rate value to be set
 * @retval 0 in case of success, an error code otherwise
 */
LPS22HHStatusTypeDef LPS22HH::SetOutputDataRate_When_Enabled(float Odr)
{
  lps22hh_odr_t new_odr;

  new_odr = (Odr <=   1.0f) ? LPS22HH_1_Hz
          : (Odr <=  10.0f) ? LPS22HH_10_Hz
          : (Odr <=  25.0f) ? LPS22HH_25_Hz
          : (Odr <=  50.0f) ? LPS22HH_50_Hz
          : (Odr <=  75.0f) ? LPS22HH_75_Hz
          : (Odr <= 100.0f) ? LPS22HH_100_Hz
          :                   LPS22HH_200_Hz;

  if (lps22hh_data_rate_set(&reg_ctx, new_odr) != LPS22HH_OK)
  {
    return LPS22HH_ERROR;
  }

  if (lps22hh_data_rate_get(&reg_ctx, &last_odr) != LPS22HH_OK)
  {
    return LPS22HH_ERROR;
  }

  return LPS22HH_OK;
}

/**
 * @brief  Set output data rate when disabled
 * @param  Odr the output data rate value to be set
 * @retval 0 in case of success, an error code otherwise
 */
LPS22HHStatusTypeDef LPS22HH::SetOutputDataRate_When_Disabled(float Odr)
{
  last_odr = (Odr <=   1.0f) ? LPS22HH_1_Hz
                 : (Odr <=  10.0f) ? LPS22HH_10_Hz
                 : (Odr <=  25.0f) ? LPS22HH_25_Hz
                 : (Odr <=  50.0f) ? LPS22HH_50_Hz
                 : (Odr <=  75.0f) ? LPS22HH_75_Hz
                 : (Odr <= 100.0f) ? LPS22HH_100_Hz
                 :                   LPS22HH_200_Hz;

  return LPS22HH_OK;
}

/**
 * @brief  Get the LPS22HH pressure value
 * @param  Value pointer where the pressure value is written
 * @retval 0 in case of success, an error code otherwise
 */
////LPS22HHStatusTypeDef LPS22HH::GetPressure(float *Value)
float LPS22HH::GetPressure()
{
  axis1bit32_t data_raw_pressure;

  (void)memset(data_raw_pressure.u8bit, 0x00, sizeof(int32_t));
  if (lps22hh_pressure_raw_get(&reg_ctx, data_raw_pressure.u8bit) != LPS22HH_OK)
  {
    return LPS22HH_ERROR;
  }

  ////*Value = LPS22HH_FROM_LSB_TO_hPa((float)(data_raw_pressure.i32bit));
  return LPS22HH_FROM_LSB_TO_hPa((float)(data_raw_pressure.i32bit));
  
  ////return LPS22HH_OK;
}

/**
 * @brief  Get the LPS22HH pressure data ready bit value
 * @param  Status the status of data ready bit
 * @retval 0 in case of success, an error code otherwise
 */
LPS22HHStatusTypeDef LPS22HH::Get_PRESS_DRDY_Status(uint8_t *Status)
{
  if (lps22hh_press_flag_data_ready_get(&reg_ctx, Status) != LPS22HH_OK)
  {
    return LPS22HH_ERROR;
  }

  return LPS22HH_OK;
}

/**
 * @brief  Get the LPS22HH temperature value
 * @param  Value pointer where the temperature value is written
 * @retval 0 in case of success, an error code otherwise
 */
LPS22HHStatusTypeDef LPS22HH::GetTemperature(float *Value)
{
  axis1bit16_t data_raw_temperature;

  (void)memset(data_raw_temperature.u8bit, 0x00, sizeof(int16_t));
  if (lps22hh_temperature_raw_get(&reg_ctx, data_raw_temperature.u8bit) != LPS22HH_OK)
  {
    return LPS22HH_ERROR;
  }

  *Value = LPS22HH_FROM_LSB_TO_degC((float)(data_raw_temperature.i16bit));

  return LPS22HH_OK;
}

/**
 * @brief  Get the LPS22HH temperature data ready bit value
 * @param  Status the status of data ready bit
 * @retval 0 in case of success, an error code otherwise
 */
LPS22HHStatusTypeDef LPS22HH::Get_TEMP_DRDY_Status(uint8_t *Status)
{
  if (lps22hh_temp_flag_data_ready_get(&reg_ctx, Status) != LPS22HH_OK)
  {
    return LPS22HH_ERROR;
  }

  return LPS22HH_OK;
}

/**
 * @brief  Get the LPS22HH register value
 * @param  Reg address to be written
 * @param  Data value to be written
 * @retval 0 in case of success, an error code otherwise
 */
LPS22HHStatusTypeDef LPS22HH::Read_Reg(uint8_t Reg, uint8_t *Data)
{
  if (lps22hh_read_reg(&reg_ctx, Reg, Data, 1) != LPS22HH_OK)
  {
    return LPS22HH_ERROR;
  }

  return LPS22HH_OK;
}

/**
 * @brief  Set the LPS22HH register value
 * @param  pObj the device pObj
 * @param  Reg address to be written
 * @param  Data value to be written
 * @retval 0 in case of success, an error code otherwise
 */
LPS22HHStatusTypeDef LPS22HH::Write_Reg(uint8_t Reg, uint8_t Data)
{
  if (lps22hh_write_reg(&reg_ctx, Reg, &Data, 1) != LPS22HH_OK)
  {
    return LPS22HH_ERROR;
  }

  return LPS22HH_OK;
}

/**
 * @brief  Get the LPS22HH FIFO data level
 * @param  Status the status of data ready bit
 * @retval 0 in case of success, an error code otherwise
 */
LPS22HHStatusTypeDef LPS22HH::Get_FIFO_Data(float *Press, float *Temp)
{
  axis1bit32_t data_raw_pressure;
  axis1bit16_t data_raw_temperature;

  (void)memset(data_raw_pressure.u8bit, 0x00, sizeof(int32_t));
  if (lps22hh_fifo_pressure_raw_get(&reg_ctx, data_raw_pressure.u8bit) != LPS22HH_OK)
  {
    return LPS22HH_ERROR;
  }

  *Press = LPS22HH_FROM_LSB_TO_hPa((float)(data_raw_pressure.i32bit));

  (void)memset(data_raw_temperature.u8bit, 0x00, sizeof(int16_t));
  if (lps22hh_fifo_temperature_raw_get(&reg_ctx, data_raw_temperature.u8bit) != LPS22HH_OK)
  {
    return LPS22HH_ERROR;
  }

  *Temp = LPS22HH_FROM_LSB_TO_degC((float)(data_raw_temperature.i16bit));

  return LPS22HH_OK;
}

/**
 * @brief  Get the LPS22HH FIFO threshold
 * @param  Status the status of data ready bit
 * @retval 0 in case of success, an error code otherwise
 */
LPS22HHStatusTypeDef LPS22HH::Get_FIFO_FTh_Status(uint8_t *Status)
{
  if (lps22hh_fifo_wtm_flag_get(&reg_ctx, Status) != LPS22HH_OK)
  {
    return LPS22HH_ERROR;
  }

  return LPS22HH_OK;
}

/**
 * @brief  Get the LPS22HH FIFO full status
 * @param  Status the status of data ready bit
 * @retval 0 in case of success, an error code otherwise
 */
LPS22HHStatusTypeDef LPS22HH::Get_FIFO_Full_Status(uint8_t *Status)
{
  if (lps22hh_fifo_full_flag_get(&reg_ctx, Status) != LPS22HH_OK)
  {
    return LPS22HH_ERROR;
  }

  return LPS22HH_OK;
}

/**
 * @brief  Get the LPS22HH FIFO OVR status
 * @param  Status the status of data ready bit
 * @retval 0 in case of success, an error code otherwise
 */
LPS22HHStatusTypeDef LPS22HH::Get_FIFO_Ovr_Status(uint8_t *Status)
{
  if (lps22hh_fifo_ovr_flag_get(&reg_ctx, Status) != LPS22HH_OK)
  {
    return LPS22HH_ERROR;
  }

  return LPS22HH_OK;
}

/**
 * @brief  Get the LPS22HH FIFO data level
 * @param  Status the status of data ready bit
 * @retval 0 in case of success, an error code otherwise
 */
LPS22HHStatusTypeDef LPS22HH::Get_FIFO_Level(uint8_t *Status)
{
  if (lps22hh_fifo_data_level_get(&reg_ctx, Status) != LPS22HH_OK)
  {
    return LPS22HH_ERROR;
  }

  return LPS22HH_OK;
}

/**
 * @brief  Reset the FIFO interrupt
 * @param  interrupt The FIFO interrupt to be reset; values: 0 = FTH; 1 = FULL; 2 = OVR
 * @retval 0 in case of success, an error code otherwise
 */
LPS22HHStatusTypeDef LPS22HH::Reset_FIFO_Interrupt(uint8_t interrupt)
{
  switch (interrupt)
  {
    case 0:
      if (lps22hh_fifo_threshold_on_int_set(&reg_ctx, PROPERTY_DISABLE) != LPS22HH_OK)
      {
        return LPS22HH_ERROR;
      }
      break;
    case 1:
      if (lps22hh_fifo_full_on_int_set(&reg_ctx, PROPERTY_DISABLE) != LPS22HH_OK)
      {
        return LPS22HH_ERROR;
      }
      break;
    case 2:
      if (lps22hh_fifo_ovr_on_int_set(&reg_ctx, PROPERTY_DISABLE) != LPS22HH_OK)
      {
        return LPS22HH_ERROR;
      }
      break;
    default:
      return LPS22HH_ERROR;
  }

  return LPS22HH_OK;
}

/**
 * @brief  Set the FIFO interrupt
 * @param  interrupt The FIFO interrupt to be reset; values: 0 = FTH; 1 = FULL; 2 = OVR
 * @retval 0 in case of success, an error code otherwise
 */
LPS22HHStatusTypeDef LPS22HH::Set_FIFO_Interrupt(uint8_t interrupt)
{
  switch (interrupt)
  {
    case 0:
      if (lps22hh_fifo_threshold_on_int_set(&reg_ctx, PROPERTY_ENABLE) != LPS22HH_OK)
      {
        return LPS22HH_ERROR;
      }
      break;
    case 1:
      if (lps22hh_fifo_full_on_int_set(&reg_ctx, PROPERTY_ENABLE) != LPS22HH_OK)
      {
        return LPS22HH_ERROR;
      }
      break;
    case 2:
      if (lps22hh_fifo_ovr_on_int_set(&reg_ctx, PROPERTY_ENABLE) != LPS22HH_OK)
      {
        return LPS22HH_ERROR;
      }
      break;
    default:
      return LPS22HH_ERROR;
  }

  return LPS22HH_OK;
}

/**
 * @brief  Set the FIFO mode
 * @param  Mode the FIFO mode to be set
 * @retval 0 in case of success, an error code otherwise
 */
LPS22HHStatusTypeDef LPS22HH::Set_FIFO_Mode(uint8_t Mode)
{
  /* Verify that the passed parameter contains one of the valid values */
  switch ((lps22hh_f_mode_t)Mode)
  {
    case LPS22HH_BYPASS_MODE:
    case LPS22HH_FIFO_MODE:
    case LPS22HH_STREAM_MODE:
    case LPS22HH_STREAM_TO_FIFO_MODE:
    case LPS22HH_BYPASS_TO_STREAM_MODE:
    case LPS22HH_BYPASS_TO_FIFO_MODE:
      break;
    default:
      return LPS22HH_ERROR;
  }

  if (lps22hh_fifo_mode_set(&reg_ctx, (lps22hh_f_mode_t)Mode) != LPS22HH_OK)
  {
    return LPS22HH_ERROR;
  }

  return LPS22HH_OK;
}

/**
 * @brief  Set the LPS22HH FIFO data level
 * @param  Status the status of data ready bit
 * @retval 0 in case of success, an error code otherwise
 */
LPS22HHStatusTypeDef LPS22HH::Set_FIFO_Watermark_Level(uint8_t Watermark)
{
  if (lps22hh_fifo_watermark_set(&reg_ctx, Watermark) != LPS22HH_OK)
  {
    return LPS22HH_ERROR;
  }

  return LPS22HH_OK;
}

/**
 * @brief  Set the LPS22HH stop on watermark function
 * @param  Stop the state of stop on watermark function
 * @retval 0 in case of success, an error code otherwise
 */
LPS22HHStatusTypeDef LPS22HH::Stop_FIFO_On_Watermark(uint8_t Stop)
{
  if (lps22hh_fifo_stop_on_wtm_set(&reg_ctx, Stop) != LPS22HH_OK)
  {
    return LPS22HH_ERROR;
  }

  return LPS22HH_OK;
}

/**
 * @brief  Set the LPS22HH One Shot Mode
 * @retval 0 in case of success, an error code otherwise
 */
LPS22HHStatusTypeDef LPS22HH::Set_One_Shot()
{
  /* Start One Shot Measurement */
  if(lps22hh_data_rate_set(&reg_ctx, LPS22HH_ONE_SHOOT) != LPS22HH_OK)
  {
    return LPS22HH_ERROR;
  }

  return LPS22HH_OK;
}

/**
 * @brief  Get the LPS22HH One Shot Status
 * @param  Status pointer to the one shot status (1 means measurements available, 0 means measurements not available yet)
 * @retval 0 in case of success, an error code otherwise
 */
LPS22HHStatusTypeDef LPS22HH::Get_One_Shot_Status(uint8_t *Status)
{
  uint8_t p_da;
  uint8_t t_da;

  /* Get DataReady for pressure */
  if(lps22hh_press_flag_data_ready_get(&reg_ctx, &p_da) != LPS22HH_OK)
  {
    return LPS22HH_ERROR;
  }

  /* Get DataReady for temperature */
  if(lps22hh_temp_flag_data_ready_get(&reg_ctx, &t_da) != LPS22HH_OK)
  {
    return LPS22HH_ERROR;
  }

  if(p_da && t_da)
  {
    *Status = 1;
  }
  else
  {
    *Status = 0;
  }

  return LPS22HH_OK;
}

int32_t LPS22HH_io_write(void *handle, uint8_t WriteAddr, uint8_t *pBuffer, uint16_t nBytesToWrite)
{
  return ((LPS22HH *)handle)->IO_Write(pBuffer, WriteAddr, nBytesToWrite);
}

int32_t LPS22HH_io_read(void *handle, uint8_t ReadAddr, uint8_t *pBuffer, uint16_t nBytesToRead)
{
  return ((LPS22HH *)handle)->IO_Read(pBuffer, ReadAddr, nBytesToRead);
}

// *********************************** *****************************************



/**
  * @brief  Read generic device register
  *
  * @param  ctx   read / write interface definitions(ptr)
  * @param  reg   register to read
  * @param  data  pointer to buffer that store the data read(ptr)
  * @param  len   number of consecutive register to read
  * @retval       interface status (MANDATORY: return 0 -> no Error)
  *
  */
int32_t lps22hh_read_reg(lps22hh_ctx_t* ctx, uint8_t reg, uint8_t* data, uint16_t len)
{
  int32_t ret;
  ret = ctx->read_reg(ctx->handle, reg, data, len);
  return ret;
}

/**
  * @brief  Write generic device register
  *
  * @param  ctx   read / write interface definitions(ptr)
  * @param  reg   register to write
  * @param  data  pointer to data to write in register reg(ptr)
  * @param  len   number of consecutive register to write
  * @retval       interface status (MANDATORY: return 0 -> no Error)
  *
  */
int32_t lps22hh_write_reg(lps22hh_ctx_t* ctx, uint8_t reg, uint8_t* data, uint16_t len)
{
  int32_t ret;
  ret = ctx->write_reg(ctx->handle, reg, data, len);
  return ret;
}

// ************  ***************  **************  ************  ************
// ************  ***************  **************  ************  ************
// ************  ***************  **************  ************  ************

int32_t lps22hh_data_rate_set(lps22hh_ctx_t *ctx, lps22hh_odr_t val)
{
  lps22hh_ctrl_reg1_t ctrl_reg1;
  lps22hh_ctrl_reg2_t ctrl_reg2;
  int32_t ret;

  ret = lps22hh_read_reg(ctx, LPS22HH_CTRL_REG1, (uint8_t*)&ctrl_reg1, 1);
  if (ret == 0) {
    ret = lps22hh_read_reg(ctx, LPS22HH_CTRL_REG2, (uint8_t*)&ctrl_reg2, 1);
  }
  if (ret == 0) {
    ctrl_reg1.odr = (uint8_t)val & 0x07U;
    ret = lps22hh_write_reg(ctx, LPS22HH_CTRL_REG1, (uint8_t*)&ctrl_reg1, 1);
  }
  if (ret == 0) {
    ctrl_reg2.low_noise_en = ((uint8_t)val & 0x10U) >> 4;
    ctrl_reg2.one_shot = ((uint8_t)val & 0x08U) >> 3;
    ret = lps22hh_write_reg(ctx, LPS22HH_CTRL_REG2, (uint8_t*)&ctrl_reg2, 1);
  }
  return ret;
}

/**
  * @brief  Low-pass bandwidth selection.[set]
  *
  * @param  ctx      read / write interface definitions
  * @param  val      change the values of lpfp_cfg in reg CTRL_REG1
  * @retval          interface status (MANDATORY: return 0 -> no Error)
  *
  */
int32_t lps22hh_lp_bandwidth_set(lps22hh_ctx_t *ctx, lps22hh_lpfp_cfg_t val)
{
  lps22hh_ctrl_reg1_t reg;
  int32_t ret;

  ret = lps22hh_read_reg(ctx, LPS22HH_CTRL_REG1, (uint8_t*) &reg, 1);
  if (ret == 0) {
    reg.lpfp_cfg = (uint8_t)val;
    ret = lps22hh_write_reg(ctx, LPS22HH_CTRL_REG1, (uint8_t*) &reg, 1);
  }
  return ret;
}

/**
  * @brief  Low-pass bandwidth selection.[get]
  *
  * @param  ctx      read / write interface definitions
  * @param  val      Get the values of lpfp_cfg in reg CTRL_REG1
  * @retval          interface status (MANDATORY: return 0 -> no Error)
  *
  */
int32_t lps22hh_lp_bandwidth_get(lps22hh_ctx_t *ctx, lps22hh_lpfp_cfg_t *val)
{
  lps22hh_ctrl_reg1_t reg;
  int32_t ret;

  ret = lps22hh_read_reg(ctx, LPS22HH_CTRL_REG1, (uint8_t*) &reg, 1);
  switch (reg.lpfp_cfg) {
      case LPS22HH_LPF_ODR_DIV_2:
        *val = LPS22HH_LPF_ODR_DIV_2;
        break;
      case LPS22HH_LPF_ODR_DIV_9:
        *val = LPS22HH_LPF_ODR_DIV_9;
        break;
      case LPS22HH_LPF_ODR_DIV_20:
        *val = LPS22HH_LPF_ODR_DIV_20;
        break;
      default:
        *val = LPS22HH_LPF_ODR_DIV_2;
        break;
    }

  return ret;
}

/**
  * @brief  Block Data Update.[set]
  *
  * @param  ctx      read / write interface definitions
  * @param  val      change the values of bdu in reg CTRL_REG1
  * @retval          interface status (MANDATORY: return 0 -> no Error)
  *
  */
int32_t lps22hh_block_data_update_set(lps22hh_ctx_t *ctx, uint8_t val)
{
  lps22hh_ctrl_reg1_t reg;
  int32_t ret;

  ret = lps22hh_read_reg(ctx, LPS22HH_CTRL_REG1, (uint8_t*) &reg, 1);
  if (ret == 0) {
    reg.bdu = val;
    ret = lps22hh_write_reg(ctx, LPS22HH_CTRL_REG1, (uint8_t*) &reg, 1);
  }
  return ret;
}

/**
  * @brief  Register address automatically
  *         incremented during a multiple byte access
  *         with a serial interface.[set]
  *
  * @param  ctx      read / write interface definitions
  * @param  val      change the values of if_add_inc in reg CTRL_REG2
  * @retval          interface status (MANDATORY: return 0 -> no Error)
  *
  */
int32_t lps22hh_auto_increment_set(lps22hh_ctx_t *ctx, uint8_t val)
{
  lps22hh_ctrl_reg2_t reg;
  int32_t ret;

  ret = lps22hh_read_reg(ctx, LPS22HH_CTRL_REG2, (uint8_t*) &reg, 1);
  if (ret == 0) {
    reg.if_add_inc = val;
    ret = lps22hh_write_reg(ctx, LPS22HH_CTRL_REG2, (uint8_t*) &reg, 1);
  }
  return ret;
}

/**
  * @brief  Output data rate selection.[get]
  *
  * @param  ctx      read / write interface definitions
  * @param  val      Get the values of odr in reg CTRL_REG1
  * @retval          interface status (MANDATORY: return 0 -> no Error)
  *
  */
int32_t lps22hh_data_rate_get(lps22hh_ctx_t *ctx, lps22hh_odr_t *val)
{
  lps22hh_ctrl_reg1_t ctrl_reg1;
  lps22hh_ctrl_reg2_t ctrl_reg2;
  int32_t ret;

  ret = lps22hh_read_reg(ctx, LPS22HH_CTRL_REG1, (uint8_t*)&ctrl_reg1, 1);
  if (ret == 0) {
    ret = lps22hh_read_reg(ctx, LPS22HH_CTRL_REG2, (uint8_t*)&ctrl_reg2, 1);
  }
  if (ret == 0) {
    ret = lps22hh_read_reg(ctx, LPS22HH_CTRL_REG2, (uint8_t*)&ctrl_reg2, 1);
    switch (((ctrl_reg2.low_noise_en << 4) + (ctrl_reg2.one_shot << 3) +
            ctrl_reg1.odr )) {
      case LPS22HH_POWER_DOWN:
        *val = LPS22HH_POWER_DOWN;
        break;
      case LPS22HH_ONE_SHOOT:
        *val = LPS22HH_ONE_SHOOT;
        break;
      case LPS22HH_1_Hz:
        *val = LPS22HH_1_Hz;
        break;
      case LPS22HH_10_Hz:
        *val = LPS22HH_10_Hz;
        break;
      case LPS22HH_25_Hz:
        *val = LPS22HH_25_Hz;
        break;
      case LPS22HH_50_Hz:
        *val = LPS22HH_50_Hz;
        break;
      case LPS22HH_75_Hz:
        *val = LPS22HH_75_Hz;
        break;
      case LPS22HH_1_Hz_LOW_NOISE:
        *val = LPS22HH_1_Hz_LOW_NOISE;
        break;
      case LPS22HH_10_Hz_LOW_NOISE:
        *val = LPS22HH_10_Hz_LOW_NOISE;
        break;
      case LPS22HH_25_Hz_LOW_NOISE:
        *val = LPS22HH_25_Hz_LOW_NOISE;
        break;
      case LPS22HH_50_Hz_LOW_NOISE:
        *val = LPS22HH_50_Hz_LOW_NOISE;
        break;
      case LPS22HH_75_Hz_LOW_NOISE:
        *val = LPS22HH_75_Hz_LOW_NOISE;
        break;
      case LPS22HH_100_Hz:
        *val = LPS22HH_100_Hz;
        break;
      case LPS22HH_200_Hz:
        *val = LPS22HH_200_Hz;
        break;
      default:
        *val = LPS22HH_POWER_DOWN;
        break;
    }
  }
  return ret;
}

/**
  * @brief  Pressure output value.[get]
  *
  * @param  ctx      read / write interface definitions
  * @param  buff     buffer that stores data read
  * @retval          interface status (MANDATORY: return 0 -> no Error)
  *
  */
int32_t lps22hh_pressure_raw_get(lps22hh_ctx_t *ctx, uint8_t *buff)
{
  int32_t ret;
  ret =  lps22hh_read_reg(ctx, LPS22HH_PRESSURE_OUT_XL, buff, 3);
  return ret;
}

/**
  * @brief  Temperature output value.[get]
  *
  * @param  ctx      read / write interface definitions
  * @param  buff     buffer that stores data read
  * @retval          interface status (MANDATORY: return 0 -> no Error)
  *
  */
int32_t lps22hh_temperature_raw_get(lps22hh_ctx_t *ctx, uint8_t *buff)
{
  int32_t ret;
  ret =  lps22hh_read_reg(ctx, LPS22HH_TEMP_OUT_L, buff, 2);
  return ret;
}

/**
  * @brief  Pressure new data available.[get]
  *
  * @param  ctx      read / write interface definitions
  * @param  val      change the values of p_da in reg STATUS
  * @retval          interface status (MANDATORY: return 0 -> no Error)
  *
  */
int32_t lps22hh_press_flag_data_ready_get(lps22hh_ctx_t *ctx, uint8_t *val)
{
  lps22hh_status_t reg;
  int32_t ret;

  ret = lps22hh_read_reg(ctx, LPS22HH_STATUS, (uint8_t*) &reg, 1);
  *val = reg.p_da;

  return ret;
}

/**
  * @brief  Temperature data available.[get]
  *
  * @param  ctx      read / write interface definitions
  * @param  val      change the values of t_da in reg STATUS
  * @retval          interface status (MANDATORY: return 0 -> no Error)
  *
  */
int32_t lps22hh_temp_flag_data_ready_get(lps22hh_ctx_t *ctx, uint8_t *val)
{
  lps22hh_status_t reg;
  int32_t ret;

  ret = lps22hh_read_reg(ctx, LPS22HH_STATUS, (uint8_t*) &reg, 1);
  *val = reg.t_da;

  return ret;
}

/**
  * @brief  Pressure output from FIFO value.[get]
  *
  * @param  ctx      read / write interface definitions
  * @param  buff     buffer that stores data read
  * @retval          interface status (MANDATORY: return 0 -> no Error)
  *
  */
int32_t lps22hh_fifo_pressure_raw_get(lps22hh_ctx_t *ctx, uint8_t *buff)
{
  int32_t ret;
  ret =  lps22hh_read_reg(ctx, LPS22HH_FIFO_DATA_OUT_PRESS_XL, buff, 3);
  return ret;
}

/**
  * @brief  Temperature output from FIFO value.[get]
  *
  * @param  ctx      read / write interface definitions
  * @param  buff     buffer that stores data read
  * @retval          interface status (MANDATORY: return 0 -> no Error)
  *
  */
int32_t lps22hh_fifo_temperature_raw_get(lps22hh_ctx_t *ctx, uint8_t *buff)
{
  int32_t ret;
  ret =  lps22hh_read_reg(ctx, LPS22HH_FIFO_DATA_OUT_TEMP_L, buff, 2);
  return ret;
}

/**
  * @brief  FIFO watermark status.[get]
  *
  * @param  ctx      read / write interface definitions
  * @param  val      change the values of fifo_wtm_ia in reg FIFO_STATUS2
  * @retval          interface status (MANDATORY: return 0 -> no Error)
  *
  */
int32_t lps22hh_fifo_wtm_flag_get(lps22hh_ctx_t *ctx, uint8_t *val)
{
  lps22hh_fifo_status2_t reg;
  int32_t ret;

  ret = lps22hh_read_reg(ctx, LPS22HH_FIFO_STATUS2, (uint8_t*) &reg, 1);
  *val = reg.fifo_wtm_ia;

  return ret;
}

/**
  * @brief  Read all the FIFO status flag of the device.[get]
  *
  * @param  ctx      read / write interface definitions
  * @param  val      registers FIFO_STATUS2
  * @retval          interface status (MANDATORY: return 0 -> no Error)
  *
  */
int32_t lps22hh_fifo_src_get(lps22hh_ctx_t *ctx, lps22hh_fifo_status2_t *val)
{
  int32_t ret;
  ret =  lps22hh_read_reg(ctx, LPS22HH_FIFO_STATUS2, (uint8_t*) val, 1);
  return ret;
}

/**
  * @brief  Smart FIFO full status.[get]
  *
  * @param  ctx      read / write interface definitions
  * @param  val      change the values of fifo_full_ia in reg FIFO_STATUS2
  * @retval          interface status (MANDATORY: return 0 -> no Error)
  *
  */
int32_t lps22hh_fifo_full_flag_get(lps22hh_ctx_t *ctx, uint8_t *val)
{
  lps22hh_fifo_status2_t reg;
  int32_t ret;

  ret = lps22hh_read_reg(ctx, LPS22HH_FIFO_STATUS2, (uint8_t*) &reg, 1);
  *val = reg.fifo_full_ia;

  return ret;
}

/**
  * @brief  FIFO overrun status.[get]
  *
  * @param  ctx      read / write interface definitions
  * @param  val      change the values of fifo_ovr_ia in reg FIFO_STATUS2
  * @retval          interface status (MANDATORY: return 0 -> no Error)
  *
  */
int32_t lps22hh_fifo_ovr_flag_get(lps22hh_ctx_t *ctx, uint8_t *val)
{
  lps22hh_fifo_status2_t reg;
  int32_t ret;

  ret = lps22hh_read_reg(ctx, LPS22HH_FIFO_STATUS2, (uint8_t*) &reg, 1);
  *val = reg.fifo_ovr_ia;

  return ret;
}

/**
  * @brief  FIFO stored data level.[get]
  *
  * @param  ctx      read / write interface definitions
  * @param  buff     buffer that stores data read
  * @retval          interface status (MANDATORY: return 0 -> no Error)
  *
  */
int32_t lps22hh_fifo_data_level_get(lps22hh_ctx_t *ctx, uint8_t *buff)
{
  int32_t ret;
  ret =  lps22hh_read_reg(ctx, LPS22HH_FIFO_STATUS1, buff, 1);
  return ret;
}

/**
  * @brief  FIFO watermark status on INT_DRDY pin.[set]
  *
  * @param  lps22hh_ctx_t *ctx: read / write interface definitions
  * @param  uint8_t val: change the values of f_fth in reg CTRL_REG3
  *
  */
int32_t lps22hh_fifo_threshold_on_int_set(lps22hh_ctx_t *ctx, uint8_t val)
{
  lps22hh_reg_t reg;
  int32_t ret;

  ret = lps22hh_read_reg(ctx, LPS22HH_CTRL_REG3, &(reg.byte), 1);
  reg.ctrl_reg3.int_f_wtm = val;
  ret = lps22hh_write_reg(ctx, LPS22HH_CTRL_REG3, &(reg.byte), 1);

  return ret;
}

/**
  * @brief  FIFO full flag on INT_DRDY pin.[set]
  *
  * @param  lps22hh_ctx_t *ctx: read / write interface definitions
  * @param  uint8_t val: change the values of f_fss5 in reg CTRL_REG3
  *
  */
int32_t lps22hh_fifo_full_on_int_set(lps22hh_ctx_t *ctx, uint8_t val)
{
  lps22hh_reg_t reg;
  int32_t ret;

  ret = lps22hh_read_reg(ctx, LPS22HH_CTRL_REG3, &(reg.byte), 1);
  reg.ctrl_reg3.int_f_full = val;
  ret = lps22hh_write_reg(ctx, LPS22HH_CTRL_REG3, &(reg.byte), 1);

  return ret;
}

/**
  * @brief  FIFO overrun interrupt on INT_DRDY pin.[set]
  *
  * @param  lps22hh_ctx_t *ctx: read / write interface definitions
  * @param  uint8_t val: change the values of f_ovr in reg CTRL_REG3
  *
  */
int32_t lps22hh_fifo_ovr_on_int_set(lps22hh_ctx_t *ctx, uint8_t val)
{
  lps22hh_reg_t reg;
  int32_t ret;

  ret = lps22hh_read_reg(ctx, LPS22HH_CTRL_REG3, &(reg.byte), 1);
  reg.ctrl_reg3.int_f_ovr = val;
  ret = lps22hh_write_reg(ctx, LPS22HH_CTRL_REG3, &(reg.byte), 1);

  return ret;
}

/**
  * @brief  Fifo Mode selection.[set]
  *
  * @param  ctx      read / write interface definitions
  * @param  val      change the values of f_mode in reg FIFO_CTRL
  * @retval          interface status (MANDATORY: return 0 -> no Error)
  *
  */
  
int32_t lps22hh_fifo_mode_set(lps22hh_ctx_t *ctx, lps22hh_f_mode_t val)
{
  lps22hh_fifo_ctrl_t reg;
  int32_t ret;

  ret = lps22hh_read_reg(ctx, LPS22HH_FIFO_CTRL, (uint8_t*) &reg, 1);
  if (ret == 0) {
    reg.f_mode = (uint8_t)val;
    ret = lps22hh_write_reg(ctx, LPS22HH_FIFO_CTRL, (uint8_t*) &reg, 1);
  }
  return ret;
}

/**
  * @brief  Fifo Mode selection.[get]
  *
  * @param  ctx      read / write interface definitions
  * @param  val      Get the values of f_mode in reg FIFO_CTRL
  * @retval          interface status (MANDATORY: return 0 -> no Error)
  *
  */
  
int32_t lps22hh_fifo_mode_get(lps22hh_ctx_t *ctx, lps22hh_f_mode_t *val)
{
  lps22hh_fifo_ctrl_t reg;
  int32_t ret;

  ret = lps22hh_read_reg(ctx, LPS22HH_FIFO_CTRL, (uint8_t*) &reg, 1);

  switch (reg.f_mode) {
    case LPS22HH_BYPASS_MODE:
      *val = LPS22HH_BYPASS_MODE;
      break;
    case LPS22HH_FIFO_MODE:
      *val = LPS22HH_FIFO_MODE;
      break;
    case LPS22HH_STREAM_MODE:
      *val = LPS22HH_STREAM_MODE;
      break;
    case LPS22HH_DYNAMIC_STREAM_MODE:
      *val = LPS22HH_DYNAMIC_STREAM_MODE;
      break;
    case LPS22HH_BYPASS_TO_FIFO_MODE:
      *val = LPS22HH_BYPASS_TO_FIFO_MODE;
      break;
    case LPS22HH_BYPASS_TO_STREAM_MODE:
      *val = LPS22HH_BYPASS_TO_STREAM_MODE;
      break;
    case LPS22HH_STREAM_TO_FIFO_MODE:
      *val = LPS22HH_STREAM_TO_FIFO_MODE;
      break;
    default:
      *val = LPS22HH_BYPASS_MODE;
      break;
  }

  return ret;
}

/**
  * @brief  Sensing chain FIFO stop values memorization at
  *         threshold level.[set]
  *
  * @param  ctx      read / write interface definitions
  * @param  val      change the values of stop_on_wtm in reg FIFO_CTRL
  * @retval          interface status (MANDATORY: return 0 -> no Error)
  *
  */
int32_t lps22hh_fifo_stop_on_wtm_set(lps22hh_ctx_t *ctx, uint8_t val)
{
  lps22hh_fifo_ctrl_t reg;
  int32_t ret;

  ret = lps22hh_read_reg(ctx, LPS22HH_FIFO_CTRL, (uint8_t*) &reg, 1);
  if (ret == 0) {
    reg.stop_on_wtm = val;
    ret = lps22hh_write_reg(ctx, LPS22HH_FIFO_CTRL, (uint8_t*) &reg, 1);
  }
  return ret;
}

/**
  * @brief  FIFO watermark level selection.[set]
  *
  * @param  ctx      read / write interface definitions
  * @param  val      change the values of wtm in reg FIFO_WTM
  * @retval          interface status (MANDATORY: return 0 -> no Error)
  *
  */
int32_t lps22hh_fifo_watermark_set(lps22hh_ctx_t *ctx, uint8_t val)
{
  lps22hh_fifo_wtm_t reg;
  int32_t ret;

  ret = lps22hh_read_reg(ctx, LPS22HH_FIFO_WTM, (uint8_t*) &reg, 1);
  if (ret == 0) {
    reg.wtm = val;
    ret = lps22hh_write_reg(ctx, LPS22HH_FIFO_WTM, (uint8_t*) &reg, 1);
  }
  return ret;
}