/* mbed Wiegand Library
 * Copyright (c) 2017-2019, Gregorio Marquez
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */
 
#ifndef MBED_MICIO_H
#define MBED_MICIO_H

#include "mbed.h"

/*
    This library reads Wiegand, saving it into a buffer, calling a callback function after it is full or a timeout is reached.
    Data can be extracted as :
        * Raw -- For manipulation
        * Base 10 Integer 
        * Base 10 String    
        * Base 16 String
    Always call reset after saving Wiegand data.
    It automatically calibrates the timeout during the first reading unless specified.
    Timeout can be set using setTimout, units are in milliseconds (first call stopCalibrating()).
    EventQueue is required for deferring from ISR, it is possible to dispatch the EventQueue from the mainQueue or any other thread. 
*/

class Wiegand {
private:
    Callback<void(Wiegand* wiegandReader)> _callback;

    InterruptIn             _d0;
    InterruptIn             _d1;
    EventQueue*             _eventQueue;
    Timer                   _interPulseGapTimer;
    Timeout                 _timeout;

    bool                    _autoCalibration;
    unsigned char           _bits;
    volatile unsigned char  _bitsRead;
    uint8_t*                _buffer;
    unsigned char           _bufferSize;
    unsigned char           _decDigits;
    bool                    _firstReading;
    unsigned char           _hexDigits;
    unsigned char           _id;
    volatile int*           _interPulseGapBuffer;
    unsigned int            _transmissionTimeout;

    void _data0ISR(void);
    void _data0ISRRise(void);
    void _data1ISR(void);
    void _data1ISRRise(void);
    void _onWiegandISR(void);
    void _onWiegandNonISR(void);
    void _shift_left(volatile unsigned char *ar, int size, int shift);

public:
    Wiegand(PinName d0, PinName d1, EventQueue* eventQueue, unsigned char i=0);
    ~Wiegand();

    void            attach(Callback<void(Wiegand *WiegandObj)> wiegandCallback);
    void            calibrateTimeout(void);
    unsigned char   getDecDigits();
    void            getDecString(volatile char* decString);
    unsigned char   getHexDigits();
    void            getHexString(volatile char* hexString);
    unsigned char   getId();
    uint8_t*        getRaw(void);
    long double     getRawInt(void);
    unsigned int    getTimeout(void);
    void            reset(void);
    void            setTimeout(unsigned int time);
    void            setBits(unsigned char bits);
    void            startCalibrating(void);
    void            stopCalibrating(void);
};

#endif