#include "mbed.h"
#include "string"
#include "C12832.h"

 
 
 


//NNN40 is server or client
#define TEST_TCP_CLIENT         (1) 
 
//debug message
#define MESSAGE_TO_LCD          (1)
#define MESSAGE_TO_UART         (1)


//measure reponse time
#define MEASURE_RESPONSE_TIME   (0)

//UART baud rate
#define BAUD_RATE_CLI           (115200)
#define BAUD_RATE_UART          (115200)

//CLI parameters
#define CLI_COMMAND_MAX_SIZE        (128)
#define CLI_SEND_DATA_MAX_SIZE      (CLI_COMMAND_MAX_SIZE - 26)
#define CLI_RECEIVE_DATA_MAX_SIZE   (128)
#define CLI_RESPONSE_MAX_SIZE       (128)
#define CLI_RETRY_MAX               (3)
#define CLI_RESPONSE_TIMEOUT        (2725350) //1 sec


#define DEFAULT_AP_NAME         "Airport123" 
#define DEFAULT_AP_PASSWORD     "12345678"
#define DEFAULT_HOST_IP         "10.0.1.3"
#define DEFAULT_HOST_PORT       (5222)
#define DEFAULT_CLI_TIMEOUT_SEC  (3) //3 sec


//version string
#define VERSION_STRING          ("NNN40_CLI_HOST_WIFI : 1.0.3")


enum  WORKINGSTAGE {INITWIFI, TCP_CONNECT_TO_HOST,  TCP_LISTEN_FROM_CLIENT, TCP_SEND, TCP_RECEIVE};


C12832 lcd(p5, p7, p6, p8, p11);
DigitalOut myled1(LED1);
DigitalOut myled2(LED2);
DigitalOut myled3(LED3);
DigitalOut myled4(LED4);


Serial pc(USBTX, USBRX);//debug port , 115200
Serial cli(p9, p10);   //CLI port, 115200

//CLI response buffer
static char RESULT[CLI_RESPONSE_MAX_SIZE];  




void message(const char * const msg, bool isClearLcd = true, int lcd_X = 0, int lcd_Y = 0)
{
    #if(MESSAGE_TO_UART)
    puts(msg);
    #endif
    
    
    #if(MESSAGE_TO_LCD)
    if(isClearLcd)lcd.cls();
    lcd.locate(lcd_X,lcd_Y);
    lcd.puts(msg);
    #endif
}

bool CLI_wait4Response(unsigned int timeout_sec)
{   
    
    bool rc = false;
    int i = 0;
    int c = 0;
    const unsigned tosec = CLI_RESPONSE_TIMEOUT * timeout_sec;
    unsigned int to = tosec;
    
    #if (MEASURE_RESPONSE_TIME)
    int j = 0;
    #endif
    
    //clear buffer
    memset(RESULT,0,sizeof(RESULT));      
    
    #if (MEASURE_RESPONSE_TIME)
    while(1)
    #else
    while(--to) 
    #endif   
    
    {   
        #if (MEASURE_RESPONSE_TIME)
        ++j;
        #endif
        
        if(cli.readable() && EOF != (c = cli.getc()))
        {           
            if( c!=0x0D && c!= 0x0A)
            {
                if(i < sizeof(RESULT))
                {
                    RESULT[i++] = (char)c;
                    to = tosec;
                }
            }
            else if( 0 != i) //get one line response
            {
                #if (MEASURE_RESPONSE_TIME)
                printf("MEASURE_RESPONSE_TIME : %d\n",j);
                #endif               
            
               
                if(0 == strncmp(RESULT,"OK",2) || 0 == strncmp(RESULT,"ERROR",5))
                {
                    break;
                }
                else  //if not OK or ERROR, keep waiting 
                {
                    message(RESULT);
                    memset(RESULT,0,sizeof(RESULT)); 
                    i = 0;   
                }          
                
            } 
        } 
    }
    
    if( 0 == to )
    {
        memcpy(RESULT,"ERROR;CLI Timeout",17);
    }
    else if (RESULT[0] == 'O' && RESULT[1] == 'K')
    {
        rc = true;
    }
    
    return rc;
}

bool CLI_command(const char * cmd, unsigned int timeout_sec)
{    
    bool rc = false;   
    
    message(cmd);  
      
    //clear rx buffer at first
    while(cli.readable())
    {
        cli.getc();        
    }
    
    //send CLI command 
    cli.puts(cmd);
    
    //wait for reponse
    rc = CLI_wait4Response(timeout_sec);
    
    message(RESULT,false,0,11);
    
    return rc;   
}

bool CLI_CMD(const char * cmd, unsigned int timeout_sec = DEFAULT_CLI_TIMEOUT_SEC, bool autoRetry = true)
{
    bool rc = false;
    
    if(autoRetry)
    {
        for(int i=0; i<=CLI_RETRY_MAX; ++i)
        {
        
            if(CLI_command(cmd, timeout_sec))
            {
                rc = true;
                break;
            }        
             
            wait(2); //wait for a while     
        } 
    }
    else
    {
        rc = CLI_command(cmd, timeout_sec);
    }
    
    return rc;  
}

bool initWifi(const char * apName = DEFAULT_AP_NAME, const char * apPassword = DEFAULT_AP_PASSWORD)
{
    //reset module
    if (!CLI_CMD("cynb reset\r")) return false;
    
    //wait for reset completely
    wait(2);
    
    //get infomation of module
    if(!CLI_CMD("cynb info\r")) return false;
    
    //switch to wifi module
    if(!CLI_CMD("cynw device_switch 1\r")) return false;
    
    //setup AP anme and password
    char cmd[CLI_COMMAND_MAX_SIZE];
    sprintf(cmd, "cynw device_network %s %s 0\r", apName, apPassword);
    if(!CLI_CMD(cmd))return false;
    
    //init ethernet
    if(!CLI_CMD("cynw ethernet_init\r",10))return false;
    
    //connecting...,
    if (!CLI_CMD("cynw ethernet_connect 40000\r",45))return false;
            
    //get mac address
    if(!CLI_CMD("cynw ethernet_mac\r"))return false;
    
    
    //get ip address
    if(!CLI_CMD("cynw ethernet_ip\r"))return false;
    
    return true;
 }
bool isTCPConnected()
{
    if(!CLI_CMD("cynw tcp_connection_is_connect\r")) return false;
    
    if(strncmp(RESULT, "OK;TRUE", 7) == 0) return true;
    
    return false;
}
bool TCPConnectToHost(const char * ip = DEFAULT_HOST_IP, unsigned int port = DEFAULT_HOST_PORT)
{   
    char cmd[CLI_COMMAND_MAX_SIZE];
    sprintf(cmd, "cynw tcp_connection_connect %s %d\r", ip, port);
   
    //timeout=20sec
    if(!CLI_CMD(cmd,20))return false; 
      
    if(!CLI_CMD("cynw tcp_connection_is_connect\r")) return false;
    
    return isTCPConnected();
}
bool TCPListenFromClient( unsigned int port = DEFAULT_HOST_PORT)
{
    char cmd[CLI_COMMAND_MAX_SIZE];
    
    //binding
    sprintf(cmd, "cynw tcp_server_bind %d\r",port);
    if(!CLI_CMD(cmd))return false;
    
    //listen
    if(!CLI_CMD("cynw tcp_server_listen\r")) return false;
    
    //accept
    if(!CLI_CMD("cynw tcp_server_accept\r",10)) return false;
    
    return true;    
}
bool send(const char* data)
{
    char cmd[CLI_COMMAND_MAX_SIZE];

    sprintf(cmd, "cynw tcp_connection_send %s\r",data);
    return CLI_CMD(cmd);
}
bool receive(char * buf, int len, int timeout_sec)
{
    if(len > 128) len = 128;
    
    char cmd[CLI_COMMAND_MAX_SIZE];
   
    if(-1 == timeout_sec)//blocking
    {
         sprintf(cmd, "%s" , "cynw tcp_connection_blocking 1\r");
    }
    else //non-blocking
    {
         sprintf(cmd, "cynw tcp_connection_blocking 0 %d\r",timeout_sec * 1000);
    }
    
    if(!CLI_CMD(cmd)) return false;
    
    
    sprintf(cmd, "cynw tcp_connection_receive %d\r" , len);
    if(!CLI_CMD(cmd,timeout_sec + 2)) return false; 
    
    return true;
    
}
int main()
{    
    WORKINGSTAGE ws = INITWIFI;
    //set baud rate
    pc.baud(BAUD_RATE_UART);
    cli.baud(BAUD_RATE_CLI);
    
    //show version string   
    message(VERSION_STRING);
    
    //NNN40 as client/server led=0/led = 1
    myled1 = TEST_TCP_CLIENT;
       
    
    #if(MESSAGE_TO_LCD)
    //give some display time for lcd 
    wait(1);   
    #endif

    while(1)
    {
        switch(ws)
        {
            case INITWIFI:
            {
                if(initWifi())
                {
                    //
                    #if(TEST_TCP_CLIENT == 1) //NNN40 as client
                        ws = TCP_CONNECT_TO_HOST;
                    #else //NNN40 as server
                        ws = TCP_LISTEN_FROM_CLIENT;
                    #endif
                } 
                break; 
            }
            case TCP_CONNECT_TO_HOST:
            {
                if(TCPConnectToHost())
                {
                   ws = TCP_SEND;
                   //ws = TCP_RECEIVE;
                } 
                else
                {
                    //check if there is IP address 
                    if(!CLI_CMD("cynw ethernet_ip\r"))
                    {
                        //if no IP , init wifi again
                        ws = INITWIFI;
                    }
                    wait(1);
                }
                break; 
            }
            case TCP_LISTEN_FROM_CLIENT:
            {
                if(TCPListenFromClient())
                {
                   ws = TCP_SEND;
                   //ws = TCP_RECEIVE;
                } 
                else
                {
                    //check if there is IP address 
                    if(!CLI_CMD("cynw ethernet_ip\r"))
                    {
                        //if no IP , init wifi again
                        ws = INITWIFI;
                    }
                    wait(1);
                }
                break; 
            }
            case TCP_SEND:
            {
                static int tx = 0;
                char buf[CLI_SEND_DATA_MAX_SIZE];
                
                sprintf(buf, "TEST-%d", tx);
                if(send(buf))
                {
                    ++tx;
                    if(tx % 10 == 0) ws = TCP_RECEIVE; //change to receive
                } 
                else if(!isTCPConnected())                    
                { 
                    #if(TEST_TCP_CLIENT == 1) //NNN40 as client
                        ws = TCP_CONNECT_TO_HOST;
                    #else //NNN40 as server
                        ws = TCP_LISTEN_FROM_CLIENT;
                    #endif                   
                }
                wait(1);
                break; 
            }
            case TCP_RECEIVE:
            {
                char buf[CLI_RECEIVE_DATA_MAX_SIZE];                 
                if(!receive(buf, 5, 5) && !isTCPConnected())
                {
                    #if(TEST_TCP_CLIENT == 1) //NNN40 as client
                        ws = TCP_CONNECT_TO_HOST;
                    #else //NNN40 as server
                        ws = TCP_LISTEN_FROM_CLIENT;
                    #endif                   
                }
                else
                {
                    ws = TCP_SEND; //change to send
                }
                break;
            }
            default:break;
            
        }//end switch
    }//end wile  
}//end main