//Sensor DS1820 y cooler con tres cables

#include "mbed.h"
//Incluyo librería del sensor
#include "DS1820.h"

//Creo los estados de las máquinas de estados
enum{LC,LC2,LA,LA2,SENSOR,PRESET,MOTOR};
//  |--------------|------------|
//    SELECCIONADOR  VELOCIDAD

//Defino como Interrupt In la entrada de los pulsos de los RPM. Entra a la función asociada a esta entrada cada vez que detecta un flanco ascendente
InterruptIn Rpm(D2);
//Defino como PWM Out la salida que va al motor del cooler. Va a controlar la velocidad del mismo 
PwmOut Motor(A2);
//Defino como Analog In la entrada que va a leer el valor analógico entregado por el preset/potenciómetro
AnalogIn Preset(A1);
//Pulsador que cambia entre Lazo Cerrado y Lazo Abierto
DigitalIn Pulsador(D5);

//Creo Ticker
Ticker mideTiempo;

//Función asociada al Ticker
void cuentaTiempo();    //Cuenta el tiempo para sacar las Preset y saca las Preset

//Función que se ejecuta con el Interrupt In
void cuentaRpm();   //Cada vez que entra por un pulso le suma 1 a una variable. Esta se reinicia cuando ya pasó 1 segundo y se sacaron las Preset

//Funciones que son las máquinas de estados
void seleccionModo();
void velocidadMotor();

//Variables que guardan los estados de las máquinas de estados
int estadoSeleccion;
int estadoVelocidad;

//Variable que guarda el tiempo para contar las Preset
int tiempo=0;

//Variable que guarda las veces que se entró a la función del Interrupt In por los pulsos que indican los Preset
int veces=0;

//Variable que guarda el valor de las Preset
int valorRpm=0;

//Constantes que indican ambos modos
const int lazoCerrado=1;
const int lazoAbierto=2;

//Variable que guarda el modo en el que se está
int modo=lazoCerrado;

//Variable que guarda el valor que indica la velocidad que tiene que tener el motor. Es un float que varía entre 0f (0V) y 1f(3,3V), teniendo en cuenta los valores intermedios
float valorVelocidad=0;
//Variable que guarda el valor que entra del preset, de 0f (0V) a 1f (3,3V)
float valorPreset;

DigitalOut  led(LED1);      //Led que indica si se están recibiendo los datos de temperatura
DS1820      ds1820(A0);     //Pin de datos del sensor de temperatura. Hay que ponerle una R de pull up de 4K7 a 3,3V
float       temp = 0;       //Variable que guarda la temperatura
int         result = 0;     //Variable para la conversión de los datos de temperatura

int main()
{
    //Vinculo el ticker con su función
    mideTiempo.attach(&cuentaTiempo,0.01);  //Hace cuentaTiempo cada 10 milisegundos
    //Vinculo el Interrupt In con su función
    Rpm.rise(&cuentaRpm);   //Entra en cuentaRpm cada vez que hay un flanco ascendente
    //Si hay sensor, empieza la conversión
    if (ds1820.begin()) {
        while (1) {
            //Hago que el valor de velocidad que tiene que ser según el programa se muestre a la salida
            Motor=valorVelocidad;
            //Hago que el valor leído del preset se guarde en la variable correspondiente
            valorPreset=Preset.read();
            //Llamo a las funciones de máquinas de estados
            seleccionModo();
            velocidadMotor();
            //Acá está toda la parte de la lectura de la temperatura, con funciones propias de la librería incluída
            ds1820.startConversion();   // start temperature conversion from analog to digital
            wait(1);                  // let DS1820 complete the temperature conversion
            result = ds1820.read(temp); // read temperature from DS1820 and perform cyclic redundancy check (CRC)
            switch (result) {
                case 0:                 // no errors -> 'temp' contains the value of measured temperature
                    printf("temp = %3.1f%cC\r\n", temp, 176);
                    break;

                case 1:                 // no sensor present -> 'temp' was not updated
                    printf("no sensor present\n\r");
                    break;

                case 2:                 // CRC error -> 'temp' was not updated
                    printf("CRC error\r\n");
            }

            led = !led;
        }
    }
    else
        printf("No DS1820 sensor found!\r\n");
}

void cuentaRpm()
{
    //Cada vez que se detecta un pulso de los Rpm, le suma 1 a veces
    veces++;
}

void cuentaTiempo()
{
    //Apenas empiezan a llegar pulsos empieza a contar el tiempo. Entra acá cada 10 milisegundos, o sea que tiempo va a ser 100 cuando halla pasado 1 segundo
    if(veces!=0)
    {
        tiempo++;   
    }
    else
    {
        valorRpm=0;    
    }
    //Cuando halla pasado 1 segundo, se ve la cantidad de veces que se entró a la función anterior y se la multiplica por 60 para sacar cuántas serían por minuto
    //Esas son las revoluciones por minuto (RPM)
    //Se reinicia veces y tiempo y se vuelve a realizar el proceso
    if(tiempo/100==1)
    {
        valorRpm=veces*60;
        veces=0;
        tiempo=0;
    }
}

void seleccionModo(){
    switch(estadoSeleccion)
    {
        default:
        case LC:
        //Empieza en modo de lazo cerrado
        modo=lazoCerrado;
        //Si se presiona el pulsador, cambia al estado LA, sino se queda en este estado
        if(Pulsador==0)
        {
            estadoSeleccion=LC;
        }
        else
        {
            estadoSeleccion=LA;
        }
        break;
        
        case LA:
        //Cambia al otro modo, el de lazo abierto
        modo=lazoAbierto;
        //Si se mantiene presionado el pulsador, continua en este estado
        //Si se suelta el pulsador, cambia al estado LA2
        if(Pulsador==1)
        {
            estadoSeleccion=LA;
        }
        else
        {
            estadoSeleccion=LA2;
        }
        break;
        
        case LA2:
        //Se mantiene en el modo de lazo abierto
        modo=lazoAbierto;
        //Si no se presiona el pulsador, se mantiene en este estado
        //Si se presiona el pulsador, cambia al estado LC2
        if(Pulsador==0)
        {
            estadoSeleccion=LA2;
        }
        else
        {
            estadoSeleccion=LC2;
        }
        break;
        
        case LC2:
        //Se cambia al modo de lazo cerrado
        modo=lazoCerrado;
        //Si se mantiene presionado el pulsador, se mantiene en este estado
        //Si se suelta el pulsador, vuelve al estado LC
        if(Pulsador==1)
        {
            estadoSeleccion=LC2;   
        }
        else
        {
            estadoSeleccion=LC;   
        }
        break;  
    }
}

void velocidadMotor(){
    //Printf que indican el modo y el valor de las variables
    if(modo==lazoCerrado)
    {
        printf("Lazo cerrado \n");   
    }
    else if(modo==lazoAbierto)
    {
        printf("Lazo abierto \n");      
    }
    printf("Velocidad Motor (0-1): %1.2f \n",valorVelocidad);
    printf("percentage: %3.3f%%\n", valorPreset*100.0f);
    printf("RPM: %d \n\n",valorRpm);
    switch(estadoVelocidad)
    {
        default:
        case SENSOR:
        //Si el modo es lazo cerrado, se cambia al estado de velocidad del motor (MOTOR). Sino, cambia al estado PRESET
        if(modo==lazoCerrado)
        {
            estadoVelocidad=MOTOR;
        }
        else
        {
            estadoVelocidad=PRESET;
        }
        break;
        
        case PRESET:
        //Si el modo es lazo abierto, se cambia al estado de velocidad del motor (MOTOR). Sino, cambia al estado SENSOR
        if(modo==lazoAbierto)
        {  
            estadoVelocidad=MOTOR;
        }
        else
        {
            estadoVelocidad=SENSOR;
        }
        break;
        
        case MOTOR:
        //Si el modo es lazo cerrado, la velocidad del motor es en base a la temperatura. Vuelve al estado inicial
        if(modo==lazoCerrado)
        {
            valorVelocidad=temp/float(100);
            estadoVelocidad=SENSOR;
        }
        //Si el modo es lazo abierto, la velocidad del motor es en base al preset. Vuelve al estado de PRESET
        else if(modo==lazoAbierto)
        {
            valorVelocidad=valorPreset; 
            estadoVelocidad=PRESET;
        }
        break;
        
    }
}