/**
 * @author Gonçalo Costa
 * 
 * @section LICENSE
 *
 * Copyright (c) 2010
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 *
 * @section DESCRIPTION
 *
 * Pololu Micro Dual Serial Motor Controller.
 *
 * Datasheet:
 *
 * http://www.pololu.com/file/0J36/smc02b_guide.pdf
 *
 */
 
#ifndef SMC02B_H
#define SMC02B_H
/**
 * Includes
 */
#include "mbed.h"

/**
 * Defines
 */
#define SMC02B_START_BYTE       0x80
#define SMC02B_CONFIG_BYTE      0x02
#define SMC02B_DEVICE_ID        0x00
#define SMC02B_BYTE3            0x00
#define SMC02B_BYTE4            0x00
#define SMC02B_MOTOR1_DIR       0x00
#define SMC02B_MOTOR1_SPEED     0x00
#define SMC02B_MOTOR1_NUMBER    0x00
#define SMC02B_MOTOR2_DIR       0x00
#define SMC02B_MOTOR2_SPEED     0x00
#define SMC02B_MOTOR2_NUMBER    0x00

/**
*
* Pololu Micro Dual Serial Motor Controller.
*
*/

class SMC02B {

public:

    /**
     * Constructor.
     *
     * @param tx mbed pin to use for TX line of Serial.
     * @param rx mbed pin to use for RX line of Serial.
     * @param rst mbed pin to use for RESET.
     */
    SMC02B(PinName tx, PinName rx, PinName rst);
    
    /**
     * Reset SMC02B.
     *
     */
    void SMC02B_RST(void);

    /**
     * Set Motor.
     *
     * @param motor - Motor to be controlled.
     * @param direction - Motor Direction.
     *                    0 - Reverse
     *                    1 - Forward
     * @param speed - Motor Speed.
     *                    0 - 127 ( 0x7F )
     */
    void SMC02B_MOTOR_SET( char motor, char direction, char speed );

    /**
     * Set Motor to Brake.
     *
     * @param motor - Motor to be controlled.
     */
    void SMC02B_MOTOR_BRAKE( char motor );

    /**
     * Set Motor to idle.
     *
     * @param motor - Motor to be controlled.
     */
    void SMC02B_MOTOR_COAST( char motor );

    /**
     * Configure SMC02B.
     *
     * @param mode - Set mode 1 or 2 motor mode.
     *              0 - 2 motors
     *              1 - 1 motor
     * @param motor - Motor number.
     *
     *  Allways reset SMC02B after a config
     */
    void SMC02B_CONFIG( char mode, char motor );

private:
    DigitalOut _rst;
    Serial* _SMC02B;

};

#endif
