/***************************************************************************//**
 * @file BufferedDisplay.cpp
 * @brief Buffered version of GraphicsDisplay
 *******************************************************************************
 * @section License
 * <b>(C) Copyright 2015 Silicon Labs, http://www.silabs.com</b>
 *******************************************************************************
 *
 * Permission is granted to anyone to use this software for any purpose,
 * including commercial applications, and to alter it and redistribute it
 * freely, subject to the following restrictions:
 *
 * 1. The origin of this software must not be misrepresented; you must not
 *    claim that you wrote the original software.
 * 2. Altered source versions must be plainly marked as such, and must not be
 *    misrepresented as being the original software.
 * 3. This notice may not be removed or altered from any source distribution.
 *
 * DISCLAIMER OF WARRANTY/LIMITATION OF REMEDIES: Silicon Labs has no
 * obligation to support this Software. Silicon Labs is providing the
 * Software "AS IS", with no express or implied warranties of any kind,
 * including, but not limited to, any implied warranties of merchantability
 * or fitness for any particular purpose or warranties against infringement
 * of any proprietary rights of a third party.
 *
 * Silicon Labs will not be liable for any consequential, incidental, or
 * special damages, or any other relief, or for any claim by any third party,
 * arising from your use of this Software.
 *
 ******************************************************************************/

#include "BufferedDisplay.h"

#define SWAP8(a) ((((a) & 0x80) >> 7) | (((a) & 0x40) >> 5) | (((a) & 0x20) >> 3) | (((a) & 0x10) >> 1) | (((a) & 0x08) << 1) | (((a) & 0x04) << 3) | (((a) & 0x02) << 5) | (((a) & 0x01) << 7))

namespace silabs {

	BufferedDisplay::BufferedDisplay(const char *name) : GraphicsDisplay(name) {
		memset((uint8_t*)_pixelBuffer, White, sizeof(_pixelBuffer));	// init full frame buffer
		memset((uint8_t*)_dirtyRows, 0xFF, sizeof(_dirtyRows)); 		// init dirty status
	}

	/**
	 * Override of GraphicsDisplay's pixel()
	 */

	void BufferedDisplay::pixel(int x, int y, int colour) {
	    uint8_t swapx = 7 - ((unsigned int)x & 0x07);
	    x = ((unsigned int)x & 0xFFFFFFF8) | swapx;

	    // determine change
	    bool change = ((_pixelBuffer[((y * DISPLAY_WIDTH) + x) / DISPLAY_BUFFER_TYPE_SIZE] & (1 << (x % DISPLAY_BUFFER_TYPE_SIZE))) != ((colour & 0x01) << (x % DISPLAY_BUFFER_TYPE_SIZE)));
		if(change) {
            // xor operation
            _pixelBuffer[((y * DISPLAY_WIDTH) + x) / DISPLAY_BUFFER_TYPE_SIZE] ^= (1 << (x % DISPLAY_BUFFER_TYPE_SIZE));

            // notify dirty status of this line
            _dirtyRows[y / DISPLAY_BUFFER_TYPE_SIZE] |= (1 << (y % DISPLAY_BUFFER_TYPE_SIZE));
		}
	}

	int BufferedDisplay::width() {
		return DISPLAY_WIDTH;
	}
	int BufferedDisplay::height() {
		return DISPLAY_HEIGHT;
	}

	/**
	 * Function to move bitmap into frame buffer
	 * arguments:
	 * 	* bitmap: pointer to uint8 array containing horizontal pixel data
	 * 	* bmpWidth: width of the bitmap in pixels (must be byte multiple)
	 * 	* bmpHeight: height of the bitmap in pixels (must be byte multiple)
	 * 	* startX: starting position to apply bitmap in horizontal direction (0 = leftmost) (must be byte multiple)
	 * 	* startY: starting position to apply bitmap in vertical direction (0 = topmost) (must be byte multiple)
	 */
	void BufferedDisplay::showBMP(const uint8_t* bitmap, const uint32_t bmpWidth, const uint32_t bmpHeight, const uint32_t startX, const uint32_t startY) {
		uint32_t iterBMP = 0, iterFB = startY;

		/* Apply constraints */
		if((bmpWidth & 0x07) != 0) return;
		if((bmpHeight & 0x07) != 0) return;
		if((startX & 0x07) != 0) return;
		if(startX > DISPLAY_WIDTH) return;
		if((startY & 0x07) != 0) return;

		/* Copy over bytes to the framebuffer */
		for(; iterFB < DISPLAY_HEIGHT; iterFB++) {
			memcpy( (void*) &(_pixelBuffer[((iterFB * DISPLAY_WIDTH) + startX) / DISPLAY_BUFFER_TYPE_SIZE]),
					(const void*) &(bitmap[iterBMP * (bmpWidth / 8)]),
					(DISPLAY_WIDTH - startX) / 8);

			_dirtyRows[iterFB / DISPLAY_BUFFER_TYPE_SIZE] |= (1 << (iterFB % DISPLAY_BUFFER_TYPE_SIZE));
			iterBMP++;
		}

		return;
	}
}
