/**
 * @file HMC5843.h
 * @author Jose R. Padron
 * @author Used HMC5843 library  developed by Aaron Berk as template
 * @section LICENSE
 *
 * Copyright (c) 2010 ARM Limited
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 *
 * @section DESCRIPTION
 *
 * Honeywell HMC5843 digital compass.
 *
 * Datasheet:
 *
 * http://www.ssec.honeywell.com/magnetic/datasheets/HMC5843.pdf
 */

#ifndef HMC5843_H
#define HMC5843_H

#include "mbed.h"


#define HMC5843_I2C_ADDRESS 0x1E //7-bit address. 0x3C write, 0x3D read.
#define HMC5843_I2C_WRITE   0x3C 
#define HMC5843_I2C_READ    0x3D 

//Values Config A
#define HMC5843_0_5HZ_NORMAL         0x00
#define HMC5843_0_5HZ_POSITIVE       0x01
#define HMC5843_0_5HZ_NEGATIVE       0x02

#define HMC5843_1HZ_NORMAL           0x04
#define HMC5843_1HZ_POSITIVE         0x05
#define HMC5843_1HZ_NEGATIVE         0x06

#define HMC5843_2HZ_NORMAL           0x08
#define HMC5843_2HZ_POSITIVE         0x09
#define HMC5843_2HZ_NEGATIVE         0x0A

#define HMC5843_5HZ_NORMAL           0x0C
#define HMC5843_5HZ_POSITIVE         0x0D
#define HMC5843_5HZ_NEGATIVE         0x0E

#define HMC5843_10HZ_NORMAL           0x10
#define HMC5843_10HZ_POSITIVE         0x11
#define HMC5843_10HZ_NEGATIVE         0x12

#define HMC5843_20HZ_NORMAL           0x14
#define HMC5843_20HZ_POSITIVE         0x15
#define HMC5843_20HZ_NEGATIVE         0x16

#define HMC5843_50HZ_NORMAL           0x18
#define HMC5843_50HZ_POSITIVE         0x19
#define HMC5843_50HZ_NEGATIVE         0x1A

//Values Config B
#define HMC5843_0_7GA         0x00
#define HMC5843_1_0GA         0x20
#define HMC5843_1_5GA         0x40
#define HMC5843_2_0GA         0x60
#define HMC5843_3_2GA         0x80
#define HMC5843_3_8GA         0xA0
#define HMC5843_4_5GA         0xC0
#define HMC5843_6_5GA         0xE0

//Values MODE
#define HMC5843_CONTINUOUS   0x00
#define HMC5843_SINGLE         0x01
#define HMC5843_IDLE         0x02
#define HMC5843_SLEEP         0x03



#define HMC5843_CONFIG_A     0x00
#define HMC5843_CONFIG_B     0x01
#define HMC5843_MODE         0x02
#define HMC5843_X_MSB        0x03
#define HMC5843_X_LSB        0x04
#define HMC5843_Y_MSB        0x05
#define HMC5843_Y_LSB        0x06
#define HMC5843_Z_MSB        0x07
#define HMC5843_Z_LSB        0x08
#define HMC5843_STATUS       0x09
#define HMC5843_IDENT_A      0x0A
#define HMC5843_IDENT_B      0x0B
#define HMC5843_IDENT_C      0x0C



/**
 * Honeywell HMC5843 digital compass.
 */
class HMC5843 {

public:

    /**
     * The I2C address that can be passed directly to i2c object (it's already shifted 1 bit left).
     *
     * You don't need to manually set or clear the LSB when calling I2C::read() or I2C::write(),
     * the library takes care of it.  We just always clear the LSB.
     */
    static const int I2C_ADDRESS = HMC5843_I2C_WRITE;

    /**
     * Constructor.
     *
     * @param sda mbed pin to use for SDA line of I2C interface.
     * @param scl mbed pin to use for SCL line of I2C interface.
     */
    HMC5843(PinName sda, PinName scl);
    
    /**
     * Constructor that accepts external i2c interface object.
     *
     * @param i2c The I2C interface object to use.
     */
    HMC5843(I2C &i2c) : i2c_(i2c), myI2c(NULL){}
    
    /**
     * Destructor that frees self-allocated I2C object.
     */
    ~HMC5843(){
        delete myI2c;
    }

        
     /**
     * Enter into sleep mode.
     *
     */
    void setSleepMode();
    
       
     /**
     * Set Device in Default Mode.
     * HMC5843_CONTINUOUS, HMC5843_10HZ_NORMAL HMC5843_1_0GA
     */
    void setDefault();
    
       
    /**
     * Read the memory location on the device which contains the address.
     *
     * @param Pointer to a buffer to hold the address value
     * Expected     H, 4 and 3.
     */
    void getAddress(char * address);


    
    /**
     * Set the operation mode.
     *
     * @param mode 0x00 -> Continuous
     *             0x01 -> Single
     *             0x02 -> Idle
     * @param ConfigA values
    * @param ConfigB values
     */
    void setOpMode(int mode, int ConfigA, int ConfigB);
    
     /**
     * Write to  on the device.
     *
     * @param address Address to write to.
     * @param data Data to write.
     */
    
    void write(int address, int data);

     /**
     * Get the output of all three axes.
     *
     * @param Pointer to a buffer to hold the magnetics value for the
     *        x-axis, y-axis and z-axis [in that order].
     */
    void readData(int* readings);
    
    /**
     * Get the output of X axis.
     *
     * @return x-axis magnetic value
     */
    int getMx(){ return getWord(HMC5843_X_MSB); }
    
    /**
     * Get the output of Y axis.
     *
     * @return y-axis magnetic value
     */
    int getMy(){ return getWord(HMC5843_Y_MSB); }
    
    /**
     * Get the output of Z axis.
     *
     * @return z-axis magnetic value
     */
    int getMz(){ return getWord(HMC5843_Z_MSB); }
   
    
    /**
     * Get the current operation mode.
     *
     * @return Status register values
     */
    int getStatus(void);


protected:

    /**
     * Reads a word (2 bytes) from the sensor via I2C bus.
     *
     * The queried value is assumed big-endian, 2's complement value.
     *
     * This protected function is added because we shouldn't write getMx(), getMy() and getMz()
     * independently, but collect common codes.
     *
     * @param regi Register address to be read.
     */
    int getWord(int regi);
  
private:

    /**
     * Internal I2C object for communicating with the device.
     *
     * Changed from a pointer (new allocated object) from embedded object.
     * The I2C class object is not meant to be allocated in the heap, but
     * in the global storage.
     * There's no point dynamically allocating an I2C object in this class,
     * at least you should write destructor to release it!
     */
    I2C &i2c_;
    
    I2C *myI2c;

    /**
     * Converts big-endian 2's complement byte pair to native byte order of
     * the CPU and then sign extend it to the CPU's register size.
     *
     * Implemented here to make the compiler inline expand it.
     */
    int swapExtend(const char rx[2]){
        // Readings are expressed in 16bit 2's complement, so we must first
        // concatenate two bytes to make a word and sign extend it to obtain
        // correct negative values.
        // ARMCC compiles char as unsigned, which means no sign extension is
        // performed during bitwise operations to chars. But we should make sure
        // that lower byte won't extend its sign past upper byte for other
        // compilers if we want to keep it portable.
        return int16_t(((unsigned char)rx[0] << 8) | (unsigned char)rx[1]);
    }

};

#endif /* HMC5843_H */
