/* Adafruit I2C Temp Sensor MCP9808 -> Multitech mDot -> The Things Network (TTN)
 *
 * This program connects the Adafruit temp sensor board MCP9808 through I2C
 * and sends the temperature output to the The Things Network.
 *
 * Based on:
 * - mDot TTN Connect Example: https://developer.mbed.org/users/ropu/code/mDot_LoRa_example_TTN_connect/rev/609e7bb06486
 * - Adafruit MCP9808 Library: https://github.com/adafruit/Adafruit_MCP9808_Library
 * - mbed I2C Example: https://developer.mbed.org/handbook/I2C
 * - TTN Backend: https://www.thethingsnetwork.org/wiki/Backend/Overview
 *
 * Requiremens:
 * - Multitech UDK board and mDot: https://developer.mbed.org/platforms/MTS-mDot-F411/
 * - Adafruit MCP9809: https://learn.adafruit.com/adafruit-mcp9808-precision-i2c-temperature-sensor-guide/overview
 * - Serial to USB like this or similar: https://www.amazon.com/Sabrent-Serial-RS-232-Converter-CB-DB9P/dp/B00IDSM6BW/ref=sr_1_5?ie=UTF8&qid=1470167137&sr=8-5&keywords=serial+usb
 * - Default serial port baudrate is 9600
 *
 * Wiring:
 * MCP9808      :       UDK
 * - Vdd        :       - D8 3.3V
 * - Gnd        :       - D3 GND
 * - SCL        :       - D15
 * - SDA        :       - D14   
 */

#include "mbed.h"
#include "mDot.h"
#include "MTSLog.h"
#include <string>
#include <vector>
#include <algorithm>

    AnalogIn ain(PB_1);
    DigitalOut dout(PA_0);


// TTN Keys, setup your application and get your own keys at https://www.thethingsnetwork.org/wiki/Backend/Overview 
static const uint8_t netowork_session_key_array[] = { 0x22, 0xDF, 0x20, 0x0C, 0x71, 0x5A, 0xCC, 0x35, 0xB8, 0x20, 0x5F, 0xAE, 0xC7, 0x00, 0x5A, 0x63 };//

//app session key
static const uint8_t data_session_key_array[] = { 0x66, 0xF2, 0x8B, 0x88, 0x76, 0x0D, 0xB0, 0xE2, 0xAE, 0x16, 0x29, 0x81, 0x29, 0xC9, 0x25, 0x11 };

//dev address (keep in mind these are all under abp)
static const uint8_t network_address_array[] = { 0x30, 0xE2, 0x6C, 0xAF };

//static const uint8_t netowork_session_key_array[] = { 0x35, 0x41, 0x32, 0x67, 0x79, 0x76, 0x49, 0x8B, 0xBE, 0x98, 0x10, 0x80, 0x76, 0xB7, 0x61, 0x8B };
//static const uint8_t data_session_key_array[] = { 0xD3, 0x6F, 0x5E, 0x66, 0x4A, 0x1B, 0xEC, 0x0C, 0x4A, 0x63, 0x8E, 0x1C, 0x2D, 0xB3, 0x18, 0xA4 };
//static const uint8_t network_address_array[] = { 0x10, 0x81, 0xDA, 0x6F };

static std::vector<uint8_t> netowork_session_key (netowork_session_key_array, netowork_session_key_array + sizeof(netowork_session_key_array) / sizeof(uint8_t));
static std::vector<uint8_t> data_session_key (data_session_key_array, data_session_key_array + sizeof(data_session_key_array) / sizeof(uint8_t));
static std::vector<uint8_t> network_address (network_address_array, network_address_array + sizeof(network_address_array) / sizeof(uint8_t));

static uint8_t config_frequency_sub_band = 2;

// Initialize I2C
I2C i2c(I2C_SDA , I2C_SCL ); 
const int addr7bit = 0x1E;      // 7 bit I2C address
const int addr8bit = addr7bit << 1; // 8bit I2C address, 0x90


int main() {
            dout = 0;

    // Declare and initialize variables
    int16_t system_voltage = 3300;
    float conversion_ratio = 0.2;
    float zero_dust_voltage = 0.4;
    float voltage = 0;
    float concentration = 0;
    
    int32_t ret;
    mDot* dot;
    std::vector<uint8_t> data;          // mDot data->send variable
    char data_str[64];                  // Intermediate conversion variable
    uint32_t update_interval = 4000;   // TTN transmission interval (loop interval)
    float analog_input;
    //initialize pins for analog input
    
    // get a mDot handle
    dot = mDot::getInstance();

    dot->resetConfig();

    dot->setLogLevel(mts::MTSLog::INFO_LEVEL);

    // too lazzy to check all errors
    dot->setJoinMode(mDot::MANUAL);
    dot->setPublicNetwork(true);
    dot->setFrequencySubBand(config_frequency_sub_band);
    dot->setNetworkSessionKey(netowork_session_key);
    dot->setDataSessionKey(data_session_key);
    dot->setNetworkAddress(network_address);
    
    
    // a higher spreading factor allows for longer range but lower throughput
    // in the 915 (US) frequency band, spreading factors 7 - 10 are available
    // in the 868 (EU) frequency band, spreading factors 7 - 12 are available
    logInfo("setting TX spreading factor");
    if ((ret = dot->setTxDataRate(mDot::SF_10)) != mDot::MDOT_OK) {
        logError("failed to set TX datarate %d:%s", ret, mDot::getReturnCodeString(ret).c_str());
    }
    
    
    // request receive confirmation of packets from the gateway
    logInfo("enabling ACKs");
    if ((ret = dot->setAck(0)) != mDot::MDOT_OK) {
        logError("failed to enable ACKs %d:%s", ret, mDot::getReturnCodeString(ret).c_str());
    }
    
    // save this configuration to the mDot's NVM
    logInfo("saving config");
    if (! dot->saveConfig()) {
        logError("failed to save configuration");
    }
    //*******************************************
    // end of configuration
    //*******************************************

    // attempt to join the network
    logInfo("joining network");
    while ((ret = dot->joinNetwork()) != mDot::MDOT_OK) {
        logError("failed to join network %d:%s", ret, mDot::getReturnCodeString(ret).c_str());
        // in the 868 (EU) frequency band, we need to wait until another channel is available before transmitting again
        osDelay(std::max((uint32_t)1000, (uint32_t)dot->getNextTxMs()));
    }

    while (true) {   
        dout = 1;
        wait_us(280);
        analog_input = ain; 
        dout = 0;
        wait_us(720);
        
        //convert to voltage in mv
        voltage = (system_voltage / 1024.0 ) * analog_input;
                logInfo("voltage = %f", voltage);

     
            //voltage to concentration
            concentration = voltage * conversion_ratio;
    
        
        


        logInfo("concentration [ug/m^3] = %f", concentration);
        
        
        // Empty data vector
        data.clear();
        
        // Push temperature value into data array
        sprintf(data_str, "%f", analog_input);
        for (int i = 0; i<strlen(data_str); i++)
        {
            data.push_back(((char*)data_str)[i]);
        }
                
        // send the data to the gateway
        if ((ret = dot->send(data)) != mDot::MDOT_OK) {
            logError("failed to send", ret, mDot::getReturnCodeString(ret).c_str());
        } else {
            logInfo("successfully sent data to gateway: %s", data_str);
        }

        // in the 868 (EU) frequency band, we need to wait until another channel is available before transmitting again
        osDelay(std::max(update_interval, (uint32_t)dot->getNextTxMs()));

    }

    return 0;
}
