/*
 * LTC2990 voltage/temprature monitor library
 *
 *
 * Copyright (c) 2013 Davy Van Belle, MIT License
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy of this software 
 * and associated documentation files (the "Software"), to deal in the Software without restriction, 
 * including without limitation the rights to use, copy, modify, merge, publish, distribute, 
 * sublicense, and/or sell copies of the Software, and to permit persons to whom the Software is 
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all copies or 
 * substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED, INCLUDING 
 * BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND 
 * NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, 
 * DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, 
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 */

/** @file
 * @brief LTC29990 I2C
 */
 
#ifndef LTC2990_H
#define LTC2990_H

#include "mbed.h"

#define STATUS 0x00
#define CONTROL 0x01
#define TRIGGER 0x02

#define TINT_MSB 0x04
#define TINT_LSB 0x05

#define V1_MSB 0x06
#define V1_LSB 0x07

#define V2_MSB 0x08
#define V2_LSB 0x09

#define V3_MSB 0x0A
#define V3_LSB 0x0B

#define V4_MSB 0x0C
#define V4_LSB 0x0D

#define VCC_MSB 0x0E
#define VCC_LSB 0x0F

#define SINGLE 0.00030518 //LSB factor for single ended measurments (305.18µV)
#define DIFF 0.00001942 //LSB factor for differential measurments (19.42µV)


enum T_SOURCE {
    INT,
    TR1,
    TR2,
};

enum V_SOURCE {
    V1,
    V2,
    V3,
    V4,
    VCC,
    V1_V2,
    V3_V4,
};


/** LTC2990 class 
 */
class LTC2990 {
public:
    /** init LTC2990 class
     * @param *i2c pointer to I2C serial interface
     * @param addr sensor I2C address
     */
    LTC2990 (I2C* i2c, char addr); 

    /** Initiate the device
     * @param control desired control register bits
     */
    void init (char control);
    
    /** Get device STATUS register
     */  
    char status();

    /** Start a conversion
     */
    void start();
    
    /** Get temperature in °C or °K   
    * @param source define which temperature to measure (INT, TR1, TR2)
    */
    float getTemperature(T_SOURCE source);
    
    /** Get voltage
    * @param source define which voltage te measure (V1, V2, V3, V4, VCC, V1-V2, V3-V4)
    */
    float getVoltage(V_SOURCE source);
    
    /** 
    *@param *ack pointer to return succesfull acknowledgement
    *@return True: Converion in progress
    *@return False: Acquisition Cycle Complete
    */
    bool isBusy();
    
    /** 
    *@return True: Register contains new data
    *@return False: Register contains old data
    *@param source define which register to check (V1, V2, V3, V4, VCC, V1-V2, V3-V4)
    */
    bool isReady(V_SOURCE source);
    
    /** 
    $@return True: Register contains new data
    *@return False: Register contains old data
    *@param source define which register to check (INT, TR1, TR2)
    */
    bool isReady(T_SOURCE source);
    
protected:


private:
    char _addr;
    I2C *_i2c;
    
};
    
#endif    