#ifndef Packet_H
#define Packet_H

#include "mbed.h"
#include <vector>
#include <string>

#define FRAME_BYTE_1 0xAA
#define FRAME_BYTE_2 0x55
#define SERVER_PORT   4000

class Packet
{
public:

    /**
     * Makes an object of Packet with initial values 0
     */
    Packet();
    
    /**
     * Converts the data from the packet to a data string used to send to an other MBED
     *
     * @return the data string with the packet data included
     * @ref Communication
     */
    std::string toDataString();
    
    /**
     * Inputs the data string received by the Communication class, takes the data
     * out of this string to input this in the packet
     *
     * @param str the received data from the Communication class
     * @ref Communication
     */
    void dataInput(std::string str);

    /**
     * Adds and modifies the inserted data to the packet
     *
     * @param num the number of ids that has been visited (will be added +1 in the function)
     * @param destID the destination id that has to be changed
     * @param pwm the pwm that has to be changed
     * @param ownID the id that has to be added to the list
     * @param temp the temperature that has to be added to the list
     */
    void addValues(uint8_t num, uint8_t destID,uint8_t pwm,uint8_t ownID,int16_t temp);

    /**
     * Calculates the crc checksum of a given string, ending with a trailing character
     * chk.
     * - If this check byte is zero it will calculate the crc checksum itself
     * - If the check byte has a value, the returned char will be 0 if chk is
     *      the correct checksum value.
     *
     * @param str the string where the checksum has to be calculated from
     * @param chk the checksum value that has to be checked (will be 0 if the crc checksum has to be calculated)
     *
     * @return the calculated checksum char
     */
    unsigned char crc(std::string str, char chk);

    /**
     * Checks if the CRC is the correct value in the packet
     *
     * @return true if the crc is correct, else false
     */
    bool crcCheck();

    /**
     * Checks if the IDN vector contains an ID
     *
     * @param ID the ID that has to be checked
     * @return true if IDN vector contains the ID, else false
     */
    bool idCheck(int ID);

    /** Sets the number of MBEDs visited
     *
     * @param NUM the number of MBEDs visited
     */
    void setNUM(uint8_t NUM);

    /** Returns the amount of mbeds visited
     *
     * @return number of MBEDs visited
     */
    uint8_t getNUM();

    /** Sets the destination ID
     *
     * @param IDD the destination ID
     */
    void setIDD(uint8_t IDD);

    /** Returns the destination ID
     *
     * @return the destination ID
     */
    uint8_t getIDD();

    /** Sets the CRC byte
     *
     * @param CRC the CRC byte
     */
    void setCRC(uint8_t CRC);

    /** Returns the CRC byte
     *
     * @return the CRC byte
     */
    uint8_t getCRC();

    /** Sets the pulse width modulation value
     *
     * @param PWM the pulse width modulation value
     */
    void setPWM(uint8_t PWM);

    /** Returns the pulse width modulation value
     *
     * @return the pulse width modulation value
     */
    uint8_t getPWM();

    /** Sets vector IDN to an other vector
     *
     * @param IDN a vector with the visited IDS
     */
    void setIDN(vector<uint8_t> IDN);

    /** Returns the vector with all visited IDs
     *
     * @return the vector with all visited IDs
     */
    vector<uint8_t> getIDN();

    /** Sets the vector TMP to an other vector
     *
     * @param TMP a vector with all visted MBED's temperatures
     */
    void setTMP(vector<int16_t> TMP);

    /** Returns the vector of Temperatures
     *
     * @return the vector of Temperatures
     */
    vector<int16_t> getTMP();


private:
    std::string crcString();
    uint8_t NUM;
    uint8_t IDD;
    uint8_t PWM;
    uint8_t CRC;
    vector<uint8_t> IDN;
    vector<int16_t> TMP;


};

#endif