 /* Copyright (c) 2009 Nordic Semiconductor. All Rights Reserved.
 *
 * The information contained herein is property of Nordic Semiconductor ASA.
 * Terms and conditions of usage are described in detail in NORDIC
 * SEMICONDUCTOR STANDARD SOFTWARE LICENSE AGREEMENT.
 *
 * Licensees are granted free, non-transferable use of the information. NO
 * WARRANTY of ANY KIND is provided. This heading must NOT be removed from
 * the file.
 *
 */

#ifndef SPI_MASTER_H
#define SPI_MASTER_H

#include <stdbool.h>
#include <stdint.h>

/* @file
* @brief Software controlled SPI Master driver.
*
*
* @defgroup lib_driver_spi_master Software controlled SPI Master driver
* @{
* @ingroup nrf_drivers
* @brief Software controlled SPI Master driver.
*
* Supported features:
* - Operate two SPI masters independently or in parallel.
* - Transmit and Receive given size of data through SPI.
* - configure each SPI module separately through @ref spi_master_init.
*/

#if defined(TARGET_DELTA_DFCM_NNN40) 

#define COEX_B          5      //Set to 26 as default, BLE coexistence pin config, used only on Delta DFCM-NNN40 platform
#define COEX_W          6      //Set to 27 as default, WiFi coexistence pin config, used only on Delta DFCM-NNN40 platform
#define WIFI_SLEEP      NC      //Set to NC, do NOT modify

#define WIFI_BOOT       p28     //Set to p28, do NOT modify
#define WIFI_SPI_IRQN   p14     //Set to p14, do NOT modify 
#define WIFI_SPI_MOSI   p15     //Set to p15, do NOT modify
#define WIFI_SPI_MISO   p9      //Set to p9, do NOT modify
#define WIFI_SPI_SCK    p11     //Set to p11, do NOT modify
#define WIFI_SPI_SSN    p12     //Set to p12, do NOT modify
#define WIFI_SPI_Frequency    4000000   //Set to 4M (maximum), do NOT modify

#else

/* Note that the logic level of all NNN40 module GPIO is 1.8V */
#define COEX_B          NC      //Set to NC, do NOT modify
#define COEX_W          NC      //Set to NC, do NOT modify
#define WIFI_SLEEP      PC_0    //Configure host GPIO to connect to NNN40 module pin 24, set to NC if module power can be on/off by host 

#define WIFI_BOOT       PC_1    //Configure host GPIO to connect to NNN40 module pin 30 
#define WIFI_SPI_IRQN   PB_0    //Configure host GPIO to connect to NNN40 module pin 14 
#define WIFI_SPI_MOSI   SPI_MOSI//Configure host GPIO to connect to NNN40 module pin 15 
#define WIFI_SPI_MISO   SPI_MISO//Configure host GPIO to connect to NNN40 module pin 31 
#define WIFI_SPI_SCK    SPI_SCK //Configure host GPIO to connect to NNN40 module pin 32 
#define WIFI_SPI_SSN    PB_6    //Configure host GPIO to connect to NNN40 module pin 33 
#define WIFI_SPI_Frequency    4000000   //Set to 4M (maximum), do NOT modify

#endif


#ifdef __cplusplus//Tsungta, 2015/10/04, added so spi_master.c can convert to cpp (mbed SPI)
extern "C"
{
#endif
/**
 * @brief Function for initializing given SPI master with given configuration.
 *
 * After initializing the given SPI master with given configuration, this function also test if the
 * SPI slave is responding with the configurations by transmitting few test bytes. If the slave did not
 * respond then error is returned and contents of the rx_data are invalid.
 *
 * @return
 * @retval pointer to direct physical address of the requested SPI module if init was successful
 * @retval 0, if either init failed or slave did not respond to the test transfer
 */
uint32_t* spi_master_init();

/**
 * @brief Function for transferring/receiving data over SPI bus.
 *
 * If TWI master detects even one NACK from the slave or timeout occurs, STOP condition is issued
 * and the function returns false.
 *
 * @note Make sure at least transfer_size number of bytes is allocated in tx_data/rx_data.
 *
 * @param spi_base_address  register base address of the selected SPI master module
 * @param transfer_size  number of bytes to transmit/receive over SPI master
 * @param tx_data pointer to the data that needs to be transmitted
 * @param rx_data pointer to the data that needs to be received
 * @return
 * @retval true if transmit/reveive of transfer_size were completed.
 * @retval false if transmit/reveive of transfer_size were not complete and tx_data/rx_data points to invalid data.
 */
//bool spi_master_tx_rx(uint32_t *spi_base_address, uint16_t transfer_size, const uint8_t *tx_data, uint8_t *rx_data);
bool spi_master_tx_rx(uint8_t module_number, uint16_t transfer_size, const uint8_t *tx_data, uint8_t *rx_data);
#ifdef __cplusplus
}
#endif
/**
 *@}
 **/
 
#endif /* SPI_MASTER_H */

