/**
 * File: wifi-cli.c
 * Description: WIFI CLI commands used by all applications WIFI of profile.
 *
 * Copyright 2015 by CYNTEC Corporation.  All rights reserved.
 * Author: Lester Lee
 */

#include "wifi_cli.h"

#define AP_SSID_MAX_LEN 33
#define AP_PW_MAX_LEN 64
#define STATIC_IP_MAX_LEN 15
#define TCP_SEND_MAX_LEN 1400
#define UDP_SEND_MAX_LEN 1400
//#define DELTA_WIFI_DEBUG
#define WIFI_MAX_AP_NUMBER 15
// WiFI_API_32kRAM ver. 8
#define MAX_MEM_WRITE_LENGTH 0x3FFFF
#define MAX_MEM_ERASE_ADDR 0x3F000
#define MAX_DATA_LENGTH 256

// General configuration parameters
WIFIDevice wifiDevice;
TCPSocketConnection tcpConnect;
UDPSocket udpSocket;
Endpoint cliEndpoint;
TCPSocketServer tcpServer;
//extern uint8_t isValidGPIO(uint8_t); // from ble_cli.cpp
uint8_t static_ip[STATIC_IP_MAX_LEN] = "192.168.1.100";
extern Serial console;
//Serial console(p25,p23);
extern const char* cyntecCommandErrorNames[];

static uint8_t is_Listen = false;
static uint8_t apInd;
static unsigned char *preSSID;
static unsigned char *newSSID;
static scanApInfo saveAP[WIFI_MAX_AP_NUMBER];
static bool wifiIniState = false;
static bool wifiDataInt = false; // wifi receive callback flag

typedef struct deviceNetwork_s {
    char ap_ssid[AP_SSID_MAX_LEN+1];
    char ap_pw[AP_PW_MAX_LEN+1];
} deviceNetwork_t;
deviceNetwork_t devNetwork[4]; // 0-2 for setNetwork, 3 for set_ap
DigitalOut wifiWriteInt(p30);

static uint8_t cyntecIsValidIP(uint8_t *startIdx, uint8_t totalLen)
{
    uint8_t *ipIdx = startIdx;
    uint8_t *tempIdx = ipIdx;
    uint8_t IPAddr[3];
    //uint8_t IPTokenLen = 0;
    int16_t ipToken;
    while ( (tempIdx - startIdx) <= totalLen ) {
        if (*tempIdx == '.' || ((tempIdx - startIdx) == totalLen)) {
            memset( IPAddr, 0, 3);
            memcpy( IPAddr, ipIdx, (tempIdx - ipIdx));

            ipToken = atoi((const char *)IPAddr);
            if (ipToken > 255 || ipToken < 0)
                return 1;

            ipIdx = tempIdx + 1;
        }
        tempIdx++;
    }
    return 0;
}

static void cyntecPrintOk(void)
{
    console.printf("\r\nOK\r\n\r\n");
}

static void cyntecPrintError(uint8_t errIdx)
{
    console.printf("\r\nERROR;%s\r\n\r\n",cyntecCommandErrorNames[errIdx]);
}

/////////**** WIFI Device Implement ****//////////
static void cyn_wifi_device_sleep()
{
    if (cyntecGetCommandTokenCnt() != 2) {
        cyntecPrintError(CYNTEC_CMD_ERR_WRONG_NUMBER_OF_ARGUMENTS);
        return;
    }
    uint8_t err_code = wifiDevice.sleep();
    if (err_code == 0)
        cyntecPrintOk();
    else {
        console.printf("ERROR;%d\r\n",err_code);
        return;
    }
}

static void cyn_wifi_device_coex()
{
    if (cyntecGetCommandTokenCnt() != 2) {
        cyntecPrintError(CYNTEC_CMD_ERR_WRONG_NUMBER_OF_ARGUMENTS);
        return;
    }
    uint8_t err_code = wifiDevice.enableCoexistance();
    if (err_code == 0)
        cyntecPrintOk();
    else {
        console.printf("ERROR;%d\r\n",err_code);
        return;
    }
}

static void cyn_wifi_device_network()
{
    if (wifiIniState== true) {
        cyntecPrintError(CYNTEC_CMD_ERR_WRONG_CMD_ORDER);
        return;
    }
    if (cyntecGetCommandTokenCnt() == 5) {
        uint8_t argLen = 0;
        uint8_t *argSSID;
        uint8_t *argPW;
        uint8_t *argPriority;
        /* 0~2, 0 is highest */
        uint8_t priority = 0;

        /* handle priority */
        argPriority = cyntecGetCommandArgument(2, NULL);
        switch(*argPriority) {
            case '0':
                priority = 0;
                break;
            case '1':
                priority = 1;
                break;
            case '2':
                priority = 2;
                break;
            default:
                cyntecPrintError(CYNTEC_CMD_ERR_ARGUMENT_OUT_OF_RANGE);
                return;
        }

        /* handle SSID */
        argSSID = cyntecGetCommandArgument(0, &argLen);
        if ( argLen > AP_SSID_MAX_LEN ) {
            cyntecPrintError(CYNTEC_CMD_ERR_ARGUMENT_OUT_OF_RANGE);
            return;
        }
        memset( devNetwork[priority].ap_ssid , 0, AP_SSID_MAX_LEN+1);
        memcpy( devNetwork[priority].ap_ssid, argSSID, argLen);

        /* handle Password */
        argPW = cyntecGetCommandArgument(1, &argLen);
        if ( argLen > AP_PW_MAX_LEN ) {
            cyntecPrintError(CYNTEC_CMD_ERR_ARGUMENT_OUT_OF_RANGE);
            return;
        }
        memset( devNetwork[priority].ap_pw, 0, AP_PW_MAX_LEN+1);
        memcpy( devNetwork[priority].ap_pw, argPW, argLen);

        /* call setup API */
        wifiDevice.setNetwork(devNetwork[priority].ap_ssid, devNetwork[priority].ap_pw, priority);
        cyntecPrintOk();
    } else {
        cyntecPrintError(CYNTEC_CMD_ERR_WRONG_NUMBER_OF_ARGUMENTS);
    }
}

// gill 20150910
// Comply to WIFI_API_32kRAM version 8
static void cyn_wifi_read_sp_version()
{
    wifiSpInfo version = wifiDevice.read_WIFI_SP_version();
    cyntecPrintOk();
    console.printf("%X;", version.chipID);
    console.printf("%i;", version.imgStaID);
    console.printf("%i\r\n", version.imgApID);
}

void scanCallback(scanApInfo result)
{
    newSSID = result.ssid.ssid;
    // Check if SSID repeated
    if (memcmp(newSSID,preSSID,result.ssid.ssid_len)) {
#ifdef DELTA_WIFI_DEBUG
        console.printf("SSID repeat\r\n");
#endif
    } else {
        console.printf("%i,", apInd);
        console.printf("%s,", result.ssid.ssid);
        //console.printf("%i,", result.ssid.ssid_len);
        console.printf("%i,", result.ApRssi);
        console.printf("%i,", result.channelNo);
        console.printf("%i\r\n", result.secType);
        apInd++;
    }
    preSSID = result.ssid.ssid;
}

static void cyn_wifi_device_ap_scan()
{
    if (wifiIniState== false) {
        cyntecPrintError(CYNTEC_CMD_ERR_WRONG_CMD_ORDER);
        return;
    }
    memset(saveAP,0,sizeof(saveAP));
    apInd = 1;
    int scanResult = wifiDevice.apScan(scanCallback);
    if (scanResult >= 0) {
#ifdef DELTA_WIFI_DEBUG
        console.printf("scan AP number:%d\r\n",scanResult);
#endif
        cyntecPrintOk();
        return;
    } else {
        cyntecPrintError(CYNTEC_CMD_ERR_CALL_FAIL);
        return;
    }
}

static void cyn_wifi_set_ap()
{
    if (cyntecGetCommandTokenCnt() != 6) {
        cyntecPrintError(CYNTEC_CMD_ERR_WRONG_NUMBER_OF_ARGUMENTS);
    }
    uint8_t argLen = 0;
    uint8_t * argSSID;
    uint8_t * argPW;
    uint8_t * argSEC;
    uint8_t * argCH;

    /* handle SSID */
    argSSID = cyntecGetCommandArgument(0, &argLen);
    if ( argLen > AP_SSID_MAX_LEN ) {
        cyntecPrintError(CYNTEC_CMD_ERR_ARGUMENT_OUT_OF_RANGE);
        return;
    }
    memset( devNetwork[3].ap_ssid , 0, AP_SSID_MAX_LEN+1);
    memcpy( devNetwork[3].ap_ssid, argSSID, argLen);

    /* handle Password */
    argPW = cyntecGetCommandArgument(1, &argLen);
    if ( argLen > AP_PW_MAX_LEN ) {
        cyntecPrintError(CYNTEC_CMD_ERR_ARGUMENT_OUT_OF_RANGE);
        return;
    }
    memset( devNetwork[3].ap_pw, 0, AP_PW_MAX_LEN+1);
    memcpy( devNetwork[3].ap_pw, argPW, argLen);

    /* handle Security */
    security_t setSec  = SECURITY_WPA2_AES_PSK;
    argSEC = cyntecGetCommandArgument(2, &argLen);
    char numSec[argLen];
    memcpy( numSec, argSEC, argLen);
    uint8_t sec = atoi(numSec);
    //uint8_t numSec = atoi((const char *)arg);

    if ( (int)sec > 7 || (int)sec < 0) {
        cyntecPrintError(CYNTEC_CMD_ERR_ARGUMENT_OUT_OF_RANGE);
        return;
    }
    security_t arrySec[8]= {SECURITY_OPEN,
                            SECURITY_WEP_PSK,
                            SECURITY_WEP_SHARED,
                            SECURITY_WPA_TKIP_PSK,
                            SECURITY_WPA_AES_PSK,
                            SECURITY_WPA2_AES_PSK,
                            SECURITY_WPA2_TKIP_PSK,
                            SECURITY_WPA2_MIXED_PSK
                           };
    for (uint8_t i=0; i<8; i++) {
        if(sec==i)
            setSec = arrySec[i];
    }

    /* handle Channel */
    argCH = cyntecGetCommandArgument(3, &argLen);
    uint8_t setChan = atoi((char *)argCH);

    if ( (int)setChan > 14 || (int)setChan < 1) {
        cyntecPrintError(CYNTEC_CMD_ERR_ARGUMENT_OUT_OF_RANGE);
        return;
    }
#ifdef DELTA_WIFI_DEBUG
    console.printf("argSSID:%s\r\n",devNetwork[3].ap_ssid);
    console.printf("argPW:%s\r\n",devNetwork[3].ap_pw);
    console.printf("sec:%i\r\n",sec);
    console.printf("setChan:%i\r\n",setChan);
#endif

    /* call setup API */
    wifiDevice.setAccessPoint(devNetwork[3].ap_ssid, devNetwork[3].ap_pw, setSec, setChan);
    cyntecPrintOk();

}

static void cyn_wifi_mem_erase4k()
{
    if (cyntecGetCommandTokenCnt() == 3) {
        uint8_t argLen = 0;
        uint8_t *arg;
        uint32_t setAddr;
        arg = cyntecGetCommandArgument(0, &argLen);
        if (arg[0] != '0' || arg[1] != 'x') {
            cyntecPrintError(CYNTEC_CMD_ERR_ARGUMENT_SYNTAX_ERROR);
            return;
        }
        setAddr = cyntecHexToUint32(arg+2,argLen-2);
        if (setAddr % 0x1000 != 0 || setAddr > MAX_MEM_ERASE_ADDR) {
            cyntecPrintError(CYNTEC_CMD_ERR_ARGUMENT_OUT_OF_RANGE);
            return;
        }
#ifdef DELTA_WIFI_DEBUG
        console.printf("setAddr:%08X\r\n",setAddr);
#endif
        int err_code = wifiDevice.storage_erase4KB(setAddr);
        if (err_code == 0)
            cyntecPrintOk();
        else
            console.printf("ERROR;\r\n");
    } else {
        cyntecPrintError(CYNTEC_CMD_ERR_WRONG_NUMBER_OF_ARGUMENTS);
    }
}

static void cyn_wifi_mem_write()
{
    uint8_t argLen = 0;
    uint8_t *arg;
    uint32_t setAddr;
    uint8_t setData[MAX_DATA_LENGTH] = {0};
    uint16_t writeLen;
    arg = cyntecGetCommandArgument(0, &argLen);
    if (arg[0] != '0' || arg[1] != 'x') {
        cyntecPrintError(CYNTEC_CMD_ERR_ARGUMENT_SYNTAX_ERROR);
        return;
    }
    setAddr = cyntecHexToUint32(arg+2,argLen-2);
#ifdef DELTA_WIFI_DEBUG
    console.printf("%08X\r\n",setAddr);
#endif
    if (setAddr > MAX_MEM_WRITE_LENGTH) {
        cyntecPrintError(CYNTEC_CMD_ERR_ARGUMENT_OUT_OF_RANGE);
        return;
    }
    arg = cyntecGetCommandArgument(1, &argLen);
    if (arg[0]!='0' || arg[1]!='x' || (argLen-2)%2!=0) {
        cyntecPrintError(CYNTEC_CMD_ERR_ARGUMENT_SYNTAX_ERROR);
        return;
    }
    //memset(setData,0,(argLen-2)/2);
    for (int i=0 ; i < (argLen-2)/2; i++) {
        setData[i]  = cyntecArgToUint8(arg+2*(i+1), 2);
#ifdef DELTA_WIFI_DEBUG
        console.printf("%02X",setData[i]);
#endif
    }
    writeLen = (argLen-2)/2;
    // gill need confirm
    if (writeLen > MAX_MEM_WRITE_LENGTH) {
        cyntecPrintError(CYNTEC_CMD_ERR_ARGUMENT_OUT_OF_RANGE);
        return;
    }
#ifdef DELTA_WIFI_DEBUG
    console.printf("\r\nwriteLen:%i\r\n",writeLen);
#endif
    int err_code = wifiDevice.storage_write(setAddr,setData,writeLen);
    if (err_code == 0)
        cyntecPrintOk();
    else
        console.printf("ERROR;%d\r\n",err_code);
}

static void cyn_wifi_mem_read()
{
    uint8_t argLen = 0;
    uint8_t *arg;
    uint32_t setAddr;
    uint8_t setData[MAX_DATA_LENGTH] = {0};
    uint16_t readLen;
    arg = cyntecGetCommandArgument(0, &argLen);
    if (arg[0] != '0' || arg[1] != 'x') {
        cyntecPrintError(CYNTEC_CMD_ERR_ARGUMENT_SYNTAX_ERROR);
        return;
    }
    setAddr = cyntecHexToUint32(arg+2,argLen-2);
#ifdef DELTA_WIFI_DEBUG
    console.printf("%08X\r\n",setAddr);
#endif
    if (setAddr > MAX_MEM_WRITE_LENGTH) {
        cyntecPrintError(CYNTEC_CMD_ERR_ARGUMENT_OUT_OF_RANGE);
        return;
    }
    arg = cyntecGetCommandArgument(1, &argLen);

    readLen = cyntecAtoiUint16(arg,argLen);

#ifdef DELTA_WIFI_DEBUG
    console.printf("\r\nreadLen:%04X\r\n",readLen);
#endif

    int err_code = wifiDevice.storage_read(setAddr,setData,readLen);
    if (err_code == 0) {
        cyntecPrintOk();
        for (int i=0; i<readLen; i++) {
            console.printf("%02X",setData[i]);
        }
        console.printf("\r\n");
    } else
        console.printf("ERROR;%d\r\n",err_code);
}

/////////**** WIFI Ethernet Implement ****//////////
static void cyn_wifi_ethernet_init()
{
    uint8_t argLen = 0;
    uint8_t *argIP;
    EthernetInterface ethInterface;
    uint8_t result;
    if (cyntecGetCommandTokenCnt() == 2) {
        /* use DHCP to get IP */
        result = ethInterface.init();
        if ( result == 0 ) {
            cyntecPrintOk();
            wifiIniState = true;
        } else {
            console.printf("function err:%d\r\n",result);
            cyntecPrintError(CYNTEC_CMD_ERR_CALL_FAIL);
        }

    } else if ( cyntecGetCommandTokenCnt() == 3 ) {
        /* use static to get IP */
        argIP = cyntecGetCommandArgument(0, &argLen);

        if ( cyntecIsValidIP(argIP,argLen) != 0) {
            cyntecPrintError(CYNTEC_CMD_ERR_ARGUMENT_OUT_OF_RANGE);
            return;
        }

        memset( static_ip, 0, STATIC_IP_MAX_LEN);
        memcpy( static_ip, argIP, argLen);
        if ( ethInterface.init((const char *)static_ip, NULL, NULL) == 0) {
            cyntecPrintOk();
            wifiIniState = true;
        } else
            cyntecPrintError(CYNTEC_CMD_ERR_CALL_FAIL);
    } else {
        cyntecPrintError(CYNTEC_CMD_ERR_WRONG_NUMBER_OF_ARGUMENTS);
    }

    return;
}
static void cyn_wifi_ethernet_connect()
{
    int timeout_ms = 35000;
    uint8_t *argTimeout;
    EthernetInterface ethInterface;
    if (cyntecGetCommandTokenCnt() != 2 & cyntecGetCommandTokenCnt() != 3)
        cyntecPrintError(CYNTEC_CMD_ERR_WRONG_NUMBER_OF_ARGUMENTS);
    if (cyntecGetCommandTokenCnt() == 2) {
#ifdef DELTA_WIFI_DEBUG
        console.printf("\r\n");
        console.printf("Connecting..., Waiting for 35000 ms...");
        console.printf("\r\n\r\n");
#endif
        if ( ethInterface.connect() == 0 )
            cyntecPrintOk();
        else
            cyntecPrintError(CYNTEC_CMD_ERR_CALL_FAIL);
    }
    if (cyntecGetCommandTokenCnt() == 3) {
        argTimeout = cyntecGetCommandArgument(0, NULL);
        timeout_ms = atoi((const char*)argTimeout);

        if (timeout_ms < 0) {
            cyntecPrintError(CYNTEC_CMD_ERR_ARGUMENT_OUT_OF_RANGE);
            return;
        }
#ifdef DELTA_WIFI_DEBUG
        console.printf("\r\nConnecting..., Waiting for ");
        console.printf((char*)argTimeout);
        console.printf(" ms...\r\n\r\n");
#endif
        if ( ethInterface.connect(timeout_ms) == 0 )
            cyntecPrintOk();
        else
            cyntecPrintError(CYNTEC_CMD_ERR_CALL_FAIL);
    }
    return;
}
static void cyn_wifi_ethernet_disconnect()
{
    EthernetInterface ethInterface;

    ethInterface.disconnect();
    cyntecPrintOk();
    return;
}
static void cyn_wifi_ethernet_mac()
{
    EthernetInterface ethInterface;

    char mac_addr[19];
    memset(mac_addr, 0, 19);
    memcpy(mac_addr, ethInterface.getMACAddress(), 19);

    console.printf("\r\nOK;%s\r\n\r\n",mac_addr);

    return;
}
static void cyn_wifi_ethernet_ip()
{
    EthernetInterface ethInterface;

    char ip_addr[15] = "\0";
    memset(ip_addr, 0, 15);
    memcpy(ip_addr, ethInterface.getIPAddress(), 15);

    console.printf("\r\nOK;%s\r\n\r\n",ip_addr);

    return;
}
/* not implement yet... */
////static void cyn_wifi_ethernet_gateway(){}
////static void cyn_wifi_ethernet_mask(){}
/* not implement yet... */

/////////**** WIFI TCP Socket Server Implement ****//////////
static void cyn_wifi_tcp_server_bind()
{
//		TCPSocketServer tcpServer;

    int port = 1;
    if (cyntecGetCommandTokenCnt() == 2) {
        if ( tcpServer.bind(port) == 0 )
            cyntecPrintOk();
        else
            cyntecPrintError(CYNTEC_CMD_ERR_CALL_FAIL);

    }	else if (cyntecGetCommandTokenCnt() == 3) {
        port = atoi((const char*)(cyntecGetCommandArgument(0, NULL)));
#ifdef DELTA_WIFI_DEBUG
        console.printf("port:%i\r\n",port);
#endif
        if (port < 0) {
            cyntecPrintError(CYNTEC_CMD_ERR_ARGUMENT_OUT_OF_RANGE);
            return;
        }

        if ( tcpServer.bind(port) == 0 )
            cyntecPrintOk();
        else
            cyntecPrintError(CYNTEC_CMD_ERR_CALL_FAIL);

    }	else {
        cyntecPrintError(CYNTEC_CMD_ERR_WRONG_NUMBER_OF_ARGUMENTS);
    }
}
static void cyn_wifi_tcp_server_listen()
{
//		TCPSocketServer tcpServer;

    int backlog = 1;
    if (cyntecGetCommandTokenCnt() == 2) {
        if ( tcpServer.listen(backlog) == 0 ) {
            cyntecPrintOk();
            is_Listen = true;
        } else
            cyntecPrintError(CYNTEC_CMD_ERR_CALL_FAIL);
    }	else {
        cyntecPrintError(CYNTEC_CMD_ERR_WRONG_NUMBER_OF_ARGUMENTS);
    }
}
static void cyn_wifi_tcp_server_accept()
{
//		TCPSocketServer tcpServer;

    if (cyntecGetCommandTokenCnt() == 2) {
        if (is_Listen == false) {
            cyntecPrintError(CYNTEC_CMD_ERR_INVALID_STATE_TO_PERFORM_OPERATION);
            return;
        }

        if ( tcpServer.accept(tcpConnect) == 0 ) {
            cyntecPrintOk();
            is_Listen = false;
        } else
            cyntecPrintError(CYNTEC_CMD_ERR_CALL_FAIL);

    }	else {
        cyntecPrintError(CYNTEC_CMD_ERR_WRONG_NUMBER_OF_ARGUMENTS);
    }
}
static void cyn_wifi_tcp_server_blocking()
{
//		TCPSocketServer tcpServer;

    bool blocking = false;
    unsigned int timeout;

    uint8_t *arg;

    if (cyntecGetCommandTokenCnt() == 3) {

        arg = cyntecGetCommandArgument(0, NULL);
        if (arg[0] == '1')
            blocking = true;

        tcpServer.set_blocking(blocking, 1500);
        cyntecPrintOk();

    } else if (cyntecGetCommandTokenCnt() == 4) {

        arg = cyntecGetCommandArgument(0, NULL);
        if (arg[0] == '1')
            blocking = true;

        arg = cyntecGetCommandArgument(1, NULL);
        timeout = (unsigned int)atoi((const char *)arg);

        tcpServer.set_blocking(blocking, timeout);
        cyntecPrintOk();

    }	else {
        cyntecPrintError(CYNTEC_CMD_ERR_WRONG_NUMBER_OF_ARGUMENTS);
    }
}
static void cyn_wifi_tcp_server_close()
{
//		TCPSocketServer tcpServer;

    bool shutdown = true;
    uint8_t *arg;

    if (cyntecGetCommandTokenCnt() == 2) {
        if ( tcpServer.close(shutdown) == 0 )
            cyntecPrintOk();
        else
            cyntecPrintError(CYNTEC_CMD_ERR_CALL_FAIL);

    }	else if (cyntecGetCommandTokenCnt() == 3) {
        arg = cyntecGetCommandArgument(0, NULL);
        if (arg[0] == '0')
            shutdown = false;

        if ( tcpServer.close(shutdown) == 0 )
            cyntecPrintOk();
        else
            cyntecPrintError(CYNTEC_CMD_ERR_CALL_FAIL);

    }	else {
        cyntecPrintError(CYNTEC_CMD_ERR_WRONG_NUMBER_OF_ARGUMENTS);
    }
}
/////////**** WIFI TCP Socket Connection ****//////////
static void cyn_wifi_tcp_connection_connect()
{
    uint8_t connect_ip[STATIC_IP_MAX_LEN];
    int port;

    uint8_t argLen = 0;
    uint8_t *argIP;

    if (cyntecGetCommandTokenCnt() == 4) {
        /* handle IP arg */
        argIP = cyntecGetCommandArgument(0, &argLen);
        memset( connect_ip, 0, STATIC_IP_MAX_LEN);
        memcpy( connect_ip, argIP, argLen);

        /* handle Port arg */
        port = atoi((const char *)cyntecGetCommandArgument(1, NULL));

        if ( tcpConnect.connect((const char *)connect_ip, port) == 0 )
            cyntecPrintOk();
        else
            cyntecPrintError(CYNTEC_CMD_ERR_CALL_FAIL);

    }	else {
        cyntecPrintError(CYNTEC_CMD_ERR_WRONG_NUMBER_OF_ARGUMENTS);
    }
}
static void cyn_wifi_tcp_connection_is_connected()
{
    bool is_connected = false;
    is_connected = tcpConnect.is_connected();

    if (is_connected == true)
        console.printf("\r\nOK;TRUE\r\n\r\n");
    else
        console.printf("\r\nOK;FALSE\r\n\r\n");
}

static void cyn_wifi_tcp_connection_send()
{
    char msg[TCP_SEND_MAX_LEN+1];
    int sendData;
    uint8_t * argAllBuf = cyntecGetCommandTotalBuffer();
    int sendLen = 0; // Maximum 1400
    uint8_t sendLenCharNum = 0; // Maximum 4
    uint8_t *argLeng = cyntecGetCommandArgument(0,&sendLenCharNum);
    sendLen = cyntecAtoInt(argLeng,sendLenCharNum);
    if (sendLen > TCP_SEND_MAX_LEN || sendLen < 0) {
        cyntecPrintError(CYNTEC_CMD_ERR_ARGUMENT_OUT_OF_RANGE);
        return;
    }
    if (cyntecGetCommandTokenCnt() <4 ) {
        cyntecPrintError(CYNTEC_CMD_ERR_WRONG_NUMBER_OF_ARGUMENTS);
        return;
    }
    if (cyntecGetCommandTokenCnt() >= 4) {
        if ( tcpConnect.is_connected() == false ) {
            cyntecPrintError(CYNTEC_CMD_ERR_INVALID_STATE_TO_PERFORM_OPERATION);
            return;
        }

        /* handle Message arg */
#ifdef DELTA_WIFI_DEBUG
        console.printf("sendLen:%i,Buf:%s,Index:%i\r\n",sendLen,&argAllBuf[26+sendLenCharNum],cyntecGetTotalIndex());
#endif
        memset( msg, 0, TCP_SEND_MAX_LEN+1);
        for (uint8_t i=0; i<sendLen; i++)
            msg[i] = argAllBuf[26+sendLenCharNum+i];
#ifdef DELTA_WIFI_DEBUG
        console.printf("msg:%s\r\n",msg);
#endif
        sendData = tcpConnect.send(msg, sendLen);
        if ( sendData >= 0 ) {
            console.printf("\r\nOK;");
            console.printf("%i\r\n\r\n",sendData);
        } else
            cyntecPrintError(CYNTEC_CMD_ERR_CALL_FAIL);
    }
}

static void cyn_wifi_tcp_connection_send_all()
{
    char msg[TCP_SEND_MAX_LEN+1];
    int sendData;
    uint8_t * argAllBuf = cyntecGetCommandTotalBuffer();
    int sendLen = 0; // Maximum 1400
    uint8_t sendLenCharNum = 0; // Maximum 4
    uint8_t *argLeng = cyntecGetCommandArgument(0,&sendLenCharNum);
    sendLen = cyntecAtoi(argLeng,sendLenCharNum);
    if (sendLen > TCP_SEND_MAX_LEN) {
        cyntecPrintError(CYNTEC_CMD_ERR_ARGUMENT_OUT_OF_RANGE);
        return;
    }
    if (cyntecGetCommandTokenCnt() <4 ) {
        cyntecPrintError(CYNTEC_CMD_ERR_WRONG_NUMBER_OF_ARGUMENTS);
        return;
    }
    if (cyntecGetCommandTokenCnt() >= 4) {
        if ( tcpConnect.is_connected() == false ) {
            cyntecPrintError(CYNTEC_CMD_ERR_INVALID_STATE_TO_PERFORM_OPERATION);
            return;
        }

        /* handle Message arg */
#ifdef DELTA_WIFI_DEBUG
        console.printf("sendLen:%i,Buf:%s,Index:%i\r\n",sendLen,&argAllBuf[30+sendLenCharNum],cyntecGetTotalIndex());
#endif
        memset( msg, 0, TCP_SEND_MAX_LEN+1);
        for (uint8_t i=0; i<sendLen; i++)
            msg[i] = argAllBuf[30+sendLenCharNum+i];
#ifdef DELTA_WIFI_DEBUG
        console.printf("msg:%s\r\n",msg);
#endif
        sendData = tcpConnect.send_all(msg, sendLen);
        if ( sendData >= 0 ) {
            console.printf("\r\nOK;");
            console.printf("%i\r\n\r\n",sendData);
        } else
            cyntecPrintError(CYNTEC_CMD_ERR_CALL_FAIL);
    }
}

static void cyn_wifi_tcp_connection_receive()
{
    char msg[TCP_SEND_MAX_LEN+1];
    int argLen = 0;

    if (cyntecGetCommandTokenCnt() == 3) {
        /* handle Message arg */
        argLen = atoi((const char *)cyntecGetCommandArgument(0, NULL));
        //console.printf("argLen:%d\r\n",argLen);
        if (argLen > TCP_SEND_MAX_LEN || argLen < 0) {
            cyntecPrintError(CYNTEC_CMD_ERR_ARGUMENT_OUT_OF_RANGE);
            return;
        }

        memset( msg, 0, TCP_SEND_MAX_LEN+1);

        if ( tcpConnect.receive(msg, argLen) >= 0 ) {
            console.printf("\r\nOK;%s\r\n\r\n",msg);
        } else
            cyntecPrintError(CYNTEC_CMD_ERR_CALL_FAIL);

    }	else {
        cyntecPrintError(CYNTEC_CMD_ERR_WRONG_NUMBER_OF_ARGUMENTS);
    }
}
static void cyn_wifi_tcp_connection_receive_all()
{
    char msg[TCP_SEND_MAX_LEN+1];
    int argLen = 0;

    if (cyntecGetCommandTokenCnt() == 3) {
        /* handle Message arg */
        argLen = atoi((const char *)cyntecGetCommandArgument(0, NULL));

        if (argLen > TCP_SEND_MAX_LEN || argLen < 0) {
            cyntecPrintError(CYNTEC_CMD_ERR_ARGUMENT_OUT_OF_RANGE);
            return;
        }

        memset( msg, 0, TCP_SEND_MAX_LEN+1);

        if ( tcpConnect.receive_all(msg, argLen) >= 0 ) {
            console.printf("\r\nOK;%s\r\n\r\n",msg);
        } else
            cyntecPrintError(CYNTEC_CMD_ERR_CALL_FAIL);

    }	else {
        cyntecPrintError(CYNTEC_CMD_ERR_WRONG_NUMBER_OF_ARGUMENTS);
    }
}

static void cyn_wifi_tcp_connection_blocking()
{
    bool blocking = false;
    unsigned int timeout;

    uint8_t *arg;

    if (cyntecGetCommandTokenCnt() == 3) {

        arg = cyntecGetCommandArgument(0, NULL);
        if (arg[0] == '1')
            blocking = true;

        tcpConnect.set_blocking(blocking, 1500);
        cyntecPrintOk();

    } else if (cyntecGetCommandTokenCnt() == 4) {
        arg = cyntecGetCommandArgument(0, NULL);
        if (arg[0] == '1')
            blocking = true;

        arg = cyntecGetCommandArgument(1, NULL);
        timeout = (unsigned int)atoi((const char *)arg);

        tcpConnect.set_blocking(blocking, timeout);
        cyntecPrintOk();

    }	else {
        cyntecPrintError(CYNTEC_CMD_ERR_WRONG_NUMBER_OF_ARGUMENTS);
    }
}

static void cyn_wifi_tcp_connection_close()
{
    bool shutdown = true;
    uint8_t *arg;

    if (cyntecGetCommandTokenCnt() == 2) {
        if ( tcpConnect.close(shutdown) == 0 )
            cyntecPrintOk();
        else
            cyntecPrintError(CYNTEC_CMD_ERR_CALL_FAIL);

    }	else if (cyntecGetCommandTokenCnt() == 3) {
        arg = cyntecGetCommandArgument(0, NULL);
        if (arg[0] == '0')
            shutdown = false;

        if ( tcpConnect.close(shutdown) == 0 )
            cyntecPrintOk();
        else
            cyntecPrintError(CYNTEC_CMD_ERR_CALL_FAIL);

    }	else {
        cyntecPrintError(CYNTEC_CMD_ERR_WRONG_NUMBER_OF_ARGUMENTS);
    }
}
//static void cyn_wifi_tcp_connection_reset_address(){}
//static void cyn_wifi_tcp_connection_set_address(){}
//static void cyn_wifi_tcp_connection_address(){}
//static void cyn_wifi_tcp_connection_port(){}



/////////**** WIFI UDP Socket Implement ****//////////
static void cyn_wifi_udp_init()
{
    if (cyntecGetCommandTokenCnt() == 2) {
        if ( udpSocket.init() == 0 )
            cyntecPrintOk();
        else
            cyntecPrintError(CYNTEC_CMD_ERR_CALL_FAIL);

    } else {
        cyntecPrintError(CYNTEC_CMD_ERR_WRONG_NUMBER_OF_ARGUMENTS);
    }
}
static void cyn_wifi_udp_bind()
{
    int port = 1;
    if (cyntecGetCommandTokenCnt() == 2) {
        udpSocket.bind(port);
        cyntecPrintOk();
//			if ( udpSocket.bind(port) == 0 )
//				cyntecPrintOk();
//			else
//				cyntecPrintError(CYNTEC_CMD_ERR_CALL_FAIL);

    }	else if (cyntecGetCommandTokenCnt() == 3) {
        port = atoi((const char*)(cyntecGetCommandArgument(0, NULL)));

        if (port < 0) {
            cyntecPrintError(CYNTEC_CMD_ERR_ARGUMENT_OUT_OF_RANGE);
            return;
        }

        udpSocket.bind(port);
        cyntecPrintOk();
//			if ( udpSocket.bind(port) == 0 )
//				cyntecPrintOk();
//			else
//				cyntecPrintError(CYNTEC_CMD_ERR_CALL_FAIL);

    }	else {
        cyntecPrintError(CYNTEC_CMD_ERR_WRONG_NUMBER_OF_ARGUMENTS);
    }
}
static void cyn_wifi_udp_set_broadcasting()
{
    bool broadcast = true;
    uint8_t *arg;

    if (cyntecGetCommandTokenCnt() == 2) {
        if ( udpSocket.set_broadcasting(broadcast) == 0 )
            cyntecPrintOk();
        else
            cyntecPrintError(CYNTEC_CMD_ERR_CALL_FAIL);

    }	else if (cyntecGetCommandTokenCnt() == 3) {
        arg = cyntecGetCommandArgument(0, NULL);
        if (arg[0] == '0')
            broadcast = false;

        if ( udpSocket.set_broadcasting(broadcast) == 0 )
            cyntecPrintOk();
        else
            cyntecPrintError(CYNTEC_CMD_ERR_CALL_FAIL);

    }	else {
        cyntecPrintError(CYNTEC_CMD_ERR_WRONG_NUMBER_OF_ARGUMENTS);
    }
}
//static void cyn_wifi_udp_send_to()
//{
//    char msg[UDP_SEND_MAX_LEN+1];
//
//    uint8_t *arg;
//    int argLen = 0;
//
//    if (cyntecGetCommandTokenCnt() == 3) {
//        /* handle Message arg */
//        arg = cyntecGetCommandArgument(0, (uint8_t *)&argLen);
//
//        if (argLen > UDP_SEND_MAX_LEN || argLen < 0) {
//            cyntecPrintError(CYNTEC_CMD_ERR_ARGUMENT_OUT_OF_RANGE);
//            return;
//        }
//
//        memset( msg, 0, UDP_SEND_MAX_LEN+1);
//        memcpy( msg, arg, argLen);
//
//        if ( udpSocket.sendTo(cliEndpoint, msg, argLen) >= 0 )
//            cyntecPrintOk();
//        else
//            cyntecPrintError(CYNTEC_CMD_ERR_CALL_FAIL);
//
//    }	else {
//        cyntecPrintError(CYNTEC_CMD_ERR_WRONG_NUMBER_OF_ARGUMENTS);
//    }
//}
static void cyn_wifi_udp_send_to()
{
    char msg[UDP_SEND_MAX_LEN+1];
    int sendData;
    uint8_t * argAllBuf = cyntecGetCommandTotalBuffer();
    int sendLen = 0;  // Maximum 1400, need to be integer
    uint8_t sendLenCharNum = 0; // Maximum 4
    uint8_t *argLeng = cyntecGetCommandArgument(0,&sendLenCharNum);
    sendLen = cyntecAtoInt(argLeng,sendLenCharNum); 
    if (sendLen > UDP_SEND_MAX_LEN || sendLen < 0) {
        cyntecPrintError(CYNTEC_CMD_ERR_ARGUMENT_OUT_OF_RANGE);
        return;
    }
    if (cyntecGetCommandTokenCnt() <4 ) {
        cyntecPrintError(CYNTEC_CMD_ERR_WRONG_NUMBER_OF_ARGUMENTS);
        return;
    }
    if (cyntecGetCommandTokenCnt() >= 4) {

        /* handle Message arg */
#ifdef DELTA_WIFI_DEBUG
        console.printf("sendLen:%i,Buf:%s,Index:%i\r\n",sendLen,&argAllBuf[18+sendLenCharNum],cyntecGetTotalIndex());
#endif
        memset( msg, 0, UDP_SEND_MAX_LEN+1);
        for (int i=0; i<sendLen; i++)
            msg[i] = argAllBuf[18+sendLenCharNum+i];
#ifdef DELTA_WIFI_DEBUG
        console.printf("msg:%s\r\n",msg);
#endif
        sendData = udpSocket.sendTo(cliEndpoint, msg, sendLen);
        if ( sendData >= 0 ) {
            console.printf("\r\nOK;");
            console.printf("%i\r\n\r\n",sendData);
        } else
            cyntecPrintError(CYNTEC_CMD_ERR_CALL_FAIL);
    }
}

static void cyn_wifi_udp_received_from()
{
    char msg[UDP_SEND_MAX_LEN+1];
    int argLen = 0;

    if (cyntecGetCommandTokenCnt() == 3) {
        /* handle Message arg */
        argLen = atoi((const char *)cyntecGetCommandArgument(0, NULL));

        if (argLen > UDP_SEND_MAX_LEN || argLen < 0) {
            cyntecPrintError(CYNTEC_CMD_ERR_ARGUMENT_OUT_OF_RANGE);
            return;
        }

        memset( msg, 0, UDP_SEND_MAX_LEN+1);

        if ( udpSocket.receiveFrom(cliEndpoint, msg, argLen) >= 0 ) {
            console.printf("\r\nOK;%s\r\n\r\n",msg);
        } else
            cyntecPrintError(CYNTEC_CMD_ERR_CALL_FAIL);

    }	else {
        cyntecPrintError(CYNTEC_CMD_ERR_WRONG_NUMBER_OF_ARGUMENTS);
    }
}
static void cyn_wifi_udp_blocking()
{
    bool blocking = false;
    unsigned int timeout;

    uint8_t *arg;
    if (cyntecGetCommandTokenCnt() == 3) {

        arg = cyntecGetCommandArgument(0, NULL);
        if (arg[0] == '1')
            blocking = true;

        udpSocket.set_blocking(blocking, 1500);
        cyntecPrintOk();

    } else if (cyntecGetCommandTokenCnt() == 4) {
        arg = cyntecGetCommandArgument(0, NULL);
        if (arg[0] == '1')
            blocking = true;

        arg = cyntecGetCommandArgument(1, NULL);
        timeout = (unsigned int)atoi((const char *)arg);

        udpSocket.set_blocking(blocking, timeout);
        cyntecPrintOk();

    }	else {
        cyntecPrintError(CYNTEC_CMD_ERR_WRONG_NUMBER_OF_ARGUMENTS);
    }
}
static void cyn_wifi_udp_close()
{
    bool shutdown = true;
    uint8_t *arg;

    if (cyntecGetCommandTokenCnt() == 2) {
        if ( udpSocket.close(shutdown) == 0 )
            cyntecPrintOk();
        else
            cyntecPrintError(CYNTEC_CMD_ERR_CALL_FAIL);

    }	else if (cyntecGetCommandTokenCnt() == 3) {
        arg = cyntecGetCommandArgument(0, NULL);
        if (arg[0] == '0')
            shutdown = false;

        if ( udpSocket.close(shutdown) == 0 )
            cyntecPrintOk();
        else
            cyntecPrintError(CYNTEC_CMD_ERR_CALL_FAIL);

    }	else {
        cyntecPrintError(CYNTEC_CMD_ERR_WRONG_NUMBER_OF_ARGUMENTS);
    }
}



/////////**** WIFI UDP Endpoint Implement ****//////////
static void cyn_wifi_udp_endpoint_reset_address()
{
    if (cyntecGetCommandTokenCnt() == 2) {
        cliEndpoint.reset_address();
        cyntecPrintOk();
    } else {
        cyntecPrintError(CYNTEC_CMD_ERR_WRONG_NUMBER_OF_ARGUMENTS);
    }
}
static void cyn_wifi_udp_endpoint_address()
{
    uint8_t argHost[STATIC_IP_MAX_LEN];
    int port;

    uint8_t *arg;
    uint8_t argLen;

    if ( cyntecGetCommandTokenCnt() == 4 ) {
        /* handle Host addr */
        arg = cyntecGetCommandArgument(0, &argLen);

        if ( cyntecIsValidIP(arg,argLen) != 0) {
            cyntecPrintError(CYNTEC_CMD_ERR_ARGUMENT_OUT_OF_RANGE);
            return;
        }

        memset( argHost, 0, STATIC_IP_MAX_LEN);
        memcpy( argHost, arg, argLen);

        /* Handle Port */
        port = atoi((const char *)(cyntecGetCommandArgument(1, NULL)));
        if (port < 0) {
            cyntecPrintError(CYNTEC_CMD_ERR_ARGUMENT_OUT_OF_RANGE);
            return;
        }

        if ( cliEndpoint.set_address((const char *)argHost, (const int)port) == 0)
            cyntecPrintOk();
        else
            cyntecPrintError(CYNTEC_CMD_ERR_CALL_FAIL);

    } else {
        console.printf("\r\nOK;%s\r\n\r\n",cliEndpoint.get_address());
    }
}
//static void cyn_wifi_udp_endpoint_get_address(){}
static void cyn_wifi_udp_endpoint_port()   // 2015/1/20: Lester add
{
    console.printf("\r\nOK;%d\r\n\r\n",cliEndpoint.get_port());
}

static void cyn_wifi_enint()
{
    wifiDataInt = true;
    cyntecPrintOk();
}

static void cyn_wifi_disint()
{
    wifiDataInt = false;
    cyntecPrintOk();
}


int wifi_receive_UDP_callback(uint8_t *data, int size, char* ipaddr, int port)
{
    // Do nothing until flag on
    if(wifiDataInt) {
        wifiWriteInt = !wifiWriteInt;
        wifiWriteInt = !wifiWriteInt;
#ifdef DELTA_WIFI_DEBUG
        console.printf("wifi_receive_UDP_callback:");
#endif
        console.printf("rUDP;%s,%i,%s,%d;\r\n",data, size, ipaddr, port);
    }
}
// Gill note: port number may different from original

int wifi_receive_TCP_callback(uint8_t *data, int size, char* ipaddr, int port)
{
    // Do nothing until flag on
    if(wifiDataInt) {
        wifiWriteInt = !wifiWriteInt;
        wifiWriteInt = !wifiWriteInt;
#ifdef DELTA_WIFI_DEBUG
        console.printf("wifi_receive_TCP_callback\r\n");
#endif
        console.printf("rTCP;%s,%i,%s,%d;\r\n",data, size, ipaddr, port);
    }
}
// Gill note: port number may different from original

CyntecCommandEntry wifiCommandSets[] = {
#if SIMPLE_CMD_NAME
    /////////**** WIFI Device ****//////////
    {"DSLP", cyn_wifi_device_sleep, NULL, "Set WIFI module to sleep mode"},
    {"DCOE", cyn_wifi_device_coex, NULL, "Set/Get device switch for WIFI module"},
    {"DNWK", cyn_wifi_device_network, NULL, " <SSID> <PASSWORD> <PRIORITY> Set SSID and PASSWORD for WIFI module"},
    //gill
    {"DRSV",cyn_wifi_read_sp_version, NULL, "Set network connection in priority"},
    {"DASN", cyn_wifi_device_ap_scan, NULL, "Scan for available access point on all channels"},
    {"DSAP",cyn_wifi_set_ap, NULL, " <SSID> <PASSWORD> <SECURITY> <CHANNEL>Set Access Point in given configuration"},
    {"DME4",cyn_wifi_mem_erase4k, NULL, " <MEMADDR> Erase a 4KB sector of embedded flash"},
    {"DMRD",cyn_wifi_mem_read, NULL, " <MEMADDR> <LEN> Read data from embeeded flash"},
    {"DMWT",cyn_wifi_mem_write, NULL, " <MEMADDR> <DATA> Write data into embeeded flash"},
    {"DEDI",cyn_wifi_enint, NULL, " Enable Wifi data interrupt, print data whenever received."},
    {"DDDI",cyn_wifi_disint, NULL, " Disable Wifi interrupt."},

    /////////**** WIFI Ethernet ****//////////
    {"EINT", cyn_wifi_ethernet_init, NULL, " <STATIC IP> Initialize the interface to use DHCP"},
    {"ECON", cyn_wifi_ethernet_connect, NULL, "<TIMEOUT MS> Bring up the WiFi connection"},
    {"EDCN", cyn_wifi_ethernet_disconnect, NULL, "Bring the interface down"},
    {"EMAC", cyn_wifi_ethernet_mac, NULL, "Get MAC addr of Ehternet Interface"},
    {"EGIP", cyn_wifi_ethernet_ip, NULL, "Get IP addr of Ehternet Interface"},

    /////////**** WIFI TCP Socket Server ****//////////
    {"TSBD", cyn_wifi_tcp_server_bind, NULL, " <PORT> Bind a socket to a port"},
    {"TSLN", cyn_wifi_tcp_server_listen, NULL, "Start listening for incomming connections"},
    {"TSAC", cyn_wifi_tcp_server_accept, NULL, "Accept a new connection"},
    {"TSBL", cyn_wifi_tcp_server_blocking, NULL, " <SETTING> <TIMEOUT MS> Set blocking mode and timeout"},
    {"TSCL", cyn_wifi_tcp_server_close, NULL, " <SHUTDOWN> Close the socket"},

    /////////**** WIFI TCP Socket Connection ****//////////
    {"TCCN", cyn_wifi_tcp_connection_connect, NULL, " <IPADDR> <PORT> Connects TCP socket to the server"},
    {"TCIC", cyn_wifi_tcp_connection_is_connected, NULL, "Check if the socket is connected"},
    {"TCSE", cyn_wifi_tcp_connection_send, NULL, " <DATA> Send data to the remote host"},
    {"TCSA", cyn_wifi_tcp_connection_send_all, NULL, " <DATA> Send all the data to the remote host"},
    {"TCRC", cyn_wifi_tcp_connection_receive, NULL, " <DATALEN> Receive data from the remote host"},
    {"TCRA", cyn_wifi_tcp_connection_receive_all, NULL, "<DATALEN> Receive all the data from the remote host"},
    {"TCBL ", cyn_wifi_tcp_connection_blocking, NULL, "<SETTING> <TIMEOUT MS> Set blocking mode and timeout"},
    {"TCCL", cyn_wifi_tcp_connection_close, NULL, "<SHUTDOWN> Close the connection"},

    /////////**** WIFI UDP Socket ****//////////
    {"UCIN", cyn_wifi_udp_init, NULL, "Init UDP Client Socket"},
    {"UCBI", cyn_wifi_udp_bind, NULL, " <PORT> Bind UDP Server Socket to a port"},
    {"UCSB", cyn_wifi_udp_set_broadcasting, NULL, " <IS_BROADCAST> Set socket in broadcasting"},
    {"UCSE", cyn_wifi_udp_send_to, NULL, " <DATA> Send a packet to a remote endpoint"},
    {"UCRC", cyn_wifi_udp_received_from, NULL, " <DATALEN> Receive a packet from a remote endpont"},
    {"UCBL", cyn_wifi_udp_blocking, NULL, " <DATALEN> Set blocking mode and timeout"},
    {"UCCL", cyn_wifi_udp_close, NULL, " <SHUTDOWN> Close the socket"},

    /////////**** WIFI UDP Endpoint ****//////////
    {"UERS", cyn_wifi_udp_endpoint_reset_address, NULL, "Reset the address of this endpoint"},
    {"UEAD", cyn_wifi_udp_endpoint_address, NULL, " <IPADDR> <PORT> Set/Get the address of this endpoint"},
    {"UEPT", cyn_wifi_udp_endpoint_port, NULL, "Get the port of this endpoint"},

    {NULL, NULL, NULL, NULL},
#else
    /////////**** WIFI Device ****//////////
    {"device_sleep", cyn_wifi_device_sleep, NULL, "Set WIFI module to sleep mode"},
    {"device_coex", cyn_wifi_device_coex, NULL, "Set/Get device switch for WIFI module"},
    {"device_network", cyn_wifi_device_network, NULL, " <SSID> <PASSWORD> <PRIORITY> Set SSID and PASSWORD for WIFI module"},
    //gill
    {"device_read_version",cyn_wifi_read_sp_version, NULL, "Read WIFI Service Pack version"},
    {"device_apscan", cyn_wifi_device_ap_scan, NULL, "Scan for available access point on all channels."},
    {"device_setap",cyn_wifi_set_ap, NULL, " <SSID> <PASSWORD> <SECURITY> <CHANNEL>Set Access Point in given configuration"},
    {"device_mem_erase4k",cyn_wifi_mem_erase4k, NULL, " <MEMADDR> Erase a 4KB sector of embedded flash"},
    {"device_mem_read",cyn_wifi_mem_read, NULL, " <MEMADDR> <LEN> Read data from embeeded flash"},
    {"device_mem_write",cyn_wifi_mem_write, NULL, " <MEMADDR> <DATA> Write data into embeeded flash"},
    {"device_enint",cyn_wifi_enint, NULL, " Enable Wifi data interrupt, print data whenever received."},
    {"device_disint",cyn_wifi_disint, NULL, " Disable Wifi interrupt."},


    /////////**** WIFI Ethernet ****//////////
    {"ethernet_init", cyn_wifi_ethernet_init, NULL, " <STATIC IP> Initialize the interface to use DHCP"},
    {"ethernet_connect", cyn_wifi_ethernet_connect, NULL, "<TIMEOUT MS> Bring up the WiFi connection"},
    {"ethernet_disconnect", cyn_wifi_ethernet_disconnect, NULL, "Bring the interface down"},
    {"ethernet_mac", cyn_wifi_ethernet_mac, NULL, "Get MAC addr of Ethernet Interface"},
    {"ethernet_ip", cyn_wifi_ethernet_ip, NULL, "Get IP addr of Ehternet Interface"},
    /* not implement yet... */
    ////{"ethernet_gateway", cyn_wifi_ethernet_gateway, NULL, "........."},
    ////{"ethernet_mask", cyn_wifi_ethernet_mask, NULL, "........."},
    /* not implement yet... */

    /////////**** WIFI TCP Socket Server ****//////////
    {"tcp_server_bind", cyn_wifi_tcp_server_bind, NULL, " <PORT> Bind a socket to a port"},
    {"tcp_server_listen", cyn_wifi_tcp_server_listen, NULL, "Start listening for incomming connections"},
    {"tcp_server_accept", cyn_wifi_tcp_server_accept, NULL, "Accept a new connection"},
    {"tcp_server_blocking", cyn_wifi_tcp_server_blocking, NULL, " <SETTING> <TIMEOUT MS> Set blocking mode and timeout"},
    {"tcp_server_close", cyn_wifi_tcp_server_close, NULL, " <SHUTDOWN> Close the socket"},

    /////////**** WIFI TCP Socket Connection ****//////////
    {"tcp_connection_connect", cyn_wifi_tcp_connection_connect, NULL, " <IPADDR> <PORT> Connects TCP socket to the server"},
    {"tcp_connection_is_connect", cyn_wifi_tcp_connection_is_connected, NULL, "Check if the socket is connected"},
    {"tcp_connection_send", cyn_wifi_tcp_connection_send, NULL, " <DATA> Send data to the remote host"},
    {"tcp_connection_send_all", cyn_wifi_tcp_connection_send_all, NULL, " <DATA> Send all the data to the remote host"},
    {"tcp_connection_receive", cyn_wifi_tcp_connection_receive, NULL, " <DATALEN> Receive data from the remote host"},
    {"tcp_connection_receive_all", cyn_wifi_tcp_connection_receive_all, NULL, "<DATALEN> Receive all the data from the remote host"},
    {"tcp_connection_blocking", cyn_wifi_tcp_connection_blocking, NULL, "<SETTING> <TIMEOUT MS> Set blocking mode and timeout"},
    {"tcp_connection_close", cyn_wifi_tcp_connection_close, NULL, "<SHUTDOWN> Close the connection"},
//	{"tcp_connection_reset_address", cyn_wifi_tcp_connection_reset_address, NULL, "........."},
//	{"tcp_connection_address", cyn_wifi_tcp_connection_address, NULL, "........."},
//	{"tcp_connection_port", cyn_wifi_tcp_connection_port, NULL, "........."},

    /////////**** WIFI UDP Socket ****//////////
    {"udp_init", cyn_wifi_udp_init, NULL, "Init UDP Client Socket"},
    {"udp_bind", cyn_wifi_udp_bind, NULL, " <PORT> Bind UDP Server Socket to a port"},
    {"udp_set_broadcasting", cyn_wifi_udp_set_broadcasting, NULL, " <IS_BROADCAST> Set socket in broadcasting"},
    {"udp_send_to", cyn_wifi_udp_send_to, NULL, " <DATA> Send a packet to a remote endpoint"},
    {"udp_received_from", cyn_wifi_udp_received_from, NULL, " <DATALEN> Receive a packet from a remote endpont"},
    {"udp_blocking", cyn_wifi_udp_blocking, NULL, " <DATALEN> Set blocking mode and timeout"},
    {"udp_close", cyn_wifi_udp_close, NULL, " <SHUTDOWN> Close the socket"},

    /////////**** WIFI UDP Endpoint ****//////////
    {"udp_endpoint_reset", cyn_wifi_udp_endpoint_reset_address, NULL, "Reset the address of this endpoint"},
    {"udp_endpoint_address", cyn_wifi_udp_endpoint_address, NULL, " <IPADDR> <PORT> Set/Get the address of this endpoint"},
    {"udp_endpoint_port", cyn_wifi_udp_endpoint_port, NULL, "Get the port of this endpoint"},

    {NULL, NULL, NULL, NULL},
#endif
};



