/* My Digital Signal Controller library
 * Copyright (c) 2019, Gastón H. SALAZAR-SILVA
 * SPDX-License-Identifier: Apache-2.0
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * 
 * You may obtain a copy of the License at http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied.
 * 
 * See the License for the specific language governing permissions and limitations under the License.
 */

#ifndef MYDSC_RING_BUFFER_H
#define MYDSC_RING_BUFFER_H

#include <stdint.h>

#ifdef __cplusplus
extern "C" {
#endif

  /**
   *  @author Gastón H. SALAZAR-SILVA
   *  @date   10/29/2019
   *
   *  @brief  This module.
   */
  typedef struct mydsc_ring_buffer {
    uint32_t max_position;     /**< It is used as a mask for modular arithmetic. Its value should be 2^n - 1. */
    uint32_t index;     /**< It is the actual position of the last element in the circular buffer. */
    float   *samples;   /**< The circular buffer is implemented as an array. this array is initializated with mydsc_ring_bufferinit. */
  } mydsc_ring_buffer_t;

  /**
   *    @brief  Initialize a ring buffer.
   *    @param  prb a pointer to a ring buffer.
   *    @param  size the size of the samples array. Its value should be 2^n.
   *    @return 0 on success, -1 on error.
   *
   *    @author Gastón H. SALAZAR-SILVA
   *    @date   10/29/2019
   */
  int mydsc_ring_buffer_init(mydsc_ring_buffer_t *prb, uint32_t size);

  /**
   *    @brief  Push a value into a ring buffer.
   *    @param  ring    a pointer to a ring buffer.
   *    @param  input   a float to be pushed into the ring buffer.
   *
   *    @author Gastón H. SALAZAR-SILVA
   *    @date   10/29/2019
   */
  void mydsc_ring_buffer_push(mydsc_ring_buffer_t *ring, float input);

  /**
   *    @brief  Return a pointer to a especific sample.
   *    @param  ring    a pointer to a ring buffer.
   *    @param  pos     an integer to indicate which sample to point. A negative value is expexted to past samples.
   *
   *    @author Gastón H. SALAZAR-SILVA
   *    @date   10/29/2019
   */
  static inline float* mydsc_ring_buffer_sample(mydsc_ring_buffer_t *ring, int32_t pos)
  {
    uint32_t  n;
    
    n = ring->index + pos;
    n &= ring->max_position;
    return &ring->samples[n];
  }

#ifdef __cplusplus
}
#endif

#define $(rb,n) *mydsc_ring_buffersample( &rb, n)
#define $$(prb,n) *mydsc_ring_buffersample( prb, n)

#endif // MYDSC_RING_BUFFER_H
