/*  xbeeutils.cpp
 *  Guillaume Hivert, Marc-Olivier Lavoie, Gabriel Gibeau-Sanchez
 *  Université de Sherbrooke, S5 project
 *
 *  Provides high-level functions to initialize and control xbee communication.
 *  Inspired by https://os.mbed.com/teams/Digi-International-Inc/code/XBeeLib/wiki/Discovering-nodes-in-the-network
*/

#include "xbeeutils.h"

#define MAX_NODES                   10

static XBeeZB *xbee = NULL;
static RemoteXBeeZB remote_nodes_in_network[MAX_NODES];
static unsigned int remote_nodes_count = 0;
static Serial pc_(USBTX,USBRX);
static Serial * pc=&pc_;

static void handle_discover(const RemoteXBeeZB& remote, char const * const node_id);
static void handle_receive_data(const RemoteXBeeZB& remote, bool broadcast, const uint8_t *const data, uint16_t len);
static int isNewRemote(uint16_t addr_16);

void xbee_init() {
    xbee = new XBeeZB(RADIO_TX, RADIO_RX, RADIO_RESET, NC, NC, 9600);
    RadioStatus radioStatus = xbee->init();
    radioStatus = xbee->set_panid((uint64_t) 0xABCD);                   // Don't forget to do this with config!
    xbee->register_receive_cb(handle_receive_data);
    
    // Wait for xbee to be ready
    while (!xbee->is_joined()) {
        wait_ms(1000);
        (*pc).printf(".");
    }
    (*pc).printf("OK\r\n");
}

/* This function is called when a node is discovered. It adds it to the known nodes list if it was a not previously discovered.
 *
 * remote: Node that was discovered
 * node_id: ID (NI) of the discovered node. Can be empty string "", so avoid using it.
*/
static void handle_discover(const RemoteXBeeZB& remote, char const * const node_id) {
    uint16_t addr_16 = remote.get_addr16();

    (*pc).printf("Found device '%s' [%04X]\r\n", node_id, addr_16);

    if (remote_nodes_count < MAX_NODES) {
        if(isNewRemote(addr_16)) {
            (*pc).printf("in if\n");
            remote_nodes_in_network[remote_nodes_count] = remote;
            remote_nodes_count++;
            // get configs
        }
    } else {
        (*pc).printf("Found more nodes than maximum configured (%d)\r\n", MAX_NODES);
    }
}

/* Checks if a discovered node is already known
 *
 * addr_16: 16-bit network address of discovered node.
 * Returns 1 if the node is a new node, 0 if it was already known.
*/
static int isNewRemote(uint16_t addr_16) {
    int i;
    for (i=0; i < remote_nodes_count; i++) {
        if (addr_16 == remote_nodes_in_network[i].get_addr16())
        return 0;
    }
    return 1;
}

/* This function is called when data is received from other nodes.
 *
 * remote: sender of the data.
 * broadcast: true if the message received was broadcasted, false if it was specifically sent to this module.
 * data: message received.
 * len: length of the message received.
*/
static void handle_receive_data(const RemoteXBeeZB& remote, bool broadcast, const uint8_t *const data, uint16_t len) {
    (*pc).printf("Received message from remote %X\r\n", remote.get_addr16());
    (*pc).printf("%s\r\n", data);
    
    char *s = (char*) data;
    int c = 0;
    char **arr = NULL;
    
    c = split(s, ';', &arr);
    (*pc).printf("found %d tokens.\n", c);
    
    Config config;
    
    config.temperatureMax = atof(arr[0]);
    config.temperatureMin = atof(arr[1]);
    config.phMax = atof(arr[2]);
    config.phMin = atof(arr[3]);
    config.ec = atof(arr[4]);
    
    (*pc).printf("Tmax: %f\nTmin: %f\nPHmax: %f\nPHmin: %f\nEC: %f\n", config.temperatureMax, config.temperatureMin, config.phMax, config.phMin, config.ec);
    
    setConfig(config);
}


/* Tries to discover new nodes in the network. 
 *
 * If a node answers, handle_discover() will be called with information about the answering node.
*/
void discover() {
    (*pc).printf("\r\nStarting Node Discovery\r\n\r\n");
    for(int i = 0; i < 3; i++) {
        xbee->start_node_discovery();
        
        do {
            xbee->process_rx_frames();
            wait_ms(10);
        } while(xbee->is_node_discovery_in_progress());
    }
    
    (*pc).printf("\r\nNode Discovery Finished\r\n\r\n");
    (*pc).printf("remote nodes count %i\n", remote_nodes_count);
}

/* Sends a message to a remote ZigBee node.
 *
 * message: message to send.
 * length: length of the message to send.
 * target_addr_16: 16-bit network address of the target node.
*/
void send_data_to_coordinator(char *message, int length) {    

    const TxStatus txStatus = xbee->send_data_to_coordinator((const uint8_t *)message, length);
    if (txStatus != TxStatusSuccess) {
        (*pc).printf("An error happened while sending data to remote coordinator %d\r\n", txStatus);
    }
}

RemoteXBeeZB get_remote_node() {
    return xbee->get_remote_node_by_id("Router");
}

void xbee_send(const RemoteXBeeZB& xbzb, char * msg, int length){
    xbee->send_data(xbzb, (const uint8_t *)msg, length);
}

/* Sends a message to all remote ZigBee nodes.
 *
 * message: message to send.
 * length: length of the message to send.
*/
// Change to xbee->send_data_broadcast
void xbee_broadcast(char *message, int length) {
    
    const TxStatus txStatus = xbee->send_data_broadcast((const uint8_t *) message, length);
    if (txStatus != TxStatusSuccess) {
        //(*pc).printf("An error happened while sending data to remote %X, status: %d\r\n", remote_nodes_in_network[i].get_addr16(), txStatus);
        (*pc).printf("An error happened while sending data to remote X, status: %d\r\n", txStatus);
    }
}
/* Gives processor time to process frames held in the receive buffer
 * 
 * Depending on the the type of frame received, different functions will be called.
 * See handle_discover and handle_receive_data
*/
void process_rx_frames() {
    xbee->process_rx_frames();
}

int split (char *str, char c, char ***arr)
{
    int count = 1;
    int token_len = 1;
    int i = 0;
    char *p;
    char *t;

    p = str;
    while (*p != '\0')
    {
        if (*p == c)
            count++;
        p++;
    }

    *arr = (char**) malloc(sizeof(char*) * count);
    if (*arr == NULL)
        exit(1);

    p = str;
    while (*p != '\0')
    {
        if (*p == c)
        {
            (*arr)[i] = (char*) malloc( sizeof(char) * token_len );
            if ((*arr)[i] == NULL)
                exit(1);

            token_len = 0;
            i++;
        }
        p++;
        token_len++;
    }
    (*arr)[i] = (char*) malloc( sizeof(char) * token_len );
    if ((*arr)[i] == NULL)
        exit(1);

    i = 0;
    p = str;
    t = ((*arr)[i]);
    while (*p != '\0')
    {
        if (*p != c && *p != '\0')
        {
            *t = *p;
            t++;
        }
        else
        {
            *t = '\0';
            i++;
            t = ((*arr)[i]);
        }
        p++;
    }

    return count;
}


