/* Copyright (c) 2012 Christian Buntebarth, MIT License
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy of this software 
 * and associated documentation files (the "Software"), to deal in the Software without restriction, 
 * including without limitation the rights to use, copy, modify, merge, publish, distribute, 
 * sublicense, and/or sell copies of the Software, and to permit persons to whom the Software is 
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all copies or 
 * substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED, INCLUDING 
 * BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND 
 * NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, 
 * DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, 
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 */
 
#ifndef MBED_PULSECOUNTER_H
#define MBED_PULSECOUNTER_H

#include "mbed.h"

const int _pulseCountArrayLength = 20;

/** Counts pulses on pin29 coming in with a frequency up to 48 MHz. The counter is read (and reset to 0) by rising edge (trigger event) on pin30.
 *  These events can be triggered at a rate up to 9kHz. The counter results are stored in a "circular array" of 20 elements.
 *  ("circular array" = array pointer increments each time a new value is stored into the array; if pointer reaches last element, it is redirected to first element)
 *   */
class PulseCounter
{
    public:

    /** Initializer for PulseCounter instance
     */
    void init();
 
    /** Start counter
     */
    void start();

    /** Stop counter
     */
    void stop();
    
    /** Read pulse count results into external array. Do not call this method before you called method stop() - otherwise it will return zeros.
     * @param counterArray          = Pointer to array in which the pulse count results should be read into. 
     * @param counterArrayLength    = Size of counterArray. Should be 20 or less. If more than 20, remaining elements will be filled with zeros. 
     * @code    #include "mbed.h"
                #include "PwmOscillator.h"
                #include "PulseCounter.h"

                Serial usbPC(USBTX, USBRX);                 // sends log messages via USB to PC terminal
                PwmOscillator pulseGenerator;               // generates pulses to be count (48 MHz) on pin p22. Connect this pin to p29 (counter input).
                PulseCounter pulseCounter;                  // counts the pulses on pin p29 between trigger events (= rising edges) on pin p30.

                int main() 
                {
                    usbPC.printf("---> start <---\n");
                    pulseGenerator.initWithFrequency(48000000);
                    pulseCounter.init();
                    pulseGenerator.start();
    
                    pulseCounter.start();   
                    wait(1);                                // waiting 1 second for trigger events (pin30: rising edge). Count pulses between trigger events.
                    pulseCounter.stop();
                    
                    pulseGenerator.stop();
    
                    // read & print pulseCounter results
                    int pulseCounterResults[20];
                    pulseCounter.counterArray(&pulseCounterResults[0],sizeof pulseCounterResults / sizeof (int));
    
                    for(int i = 0; i < (sizeof pulseCounterResults / sizeof (int)); i++) {
                        usbPC.printf("counter of trigger event %i = %i\n",i,pulseCounterResults[i]);
                    }
                    //
    
                    usbPC.printf("   finished.\n");
                }
       * @endcode  */
    void counterArray(uint32_t* counterArray, int counterArrayLength);
            
private:
    
    static void _triggerInterrupt();
    static PulseCounter *instance;
    bool _hasFinishedCounting;
    
    volatile int _triggerCounter;
    volatile uint32_t *_arrayPointer;
    volatile uint32_t _pulseCountArray[_pulseCountArrayLength + 1];
};

#endif
