/* mbed Microcontroller Library
 * Copyright (c) 2006-2015 ARM Limited
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
#ifndef MBED_BINDER_H
#define MBED_BINDER_H

#include "FunctionPointer.h"

#include <string.h>
#include <stdint.h>

namespace mbed {

// Binding with memory management by user
template <typename F, typename B1=void, typename B2=void, typename B3=void, typename B4=void>
class Binder;

/** Binding with memory ownership on the user
 */
template <typename R, typename B1, typename B2, typename B3, typename B4>
class Binder<R(), B1, B2, B3, B4> {
public:
    /** Create an empty Binder
     */
    Binder() {}

    /** Create a Binder, binding arguments to function
     */
    Binder(FuncPtr<R(B1, B2, B3, B4)> func, B1 b1, B2 b2, B3 b3, B4 b4) {
        attach(func, b1, b2, b3, b4);
    }

    /** Create a Binder, binding arguments to a method
     */
    template <typename T, typename M>
    Binder(T *object, M method, B1 b1, B2 b2, B3 b3, B4 b4) {
        attach(object, method, b1, b2, b3, b4);
    }

    /** Bind arguments to a function
     */
    void attach(FuncPtr<R(B1, B2, B3, B4)> func, B1 b1, B2 b2, B3 b3, B4 b4) {
        _func = func;
        _b1 = b1;
        _b2 = b2;
        _b3 = b3;
        _b4 = b4;
    }

    /** Bind arguments to a method
     */
    template <typename T, typename M>
    void attach(T *object, M method, B1 b1, B2 b2, B3 b3, B4 b4) {
        _func = FuncPtr<R(B1, B2, B3, B4)>(object, method);
        _b1 = b1;
        _b2 = b2;
        _b3 = b3;
        _b4 = b4;
    }

    /** Call the bound function
     */
    R call() {
        return _func(_b1, _b2, _b3, _b4);
    }

#ifdef MBED_OPERATORS
    R operator()() {
        return _func(_b1, _b2, _b3, _b4);
    }
#endif
private:
    FuncPtr<R(B1, B2, B3, B4)> _func;
    B1 _b1;
    B2 _b2;
    B3 _b3;
    B4 _b4;
};


/** Binding with memory ownership on the user
 */
template <typename R, typename A1, typename B1, typename B2, typename B3>
class Binder<R(A1), B1, B2, B3> {
public:
    /** Create an empty Binder
     */
    Binder() {}

    /** Create a Binder, binding arguments to function
     */
    Binder(FuncPtr<R(B1, B2, B3, A1)> func, B1 b1, B2 b2, B3 b3) {
        attach(func, b1, b2, b3);
    }

    /** Create a Binder, binding arguments to a method
     */
    template <typename T, typename M>
    Binder(T *object, M method, B1 b1, B2 b2, B3 b3) {
        attach(object, method, b1, b2, b3);
    }

    /** Bind arguments to a function
     */
    void attach(FuncPtr<R(B1, B2, B3, A1)> func, B1 b1, B2 b2, B3 b3) {
        _func = func;
        _b1 = b1;
        _b2 = b2;
        _b3 = b3;
    }

    /** Bind arguments to a method
     */
    template <typename T, typename M>
    void attach(T *object, M method, B1 b1, B2 b2, B3 b3) {
        _func = FuncPtr<R(B1, B2, B3, A1)>(object, method);
        _b1 = b1;
        _b2 = b2;
        _b3 = b3;
    }

    /** Call the bound function
     */
    R call(A1 a1) {
        return _func(_b1, _b2, _b3, a1);
    }

#ifdef MBED_OPERATORS
    R operator()(A1 a1) {
        return _func(_b1, _b2, _b3, a1);
    }
#endif
private:
    FuncPtr<R(B1, B2, B3, A1)> _func;
    B1 _b1;
    B2 _b2;
    B3 _b3;
};

/** Binding with memory ownership on the user
 */
template <typename R, typename B1, typename B2, typename B3>
class Binder<R(), B1, B2, B3> {
public:
    /** Create an empty Binder
     */
    Binder() {}

    /** Create a Binder, binding arguments to function
     */
    Binder(FuncPtr<R(B1, B2, B3)> func, B1 b1, B2 b2, B3 b3) {
        attach(func, b1, b2, b3);
    }

    /** Create a Binder, binding arguments to a method
     */
    template <typename T, typename M>
    Binder(T *object, M method, B1 b1, B2 b2, B3 b3) {
        attach(object, method, b1, b2, b3);
    }

    /** Bind arguments to a function
     */
    void attach(FuncPtr<R(B1, B2, B3)> func, B1 b1, B2 b2, B3 b3) {
        _func = func;
        _b1 = b1;
        _b2 = b2;
        _b3 = b3;
    }

    /** Bind arguments to a method
     */
    template <typename T, typename M>
    void attach(T *object, M method, B1 b1, B2 b2, B3 b3) {
        _func = FuncPtr<R(B1, B2, B3)>(object, method);
        _b1 = b1;
        _b2 = b2;
        _b3 = b3;
    }

    /** Call the bound function
     */
    R call() {
        return _func(_b1, _b2, _b3);
    }

#ifdef MBED_OPERATORS
    R operator()() {
        return _func(_b1, _b2, _b3);
    }
#endif
private:
    FuncPtr<R(B1, B2, B3)> _func;
    B1 _b1;
    B2 _b2;
    B3 _b3;
};

/** Binding with memory ownership on the user
 */
template <typename R, typename A1, typename A2, typename B1, typename B2>
class Binder<R(A1, A2), B1, B2> {
public:
    /** Create an empty Binder
     */
    Binder() {}

    /** Create a Binder, binding arguments to function
     */
    Binder(FuncPtr<R(B1, B2, A1, A2)> func, B1 b1, B2 b2) {
        attach(func, b1, b2);
    }

    /** Create a Binder, binding arguments to a method
     */
    template <typename T, typename M>
    Binder(T *object, M method, B1 b1, B2 b2) {
        attach(object, method, b1, b2);
    }

    /** Bind arguments to a function
     */
    void attach(FuncPtr<R(B1, B2, A1, A2)> func, B1 b1, B2 b2) {
        _func = func;
        _b1 = b1;
        _b2 = b2;
    }

    /** Bind arguments to a method
     */
    template <typename T, typename M>
    void attach(T *object, M method, B1 b1, B2 b2) {
        _func = FuncPtr<R(B1, B2, A1, A2)>(object, method);
        _b1 = b1;
        _b2 = b2;
    }

    /** Call the bound function
     */
    R call(A1 a1, A2 a2) {
        return _func(_b1, _b2, a1, a2);
    }

#ifdef MBED_OPERATORS
    R operator()(A1 a1, A2 a2) {
        return _func(_b1, _b2, a1, a2);
    }
#endif
private:
    FuncPtr<R(B1, B2, A1, A2)> _func;
    B1 _b1;
    B2 _b2;
};

/** Binding with memory ownership on the user
 */
template <typename R, typename A1, typename B1, typename B2>
class Binder<R(A1), B1, B2> {
public:
    /** Create an empty Binder
     */
    Binder() {}

    /** Create a Binder, binding arguments to function
     */
    Binder(FuncPtr<R(B1, B2, A1)> func, B1 b1, B2 b2) {
        attach(func, b1, b2);
    }

    /** Create a Binder, binding arguments to a method
     */
    template <typename T, typename M>
    Binder(T *object, M method, B1 b1, B2 b2) {
        attach(object, method, b1, b2);
    }

    /** Bind arguments to a function
     */
    void attach(FuncPtr<R(B1, B2, A1)> func, B1 b1, B2 b2) {
        _func = func;
        _b1 = b1;
        _b2 = b2;
    }

    /** Bind arguments to a method
     */
    template <typename T, typename M>
    void attach(T *object, M method, B1 b1, B2 b2) {
        _func = FuncPtr<R(B1, B2, A1)>(object, method);
        _b1 = b1;
        _b2 = b2;
    }

    /** Call the bound function
     */
    R call(A1 a1) {
        return _func(_b1, _b2, a1);
    }

#ifdef MBED_OPERATORS
    R operator()(A1 a1) {
        return _func(_b1, _b2, a1);
    }
#endif
private:
    FuncPtr<R(B1, B2, A1)> _func;
    B1 _b1;
    B2 _b2;
};

/** Binding with memory ownership on the user
 */
template <typename R, typename B1, typename B2>
class Binder<R(), B1, B2> {
public:
    /** Create an empty Binder
     */
    Binder() {}

    /** Create a Binder, binding arguments to function
     */
    Binder(FuncPtr<R(B1, B2)> func, B1 b1, B2 b2) {
        attach(func, b1, b2);
    }

    /** Create a Binder, binding arguments to a method
     */
    template <typename T, typename M>
    Binder(T *object, M method, B1 b1, B2 b2) {
        attach(object, method, b1, b2);
    }

    /** Bind arguments to a function
     */
    void attach(FuncPtr<R(B1, B2)> func, B1 b1, B2 b2) {
        _func = func;
        _b1 = b1;
        _b2 = b2;
    }

    /** Bind arguments to a method
     */
    template <typename T, typename M>
    void attach(T *object, M method, B1 b1, B2 b2) {
        _func = FuncPtr<R(B1, B2)>(object, method);
        _b1 = b1;
        _b2 = b2;
    }

    /** Call the bound function
     */
    R call() {
        return _func(_b1, _b2);
    }

#ifdef MBED_OPERATORS
    R operator()() {
        return _func(_b1, _b2);
    }
#endif
private:
    FuncPtr<R(B1, B2)> _func;
    B1 _b1;
    B2 _b2;
};


/** Binding with memory ownership on the user
 */
template <typename R, typename A1, typename A2, typename A3, typename B1>
class Binder<R(A1, A2, A3), B1> {
public:
    /** Create an empty Binder
     */
    Binder() {}

    /** Create a Binder, binding arguments to function
     */
    Binder(FuncPtr<R(B1, A1, A2, A3)> func, B1 b1) {
        attach(func, b1);
    }

    /** Create a Binder, binding arguments to a method
     */
    template <typename T, typename M>
    Binder(T *object, M method, B1 b1) {
        attach(object, method, b1);
    }

    /** Bind arguments to a function
     */
    void attach(FuncPtr<R(B1, A1, A2, A3)> func, B1 b1) {
        _func = func;
        _b1 = b1;
    }

    /** Bind arguments to a method
     */
    template <typename T, typename M>
    void attach(T *object, M method, B1 b1) {
        _func = FuncPtr<R(B1, A1, A2, A3)>(object, method);
        _b1 = b1;
    }

    /** Call the bound function
     */
    R call(A1 a1, A2 a2, A3 a3) {
        return _func(_b1, a1, a2, a3);
    }

#ifdef MBED_OPERATORS
    R operator()(A1 a1, A2 a2, A3 a3) {
        return _func(_b1, a1, a2, a3);
    }
#endif
private:
    FuncPtr<R(B1, A1, A2, A3)> _func;
    B1 _b1;
};

/** Binding with memory ownership on the user
 */
template <typename R, typename A1, typename A2, typename B1>
class Binder<R(A1, A2), B1> {
public:
    /** Create an empty Binder
     */
    Binder() {}

    /** Create a Binder, binding arguments to function
     */
    Binder(FuncPtr<R(B1, A1, A2)> func, B1 b1) {
        attach(func, b1);
    }

    /** Create a Binder, binding arguments to a method
     */
    template <typename T, typename M>
    Binder(T *object, M method, B1 b1) {
        attach(object, method, b1);
    }

    /** Bind arguments to a function
     */
    void attach(FuncPtr<R(B1, A1, A2)> func, B1 b1) {
        _func = func;
        _b1 = b1;
    }

    /** Bind arguments to a method
     */
    template <typename T, typename M>
    void attach(T *object, M method, B1 b1) {
        _func = FuncPtr<R(B1, A1, A2)>(object, method);
        _b1 = b1;
    }

    /** Call the bound function
     */
    R call(A1 a1, A2 a2) {
        return _func(_b1, a1, a2);
    }

#ifdef MBED_OPERATORS
    R operator()(A1 a1, A2 a2) {
        return _func(_b1, a1, a2);
    }
#endif
private:
    FuncPtr<R(B1, A1, A2)> _func;
    B1 _b1;
};

/** Binding with memory ownership on the user
 */
template <typename R, typename A1, typename B1>
class Binder<R(A1), B1> {
public:
    /** Create an empty Binder
     */
    Binder() {}

    /** Create a Binder, binding arguments to function
     */
    Binder(FuncPtr<R(B1, A1)> func, B1 b1) {
        attach(func, b1);
    }

    /** Create a Binder, binding arguments to a method
     */
    template <typename T, typename M>
    Binder(T *object, M method, B1 b1) {
        attach(object, method, b1);
    }

    /** Bind arguments to a function
     */
    void attach(FuncPtr<R(B1, A1)> func, B1 b1) {
        _func = func;
        _b1 = b1;
    }

    /** Bind arguments to a method
     */
    template <typename T, typename M>
    void attach(T *object, M method, B1 b1) {
        _func = FuncPtr<R(B1, A1)>(object, method);
        _b1 = b1;
    }

    /** Call the bound function
     */
    R call(A1 a1) {
        return _func(_b1, a1);
    }

#ifdef MBED_OPERATORS
    R operator()(A1 a1) {
        return _func(_b1, a1);
    }
#endif
private:
    FuncPtr<R(B1, A1)> _func;
    B1 _b1;
};

/** Binding with memory ownership on the user
 */
template <typename R, typename B1>
class Binder<R(), B1> {
public:
    /** Create an empty Binder
     */
    Binder() {}

    /** Create a Binder, binding arguments to function
     */
    Binder(FuncPtr<R(B1)> func, B1 b1) {
        attach(func, b1);
    }

    /** Create a Binder, binding arguments to a method
     */
    template <typename T, typename M>
    Binder(T *object, M method, B1 b1) {
        attach(object, method, b1);
    }

    /** Bind arguments to a function
     */
    void attach(FuncPtr<R(B1)> func, B1 b1) {
        _func = func;
        _b1 = b1;
    }

    /** Bind arguments to a method
     */
    template <typename T, typename M>
    void attach(T *object, M method, B1 b1) {
        _func = FuncPtr<R(B1)>(object, method);
        _b1 = b1;
    }

    /** Call the bound function
     */
    R call() {
        return _func(_b1);
    }

#ifdef MBED_OPERATORS
    R operator()() {
        return _func(_b1);
    }
#endif
private:
    FuncPtr<R(B1)> _func;
    B1 _b1;
};

} // namespace mbed

#endif
