/*  Thunker
 *  Thunk generating class for calling function objects through
 *  context-less functions
 */
#ifndef THUNKER_H
#define THUNKER_H

#include "FuncPtr.h"

extern "C" {
#include "trampoline.h"
}


/** Thunk generating class for calling function objects
 */
class Thunker {
public:
    /** Create a Thunker around a function
     */
    Thunker(FuncPtr<void()> func = 0) {
        attach(func);
        _entry = reinterpret_cast<void(*)()>(
            trampoline_init(&_trampoline, &Thunker::thunk, this));
    }

    /** Create a Thunker around a method
     */
    template <typename T, typename M>
    Thunker(T *obj, M method) {
        attach(obj, method);
        _entry = reinterpret_cast<void(*)()>(
            trampoline_init(&_trampoline, &Thunker::thunk, this));
    }

    /** Attach a function
     */
    void attach(FuncPtr<void()> func) {
        _func.attach(func);
    }

    /** Attach a method
     */
    template <typename T, typename M>
    void attach(T *obj, M method) {
        attach(FuncPtr<void()>(obj, method));
    }

    /** Call the attached function
     */
    void call() {
        return _func();
    }

    /** Call the attached function
     */
    void operator()() {
        return call();
    }

    /** Test if function has be attached
     */
    operator bool() const {
        return _func;
    }

    /** Static thunk for passing as C-style function
     *  @param func Thunker to call passed as void pointer
     */
    static void thunk(void *func) {
        return static_cast<Thunker*>(func)->call();
    }

    /** Entry point into thunk into function
     */
    entry_t *entry() {
        return _entry;
    }

    /** Entry point into thunk into function
     */
    operator entry_t*() {
        return entry();
    }

private:
    entry_t *_entry;
    trampoline_t _trampoline;
    FuncPtr<void()> _func;
};


#endif
