/*  Chainer
 *  Static function chaining
 */
#ifndef CHAINER_H
#define CHAINER_H

#include "FuncPtr.h"


/** Static function chaining
 */
template <typename F, int N>
class Chainer;

/** Static function composition
 */
template <typename A0, typename A1, typename A2, typename A3, typename A4, int N>
class Chainer<void(A0, A1, A2, A3, A4), N> {
public:
    /** Access a function in the chain
     */
    FuncPtr<void(A0, A1, A2, A3, A4)> &operator[](int i) {
        return _funcs[i];
    }

    /** Attach a function to the chain
     */
    bool attach(FuncPtr<void(A0, A1, A2, A3, A4)> func) {
        for (int i = 0; i < N; i++) {
            if (!_funcs[i]) {
                _funcs[i].attach(func);
                return true;
            }
        }

        return false;
    }

    /** Attach a function to the chain
     */
    template <typename T, typename M>
    bool attach(T *obj, M method) {
        return attach(FuncPtr<void(A0, A1, A2, A3, A4)>(obj, method));
    }

    /** Call each function in the chain
     */
    void call(A0 a0, A1 a1, A2 a2, A3 a3, A4 a4) {
        for (int i = 0; i < N; i++) {
            if (_funcs[i]) {
                _funcs[i](a0, a1, a2, a3, a4);
            }
        }
    }

    /** Call each function in the chain
     */
    void operator()(A0 a0, A1 a1, A2 a2, A3 a3, A4 a4) {
        return call(a0, a1, a2, a3, a4);
    }

    /** Test if any functions have been attached
     */
    operator bool() const {
        for (int i = 0; i < N; i++) {
            if (_funcs[i]) {
                return true;
            }
        }
        return false;
    }

    /** Static thunk for passing as C-style function
     *  @param func Chainer to call passed as void pointer
     */
    static void thunk(void *func, A0 a0, A1 a1, A2 a2, A3 a3, A4 a4) {
        return static_cast<Chainer<void(A0, A1, A2, A3, A4), N>*>(func)
                ->call(a0, a1, a2, a3, a4);
    }

private:
    FuncPtr<void(A0, A1, A2, A3, A4)> _funcs[N];
};

/** Static function composition
 */
template <typename A0, typename A1, typename A2, typename A3, int N>
class Chainer<void(A0, A1, A2, A3), N> {
public:
    /** Access a function in the chain
     */
    FuncPtr<void(A0, A1, A2, A3)> &operator[](int i) {
        return _funcs[i];
    }

    /** Attach a function to the chain
     */
    bool attach(FuncPtr<void(A0, A1, A2, A3)> func) {
        for (int i = 0; i < N; i++) {
            if (!_funcs[i]) {
                _funcs[i].attach(func);
                return true;
            }
        }

        return false;
    }

    /** Attach a function to the chain
     */
    template <typename T, typename M>
    bool attach(T *obj, M method) {
        return attach(FuncPtr<void(A0, A1, A2, A3)>(obj, method));
    }

    /** Call each function in the chain
     */
    void call(A0 a0, A1 a1, A2 a2, A3 a3) {
        for (int i = 0; i < N; i++) {
            if (_funcs[i]) {
                _funcs[i](a0, a1, a2, a3);
            }
        }
    }

    /** Call each function in the chain
     */
    void operator()(A0 a0, A1 a1, A2 a2, A3 a3) {
        return call(a0, a1, a2, a3);
    }

    /** Test if any functions have been attached
     */
    operator bool() const {
        for (int i = 0; i < N; i++) {
            if (_funcs[i]) {
                return true;
            }
        }
        return false;
    }

    /** Static thunk for passing as C-style function
     *  @param func Chainer to call passed as void pointer
     */
    static void thunk(void *func, A0 a0, A1 a1, A2 a2, A3 a3) {
        return static_cast<Chainer<void(A0, A1, A2, A3), N>*>(func)
                ->call(a0, a1, a2, a3);
    }

private:
    FuncPtr<void(A0, A1, A2, A3)> _funcs[N];
};

/** Static function composition
 */
template <typename A0, typename A1, typename A2, int N>
class Chainer<void(A0, A1, A2), N> {
public:
    /** Access a function in the chain
     */
    FuncPtr<void(A0, A1, A2)> &operator[](int i) {
        return _funcs[i];
    }

    /** Attach a function to the chain
     */
    bool attach(FuncPtr<void(A0, A1, A2)> func) {
        for (int i = 0; i < N; i++) {
            if (!_funcs[i]) {
                _funcs[i].attach(func);
                return true;
            }
        }

        return false;
    }

    /** Attach a function to the chain
     */
    template <typename T, typename M>
    bool attach(T *obj, M method) {
        return attach(FuncPtr<void(A0, A1, A2)>(obj, method));
    }

    /** Call each function in the chain
     */
    void call(A0 a0, A1 a1, A2 a2) {
        for (int i = 0; i < N; i++) {
            if (_funcs[i]) {
                _funcs[i](a0, a1, a2);
            }
        }
    }

    /** Call each function in the chain
     */
    void operator()(A0 a0, A1 a1, A2 a2) {
        return call(a0, a1, a2);
    }

    /** Test if any functions have been attached
     */
    operator bool() const {
        for (int i = 0; i < N; i++) {
            if (_funcs[i]) {
                return true;
            }
        }
        return false;
    }

    /** Static thunk for passing as C-style function
     *  @param func Chainer to call passed as void pointer
     */
    static void thunk(void *func, A0 a0, A1 a1, A2 a2) {
        return static_cast<Chainer<void(A0, A1, A2), N>*>(func)
                ->call(a0, a1, a2);
    }

private:
    FuncPtr<void(A0, A1, A2)> _funcs[N];
};

/** Static function composition
 */
template <typename A0, typename A1, int N>
class Chainer<void(A0, A1), N> {
public:
    /** Access a function in the chain
     */
    FuncPtr<void(A0, A1)> &operator[](int i) {
        return _funcs[i];
    }

    /** Attach a function to the chain
     */
    bool attach(FuncPtr<void(A0, A1)> func) {
        for (int i = 0; i < N; i++) {
            if (!_funcs[i]) {
                _funcs[i].attach(func);
                return true;
            }
        }

        return false;
    }

    /** Attach a function to the chain
     */
    template <typename T, typename M>
    bool attach(T *obj, M method) {
        return attach(FuncPtr<void(A0, A1)>(obj, method));
    }

    /** Call each function in the chain
     */
    void call(A0 a0, A1 a1) {
        for (int i = 0; i < N; i++) {
            if (_funcs[i]) {
                _funcs[i](a0, a1);
            }
        }
    }

    /** Call each function in the chain
     */
    void operator()(A0 a0, A1 a1) {
        return call(a0, a1);
    }

    /** Test if any functions have been attached
     */
    operator bool() const {
        for (int i = 0; i < N; i++) {
            if (_funcs[i]) {
                return true;
            }
        }
        return false;
    }

    /** Static thunk for passing as C-style function
     *  @param func Chainer to call passed as void pointer
     */
    static void thunk(void *func, A0 a0, A1 a1) {
        return static_cast<Chainer<void(A0, A1), N>*>(func)
                ->call(a0, a1);
    }

private:
    FuncPtr<void(A0, A1)> _funcs[N];
};

/** Static function composition
 */
template <typename A0, int N>
class Chainer<void(A0), N> {
public:
    /** Access a function in the chain
     */
    FuncPtr<void(A0)> &operator[](int i) {
        return _funcs[i];
    }

    /** Attach a function to the chain
     */
    bool attach(FuncPtr<void(A0)> func) {
        for (int i = 0; i < N; i++) {
            if (!_funcs[i]) {
                _funcs[i].attach(func);
                return true;
            }
        }

        return false;
    }

    /** Attach a function to the chain
     */
    template <typename T, typename M>
    bool attach(T *obj, M method) {
        return attach(FuncPtr<void(A0)>(obj, method));
    }

    /** Call each function in the chain
     */
    void call(A0 a0) {
        for (int i = 0; i < N; i++) {
            if (_funcs[i]) {
                _funcs[i](a0);
            }
        }
    }

    /** Call each function in the chain
     */
    void operator()(A0 a0) {
        return call(a0);
    }

    /** Test if any functions have been attached
     */
    operator bool() const {
        for (int i = 0; i < N; i++) {
            if (_funcs[i]) {
                return true;
            }
        }
        return false;
    }

    /** Static thunk for passing as C-style function
     *  @param func Chainer to call passed as void pointer
     */
    static void thunk(void *func, A0 a0) {
        return static_cast<Chainer<void(A0), N>*>(func)
                ->call(a0);
    }

private:
    FuncPtr<void(A0)> _funcs[N];
};

/** Static function composition
 */
template <int N>
class Chainer<void(), N> {
public:
    /** Access a function in the chain
     */
    FuncPtr<void()> &operator[](int i) {
        return _funcs[i];
    }

    /** Attach a function to the chain
     */
    bool attach(FuncPtr<void()> func) {
        for (int i = 0; i < N; i++) {
            if (!_funcs[i]) {
                _funcs[i].attach(func);
                return true;
            }
        }

        return false;
    }

    /** Attach a function to the chain
     */
    template <typename T, typename M>
    bool attach(T *obj, M method) {
        return attach(FuncPtr<void()>(obj, method));
    }

    /** Call each function in the chain
     */
    void call() {
        for (int i = 0; i < N; i++) {
            if (_funcs[i]) {
                _funcs[i]();
            }
        }
    }

    /** Call each function in the chain
     */
    void operator()() {
        return call();
    }

    /** Test if any functions have been attached
     */
    operator bool() const {
        for (int i = 0; i < N; i++) {
            if (_funcs[i]) {
                return true;
            }
        }
        return false;
    }

    /** Static thunk for passing as C-style function
     *  @param func Chainer to call passed as void pointer
     */
    static void thunk(void *func) {
        return static_cast<Chainer<void(), N>*>(func)
                ->call();
    }

private:
    FuncPtr<void()> _funcs[N];
};


#endif
