/*  EventLoop
 *
 *  EventQueue wrapped in a thread
 */
#ifndef EVENT_LOOP_H
#define EVENT_LOOP_H

#include "EventQueue.h"
#include "Thread.h"

/** Managed queue coupled with its own thread
 */
class EventLoop : public EventQueue {
public:
    /** Create an event loop without starting execution
     *  @param priority         Initial priority of the thread
     *                          (default: osPriorityNormal)
     *  @param event_count      Number of events to allow enqueueing at once
     *                          (default: 32)
     *  @param event_context    Max size of arguments passed with an event
     *                          (default: 0)
     *  @param event_pointer    Pointer to memory area to be used for events
     *                          (default: NULL)
     *  @param stack_size       Stack size (in bytes) requirements for the thread
     *                          (default: DEFAULT_STACK_SIZE)
     *  @param stack_pointer    Pointer to stack area to be used by the thread
     *                          (default: NULL)
     */
    EventLoop(osPriority priority=osPriorityNormal,
              unsigned event_count=32,
              unsigned event_context=0,
              unsigned char *event_pointer=NULL,
              uint32_t stack_size=DEFAULT_STACK_SIZE,
              unsigned char *stack_pointer=NULL);

    /** Create an event loop running in a dedicated thread
     *  @param start            True to start on construction
     *  @param priority         Initial priority of the thread
     *                          (default: osPriorityNormal)
     *  @param event_count      Number of events to allow enqueueing at once
     *                          (default: 32)
     *  @param event_context    Max size of arguments passed with an event
     *                          (default: 0)
     *  @param event_pointer    Pointer to memory area to be used for events
     *                          (default: NULL)
     *  @param stack_size       Stack size (in bytes) requirements for the thread
     *                          (default: DEFAULT_STACK_SIZE)
     *  @param stack_pointer    Pointer to stack area to be used by the thread
     *                          (default: NULL)
     */
    EventLoop(bool start,
              osPriority priority=osPriorityNormal,
              unsigned event_count=32,
              unsigned event_context=0,
              unsigned char *event_pointer=NULL,
              uint32_t stack_size=DEFAULT_STACK_SIZE,
              unsigned char *stack_pointer=NULL);

    /** Clean up event loop
     */
    ~EventLoop();

    /** Starts an event loop running in a dedicated thread
     */
    osStatus start();

    /** Stops an event loop cleanly, waiting for any currently executing events
     */
    osStatus stop();

private:
    static void run(const void *p);

    bool _running;
    osPriority _priority;
    uint32_t _stack_size;
    unsigned char *_stack_pointer;
    rtos::Thread *_thread;
};

#endif
