//
// This file contains an example implementation of M2X using the HTTP interface as the underlying 
// transport.
//

#include "mbed.h"
#include "PinDetect.h"
#include "WNCInterface.h"
#include "SDFileSystem.h" 
#include "FATFileSystem.h" 
#include "ConfigFile.h"

#define MBED_PLATFORM
#define M2X_ENABLE_READER

#include <jsonlite.h>
#include "M2XStreamClient.h"

#define CRLF "\n\r"


WNCInterface eth;
Client client;
MODSERIAL pc(USBTX,USBRX,256,256);

//LocalFileSystem local("local"); Not supported on K64F
SDFileSystem local(PTE3, PTE1, PTE2, PTE4, "local");
ConfigFile cfg;

char fromTime[]= "1969-12-31T19:00:01.000Z"; // yyyy-mm-ddTHH:MM:SS.SSSZ
char endTime[25];

DigitalOut led_green(LED_GREEN);
DigitalOut led_red(LED_RED);
DigitalOut led_blue(LED_BLUE);

char key1[] = "deviceId";
char deviceId[BUFSIZ];
char key2[] = "m2xKey";
char m2xKey[BUFSIZ];

char timestamp[25];
int length = 25;

M2XStreamClient m2xClient(&client, m2xKey);
TimeService timeService(&m2xClient);




//********************************************************************************************************************************************
//* Set the RGB LED's Color
//* LED Color 0=Off to 7=White.  3 bits represent BGR (bit0=Red, bit1=Green, bit2=Blue) 
//********************************************************************************************************************************************
void SetLedColor(unsigned char ucColor)
{
    //Note that when an LED is on, you write a 0 to it:
    led_red = !(ucColor & 0x1); //bit 0
    led_green = !(ucColor & 0x2); //bit 1
    led_blue = !(ucColor & 0x4); //bit 2
} //SetLedColor()
 


/*
 * Note, the PinDetect can be defined thus:-
 *     PinDetect pin( p21, PullDown );
 * This allows you to specify the DigitalIn pinmode
 * when you create the PinDetect object. This means
 * using pin.mode() later is then no longer required.
 */

PinDetect  pin ( A4, PullUp );
PinDetect  button ( SW3, PullUp );
int pinState; //0 = no trigger, 1 = triggered
int m2xOnline; //0 = offline, 1= online

// C function callbacks follow.
 
void keyPressed( void ) {
    pinState = pinState++;
    if (m2xOnline >= 1) {
        SetLedColor(1);
    }
    //pc.printf("Key Pressed\n");
}
 
void keyReleased( void ) {
    pinState = pinState--;
    if (m2xOnline >= 1) {
        SetLedColor(4);
    }
//    if(pinState == 0){
//        SetLedColor(4);
//    };
    //pc.printf("Key Released\n");
}
 
void keyPressedHeld( void ) {
    pinState = 2;
    //led3 = 1;
    //pc.printf("Key Pressed Held\n");
}
 
void keyReleasedHeld( void ) {
    //led4 = 1;
    //pc.printf("Key Released Held\n");
}

void initModem() {
    int response;
    pc.printf("Start m2x-demo-all by initializing the network" CRLF);
    eth.doDebug(2);
    response = eth.init(NULL, &pc);                     
    pc.printf("Response is: %d" CRLF, response);
    pc.printf("WNC Module %s initialized (%02X)." CRLF, response?"IS":"IS NOT", response);
    while( !response ) {
        SetLedColor(0);
        pc.printf(" - - - - - - - WAITING FOR CELL MODEM TO COME ONLINE - - - - - - - " CRLF);
        wait(30);
        response = eth.init(NULL, &pc);
        pc.printf("WNC Module %s initialized (%02X)." CRLF, response?"IS":"IS NOT", response);
    };
    
    if(response){
       eth.doDebug(0);
       SetLedColor(5);
    };
    
    response = eth.connect();                 
    pc.printf("IP Address: %s " CRLF CRLF, eth.getIPAddress());
    
}

void initM2X() {
    pc.printf("starting M2X data push..." CRLF);

    pc.printf("initialize the M2X time service" CRLF);
    if (!m2x_status_is_success(timeService.init())) 
    pc.printf("Cannot initialize time service!" CRLF);
    else {
        timeService.getTimestamp(timestamp, &length);
        pc.printf("Current timestamp: %s" CRLF, timestamp);
        strcpy(endTime,timestamp);
        m2xOnline = 1;
        SetLedColor(4);
    }
}
 
// The main program.
 
int main() {
    
    pc.baud(115200);
    pc.printf("Hello World! I am here!" CRLF);
    
    wait(5);
    
    
    /*
     * Read a configuration file from a mbed.
     */
     
    if (!cfg.read("/local/settings.txt")) {
        pc.printf("Failure to read a configuration file." CRLF);
    }
    
    /*
     * Get a configuration value.
     */
     
    if (cfg.getValue(key1, &deviceId[0], sizeof(deviceId))) {
        pc.printf("'%s'='%s'" CRLF, key1, deviceId);
    }
    
    if (cfg.getValue(key2, &m2xKey[0], sizeof(m2xKey))) {
        pc.printf("'%s'='%s'" CRLF, key2, m2xKey);
    }
    
    
    pinState = 0;
    pin.attach_asserted( &keyPressed );
    pin.attach_deasserted( &keyReleased );
    button.attach_asserted( &keyPressed );
    button.attach_deasserted( &keyReleased );
    //pin.attach_asserted_held( &keyPressedHeld );
    
    // This callback will often be of little use as it's
    // called after every assertion/deassertion. However,
    // it's provided for completeness. You may find a use
    // for it. If not, just don't attach a callback and it
    // will not activate.
    //pin.attach_deasserted_held( &keyReleasedHeld );
    
    // You can define how many continuous samples must be
    // asserted before the attach_asserted() function is called.
    //     pin.setSamplesTillAssert( 10 );
    // This would mean 10 * 20ms debounce time = 200ms.
 
    // You can define how many continuous samples must be
    // asserted before the attach_asserted_held() function is called.
    //     pin.setSamplesTillHeld( 200 );
    // This would mean 200 * 20ms debounce time = 2seconds.
 
    // By default, "asserted" assumes the pin going high from 0volts to 5volts
    // and deasserted assumes going from 5volts to 0volts. You can invert this
    // logic so that going to 0volts is asserted and going to 5volts is deasserted
    // using this setup function:-
    pin.setAssertValue( 0 );
    button.setAssertValue( 0 );
    
    int m2xresponse, cnt=1;
    int pinValue;
    
    wait(20);
    initModem(); //call function to init modem, wait till it comes online, and get IP address
    initM2X();

//    //debug
//    wait(10);   // wait for terminal connected
//    pc.printf("STARTING WNCInterface" CRLF);
//    eth.doDebug(0); // if you want a LOT of AT commands logged (1)
//    
//    // init and connect the WNCInterface cell modem
//    pc.printf("init() returned 0x%04X" CRLF, eth.init());
//    response = eth.connect();
//
//    if(response){
//        SetLedColor(2);
//    };

 //   pc.printf("IP Address: %s " CRLF, eth.getIPAddress());
 //   pc.printf("-------------------------------------" CRLF);

    
    // Start Pin Samping
    // Sampling does NOT begin until you set the frequency. So, until
    // you call this function NO callbacks will be made. With no arguments
    // passed the default is 20000 microseconds (20ms). Specifiy the sampling
    // period in microseconds if you want a different value to 20ms.
    // For example, for a sampling period of 10ms do:-
    //     pin.setSampleFrequency( 10000 );
    // Note, if you change the sampling frequency you will probably also
    // want to change the number of samples till assert and held as show
    // above.
    pin.setSampleFrequency(); // Defaults to 20ms.
    button.setSampleFrequency();
    
    while( 1 ) {
        //pc.printf("Hello World! I am still here!" CRLF);
        pc.printf("Pin state is: %d" CRLF, pinState);
        if(pinState >= 1){
            pinValue = 1;
        }
        else {
            pinValue = 0;
        }
        pc.printf("Pin value is: %d" CRLF, pinValue);
        pc.printf("cnt=%d\r\n",cnt++);
        // post the temp value
        pc.printf("Post updateStreamValue (pinValue = %d)..." CRLF, pinState);
        m2xresponse = m2xClient.updateStreamValue(deviceId, "water-sensor", pinValue);
        pc.printf("Post response code: %d" CRLF, m2xresponse);
        if (m2xresponse != 202){
            m2xOnline = 0;
            SetLedColor(2);
            pc.printf("Error connecting to M2X" CRLF);
            //pc.printf("IP Address: %s " CRLF CRLF, eth.getIPAddress());
            if (m2xresponse == -1){
                SetLedColor(0);
                pc.printf("Modem Error - attempting to reconnect" CRLF);
                initModem();
                initM2X();
            }
        }
        if (m2xresponse == 202){
            if (m2xOnline == 0) { //first time m2x came back online
                SetLedColor(4);
            }
            m2xOnline = m2xOnline++;
        }
        timeService.getTimestamp(timestamp, &length);
        pc.printf("Thats all folks, got to wait 1 second... (%s)" CRLF CRLF CRLF, timestamp);
        
        // wait 1 secs and then loop
        delay(1000);
    }
}