/* mbed library for driving the EA QVGA 2.8" OLED
 * Copyright (c) Graham Bloice 2011
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */
 
#ifndef MBED_EAQVGAOLED_H
#define MBED_EAQVGAOLED_H
 
#include "mbed.h"
#include "GraphicsDisplay.h"
 
#define   BLACK            0x0000        /*   0,   0,   0 */
#define   NAVY            0x000F      /*   0,   0, 128 */
#define   DARK_GREEN    0x03E0      /*   0, 128,   0 */
#define   DARK_CYAN        0x03EF      /*   0, 128, 128 */
#define   MAROON        0x7800      /* 128,   0,   0 */
#define   PURPLE        0x780F      /* 128,   0, 128 */
#define   OLIVE            0x7BE0      /* 128, 128,   0 */
#define   LIGHT_GRAY    0xC618      /* 192, 192, 192 */
#define   DARK_GRAY        0x7BEF      /* 128, 128, 128 */
#define   BLUE            0x001F      /*   0,   0, 255 */
#define   GREEN            0x07E0      /*   0, 255,   0 */
#define   CYAN          0x07FF      /*   0, 255, 255 */
#define   RED           0xF800      /* 255,   0,   0 */
#define   MAGENTA        0xF81F      /* 255,   0, 255 */
#define   YELLOW        0xFFE0      /* 255, 255, 0   */
#define   WHITE            0xFFFF      /* 255, 255, 255 */

#define   CURSOR_CLS    0
#define   CURSOR_UP     1
#define   CURSOR_DOWN   2
#define   CURSOR_LEFT   3
#define   CURSOR_RIGHT  4

 /**
  * @brief Uses SPI to drive the Embedded Artists 2.8" QVGA OLED panel
  * @author Graham Bloice
  * @see http://mbed.org/cookbook/EAQVGAOLED
  * @see example.cpp
  * @see API
  *
  * <b>EAQVGAOLED</b> defines a library to drive the Embedded Artists QVGA 2.8" OLED panel.
  * Currently the library uses SPI to drive the panel.
  * 
  * Standard Example:
  * @code
  * #include "mbed.h"
  * #include "EAQVGAOLED.h"
  * int main() {
  *
  *     // Create an instance of the display driver
  *     EAQVGAOLED display = EAQVGAOLED(p5, p6, p7, p8, p9, p10);
  *  
  *     display.background(BLACK);
  *     display.foreground(DARK_GRAY);
  *     display.cls();
  *     display.printf("Hello from mbed");
  * }
  * @endcode  
  */ 
class EAQVGAOLED : public GraphicsDisplay {
 
public:
 
    /**
     * @brief The driver constructor
     *
     * Create an EA QVGA OLED object connected using
     *  the specified SPI port, and two output pins;
     *  the Chip Select pin and the Backlight pin. 
     *
     * @param mosi SPI data output pin
     * @param miso SPI data input pin
     * @param sclk SPI Clock pin
     * @param cs Pin to drive the display CS
     * @param reset Pin to drive the display reset
     * @param bl Pin to drive the display backlight
     */
    EAQVGAOLED(PinName mosi, PinName miso, PinName sclk, PinName cs, PinName reset, PinName bl);
    
    // These are the functions required to support the base class
    
    /**
     * Set a pixel to the specified colour
     *
     * @ingroup API
     * @param x Pixel x position
     * @param y Pixel y position
     * @param colour Pixel colour
     */
    virtual void pixel(int x, int y, int colour);

    /**
     * Accessor to obtain the display width
     *
     * @return the width of the display in pixels 
     */
    virtual int width(void) { return 240; }
    
    /**
     * Accessor to obtain the display height
     *
     * @return the height of the display in pixels
     */
    virtual int height(void) { return 320; }

    /**
     * Clear the screen
     *
     * @ingroup API
     */           
    virtual void cls() { fillRectangle(0, 0, width(), height(), _background); }

    /* No overrides for these functions
    
    virtual void window(int x, int y, int w, int h);
    virtual void putp(int colour);
    
    virtual void fill(int x, int y, int w, int h, int colour);
    virtual void blit(int x, int y, int w, int h, const int *colour);    
    virtual void blitbit(int x, int y, int w, int h, const char* colour);
    
    virtual void character(int column, int row, int c);
    virtual int columns();
    virtual int rows();
    
    */

    // These are new functions for this class
    
    /**
     * Draw a horizontal line
     *
     * @ingroup API
     * @param x0 Starting x position
     * @param y0 Starting y position
     * @param x1 End x position
     * @param colour Colour of line
     */
    void hLine(const uint16_t x0, const uint16_t y0, const uint16_t x1, const uint16_t colour);
        
    /**
     * Draw a vertical line
     *
     * @ingroup API
     * @param x0 Starting x position
     * @param y0 Starting y position
     * @param y1 End y position
     * @param colour Colour of line
     */
    void vLine(const uint16_t x0, const uint16_t y0, const uint16_t y1, const uint16_t colour);
    
    /**
     * Draw a rectangle in the specified colour
     *
     * @ingroup API
     * @param x0 Starting x position
     * @param y0 Starting y position
     * @param x1 Ending x position
     * @param y1 Ending y position
     * @param colour Outline colour of rectangle
     */
    void rectangle(const uint16_t x0, const uint16_t y0, const uint16_t x1, const uint16_t y1, uint16_t colour); 
        
    /**
     * Draw a filled rectangle in the specified colour
     *
     * @ingroup API
     * @param x0 Starting x position
     * @param y0 Starting y position
     * @param x1 Ending x position
     * @param y1 Ending y position
     * @param colour Filled colour of rectangle
     */
    void fillRectangle(const uint16_t x0, const uint16_t y0, const uint16_t x1, const uint16_t y1, uint16_t colour);
    
    /**
     * Control the backlight
     *
     * ingroup API
     * @param on Enable the backlight
     */
    void backlightControl(bool on) {
                                       _bl = on ? 1 : 0;
                                   }
    
protected:

    /**
     * Print a character, overridden to handle control characters
     *
     * @ingroup API
     * @param value The character value to print
     * @return The character value printed
     *
     */
    virtual int _putc(int value);

private:
    bool resetDisplay(void);                                        // Reset the display
    void initHardware(void);                                        // Initialise the hardware
    inline void setRegisterIndex(const uint8_t reg);                // Write register
    inline void writeData(const uint16_t data);                     // Write data
    uint16_t readDataRegister(const uint8_t reg);                   // Read from a register
    void writeDataRegister(const uint8_t reg, const uint16_t data); // Write data to a register
    void movePen(const uint16_t x, const uint16_t y);               // Set the pen position
    
    SPI         _spi;   // The SPI Port to use for data I/O
    DigitalOut  _cs;    // The I/O pin for the Chip Select
    DigitalOut  _reset; // The I/O pin for the reset
    DigitalOut  _bl;    // The I/O pin for the blacklight
 }; 
 
 #endif