/**
 *  Ezo ph sensor library
 *
 *  @author  Louis-Philippe Gauthier
 *  @version 1.0
 *  @date    03-March-2017
 *
 *  Library for "EZO PH sensor module" from Atlas Scientific
 *    http://www.atlas-scientific.com/product_pages/circuits/ezo_ph.html
 *
 *  For more information about the EZO:
 *    http://www.atlas-scientific.com/_files/_datasheets/_circuit/pH_EZO_datasheet.pdf?
 */

#ifndef MBED_EZOPH_H
#define MBED_EZOPH_H

#include "mbed.h"
#include <string>

#define DEFAULT_SLAVE_ADDRESS (0x63 << 1)


/** EZO class
 *
 *  EZO: A library to communicate with the Ezo PH sensor
 *
 *  @endcode
 */

class EZOPH
{
public:

    /** Create a EZOPH instance
     *  which is connected to specified I2C pins with specified address
     *
     * @param sda I2C-bus SDA pin
     * @param scl I2C-bus SCL pin
     * @param slave_adr (option) I2C-bus address (default: 0x63)
     */
    EZOPH(PinName sda, PinName sck, char slave_adr = DEFAULT_SLAVE_ADDRESS);

    /** Destructor of EZOPH
     */
    virtual ~EZOPH();

    /** clear cmd buffer and response buffer
     *
     */
    void clear(void);

    /** Set the temperature compensation value in DegC
     *
     */
    int Tcompensation (float tmp);

    /** Ask the compensation temperature of the EZOPH sensor
     *
     * return the compensation temperature in float with one decimal :
     * 19.5 is the temperature in degC
     *
     */
    float QTcompensation (void);

    /** Ask EZOPH sensor information
     *
     * return a string containing the sensor information in the following template :
     * ?I,ph,1.0
     * ph = device type
     * 1.0 = firmware version number
     *
     */
    string getSensorInfo(void);

    /** Ask EZOPH sensor status (reason for last reset and value of VCC)
     *
     * return a string containing the sensor information in the following template :
     * ?STATUS,P,5.038
     * P = reason for the last reset event
     *          P = power on reset
     *          S = software reset
     *          B = nrown out reset
     *          W = watchdog reset
     *          U = unknown
     * 5.038 = voltage at VCC
     */
    string getSensorStatus(void);


    /** EZOPH sensor do a single reading
     *
     */
    float read(void);

    /** Enable or Disable the LED of the EZOPH sensor
     *
     * [in] 1 : Enable LED (default)
     * [in] 0 : Disable LED
     *
     */
    int sensorLED(bool on);

    /** Query the LED state of the EZOPH sensor
     *
     */
    int QsensorLED(void);

    /** Clear the EZOPH sensor calibration
     *
     */
    int calibrationClear(void);


    /** Ask the status of the EZOPH sensor calibration
     *
     * return a string in the following template : 
     * ?CAL,0 : not calibrated
     * ?CAL,1 : single point calibration
     * ?CAL,2 : two point calibration
     * ?CAL,3 : three point calibration
     */
    string calibrationQuery(void);


    /** Calibrate the EZOPH sensor for a PH of 7.00
      *
      */
    int calibratingMid(void);


    /** Calibrate the EZOPH sensor for a PH of 4.00
    *
    */
    int calibratingLow(void);


    /** Calibrate the EZOPH sensor for a PH of 10.00
    *
    */
    int calibratingHigh(void);


    /** Ask the precision of the actual calibration
    *     
    * return a string containing the slope information in the following template :
    * ?SLOPE,99.7,100.3
    * 99.7 : how closely the slope of the acid calibration matched the "ideal" ph probe
    * 100.3 : how closely the slope of the base calibration matched the "ideal" ph probe
    *
    */
    string slope(void);


    /** Do a factory reset on the EZOPH sensor
     *
     */
    string factoryReset(void);


    /** Put the EZOPH sensor to sleep
     *
     */
    void sleep(void);

    /** Enable or Disable the protocol lock of the EZOPH sensor
     *
     * [in] 1 : Enable protocol lock
     * [in] 0 : Disable protocol lock
     *
     */
    int EDProtocolLock(bool on);


    /** Ask the status of the protocol lock of the EZOPH sensor
     *
     * return a string containing the protocl lock information in the following template :
     * ?PLOCK,1 : Protocl lock is ON
     * ?PLOCK,0 : Protocol lock is OFF
     */
    int QProtocolLock();


    /** Change protocol of communication from UART to I2C
     *
     */
    void changeUART(int baudRate);


private:

    I2C         *i2c_p;
    I2C         &i2c;
    char        address;
    char        cmdData[16];
    char        ezodata[21];
    char        receivedValue[20];
    string      ezovalue;
    string      error255;
    string      error254;
    string      failedStr;
    string      succesStr;
    int         failed;
    int         success;
};

#endif // MBED_EZOPH_H
