/***************************************************************************
Copyright 2016 LARNAUDIE GARIVET
 
   Licensed under the Apache License, Version 2.0 (the "License");
   you may not use this file except in compliance with the License.
   You may obtain a copy of the License at
 
       http://www.apache.org/licenses/LICENSE-2.0
 
   Unless required by applicable law or agreed to in writing, software
   distributed under the License is distributed on an "AS IS" BASIS,
   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
   See the License for the specific language governing permissions and
   limitations under the License.
***************************************************************************/
#ifndef CONTROLLERCAN_H
#define CONTROLLERCAN_H

#include "mbed.h"
#include "PeripheralCAN.h"
#include <vector>

#define SIZE_FIFO 32

class PeripheralCAN;

/** My CAN Controller class
 * Used to manage reading and writing procedures on a CAN Bus
 *
 * Examples :
 * @code
 * class Motor : public PeripheralCAN{ // Your own Peripheral class (here a motor)
 *      public :
 *      Motor(ControllerCAN* controller, unsigned short Id1, unsigned short Id2) : PeripheralCAN(controller), Id1_m(Id1), Id2_m(Id2){
 *          // Your stuffs
 *      };
 *           
 *      void init(void){
 *          addIdRead(&Id1); // To be done for each incomming Id
 *          addIdRead(&Id2); // To be done for each incomming Id
 *          // Your stuffs
 *      };
 *
 *      void update(const unsigned short& Id, const CANMessage& msg){
 *          if(Id == Id1){
 *               // update your own attributes with msg content
 *          }
 *          else if(Id == Id2){
 *               // update your own attributes with msg content
 *          }
 *      };
 *       
 *      private :
 *          unsigned short Id1_m;
 *          unsigned short Id2_m;
 * };
 *
 * int main(){
 *      ControllerCAN controller;
 *      Motor peripherial(&controller, 0x100, 0x101);
 *
 *      while(1){
 *          controller.FIFOread(); // Treatement of the FIFO CAN messages
 *          // Your stuffs
 *      } 
 * }
 * @endcode
 */
class ControllerCAN {
    public :
    /** Create ControllerCAN instance
     *
     * @param rx Rd pin
     * @param tx td pin
     */
    ControllerCAN(PinName rd, PinName td);
    
    /** Destroy ControllerCAN instance
     */
    ~ControllerCAN();
    
    /** Write a CANMessage on CAN Bus
     *
     * @param Id Id message
     * @param data char array containing data to be send
     * @param len size of the data array 
     * @returns
     *   1 if write was successful,
     *   0 if write failed
     */
    long writeData(long Id, const char *data, char len);
    
    /** Write a remote CANMessage on CAN Bus
     *
     * @param Id Id message
     * @returns
     *   1 if write was successful,
     *   0 if write failed
     */
    long writeRemote(long Id);
    
    /** Attach a PeripheralCAN instance to a ControllerCAN
     *
     * @param peripheral Pointer on a PeripheralCAN instance
     */
    void attach(PeripheralCAN* peripheral);
    
    /** Read one message on the CAN FIFO
     *
     * @returns
     *  -1 if no match is found between the message Id and all Ids of 
     *     the PeripheralCAN instances attached to the ControllerCAN,
     *  1 if a match is found
     */
    char FIFOread(void);
    //void FIFO_remove_msg(void);
    
    private :
    
    CAN can;
    vector<PeripheralCAN*> peripherals;
    unsigned char FIFO_ecriture;
    signed char FIFO_lecture;
    signed char FIFO_occupation;
    signed char FIFO_max_occupation;
    CANMessage can_MsgRx[SIZE_FIFO];
    
    /** Interrupt Service Routine called whenever a CAN frame received interrupt
     *  is generated.
     */
    void can_ISR_Reader(void);
    
    //static void CAN_automate_reception(void);
};

#endif
