#ifndef __MC33926_H__
#define __MC33926_H__

#include "mbed.h"

#define MAX_SPEED   5

enum mc33926_mode {
    LOCK_ANTIPHASE,
    SIGN_MAGNITUDE_BRAKING,
    SIGN_MAGNITUDE_COASTING,
};


/** driver for the MC33926 H-Bridge
 *  
 *  This driver allow to drive the MC33926 with 3 differents technics:
 *    - Lock antiphase: alternate 1 direction with another
 *    - Sign-manigtude braking: alternate 1 direction with braking
 *    - Sign-magnitude coasting: alternate 1 direction with freewheeling
 
 *  Speed can be set from -MAX_SPEED to +MAX_SPEED
 */
class mc33926
{
public:

    /** Create a MC33926 instance
     *  @param inv pin connected to inv
     *  @param in1 pin connected to in1
     *  @param in2 pin connected to in2
     *  @param d1 pin connected to d1
     *  @param d2 pin connected to d2
     *  @param en pin connected to enable (can be unset)
     *  @param sf pin connected to sf (can be unset)
     */
    mc33926(PinName inv, PinName in1, PinName in2, PinName d1, PinName d2, PinName en = NC, PinName sf = NC);
    
    /** Set driving mode
     *  @param _mode mode to set (LOCK_ANTIPHASE, SIGN_MAGNITUDE_BRAKING or SIGN_MAGNITUDE_COASTING)
     */
    void set_mode(int _mode);
    
    /** Set speed
     *  @param _speed (from -MAX_SPPED to +MAX_SPEED)
     */
    float set_speed(int _speed);
    
    /** Set PWM period
     *  @param _period pwm period in us. Warning: a low period implies SF set, see MC339276 datasheet
     */
    void set_period(int _period);
    
    /** Return driving mode
     */
    int get_mode();
    
    /** Return speed
     */
    int get_speed();
    
    /** Return PWM period in us
     */
    int get_period();
    
    /** Enable H-Bridge. Work only if enable pin is set
    */
    void enable();
    
    /* Disable H-Bridge. Work only if enable pin is set
    */
    void disable();

private:
    PinName pin_inv;
    PinName pin_in1;
    PinName pin_in2;
    PinName pin_d1;
    PinName pin_d2;
    PinName pin_en;
    PinName pin_sf;
    
    PwmOut *pwm;

    int mode;
    int speed;
    int period;
    
    void setup();
    void setup_lock_antiphase();
    void setup_sign_magnitude_braking();
    void setup_sign_magnitude_coasting();
    
    float speed_lock_antiphase();
    float speed_sign_magnitude();
    
    void period_lock_antiphase();
    void period_sign_magnitude();
};

#endif