#ifndef _LITERM_H_
#define _LITERM_H_

#include "mbed.h"

#define HIST_SIZE 5
#define MAX_CMDLINE_SIZE 128

class literm;

struct literm_cmd {
    char name[8];
    char desc[64];
    int (*func)(literm *, char *);
    
    literm_cmd *prev;
};

#define LITERM_CMD(_name, _desc)          \
    struct literm_cmd _name##_cmd = {     \
        .name = #_name,                 \
        .desc = _desc,                  \
        .func = do_##_name, };

/** Small terminal class.
 *  This class intends to provide a small terminal for the mbed. It supports the following features:
 *    - Add command based on a name, a description and a callback
 *    - Auto help command based on name and description declarations.
 *    - Ctrl-C to clear written command.
 *    - Delete last char with backspace (if VT100 supported).
 *    - Exit the CLI with ctrl-D.
 *    - Small History (with up and down arrows)
 *
 * Example:
 * @code
 * #include "mbed.h"
 * #include "literm.h"
 * 
 * int do_get(literm *term, char * args)
 * {
 *     Serial *out = term->get_out();
 *     
 *     out->printf("\r\nGetting!");
 *     
 *     return 0;
 * }
 * 
 * int do_set(literm *term, char * args)
 * {
 *     Serial *out = term->get_out();
 * 
 *     out->printf("\r\nSetting!");
 *     
 *     return 0;
 * }
 * 
 * LITERM_CMD(get, "getting something");
 * LITERM_CMD(set, "setting something");
 * 
 * int main() 
 * {
 *     Serial pc(USBTX, USBRX);
 *     literm term(&pc);
 *     
 *     pc.baud(115200);
 *     pc.format(8, Serial::None, 1);
 *    
 *     term.set_hostname("mbed cli 1");
 *     term.set_welcome("Bonjour et bienvenue!");
 *     
 *     term.add_cmd(&get_cmd);
 *     term.add_cmd(&set_cmd);
 *     term.loop();
 *     
 *     term.set_hostname("mbed cli 2");
 *     term.loop();
 * }
 * @endcode
 */
class literm
{

 public:
 
    /** Create a literm instance.
     * @param _out serial output used to print and received
     */
    literm(Serial * _out);
    
    /** Versatile loop which start and manage the cli.
     */
    void loop();
    
    /** Set the hostname used by the cli.
     *
     * @param name hostname to use, must be < 12 chars
     */
    void set_hostname(char *name);
    
    /** Set the welcome message used by the cli.
     *
     * @param message welcome message to use, must be < 126 chars
     */
    void set_welcome(char *message);
    
    /** Add a command to the smal command line.
     *
     * @param _cmd pointer to a command to add. Commands are created with LITERM_CMD macro
     */
    void add_cmd(literm_cmd *_cmd);
    
    /** Return a pointer to the command stack.
     *
     * @returns stack command pointer
     */
    literm_cmd* get_cmds();
    
    /** Return a pointer to the Serial used as input/output for cli.
     *
     * @returns serial input/ouput 
     */
    Serial * get_out();

 private:
    Serial *out;
    char prompt[16];
    char welcome[128];
    
    void redraw(const char* line);
    literm_cmd *cmd_list;
    literm_cmd help_cmd;
};

#endif
