#ifndef _ADAFRUIT_GFX_H_
#define _ADAFRUIT_GFX_H_

static inline void swap(int16_t &a, int16_t &b)
{
    int16_t t = a;
    
    a = b;
    b = t;
}

#ifndef _BV
#define _BV(bit) (1<<(bit))
#endif

#define BLACK 0
#define WHITE 1

class Adafruit_GFX : public Stream
{
 public:
    Adafruit_GFX(int16_t w, int16_t h)
        : _rawWidth(w)
        , _rawHeight(h)
        , _width(w)
        , _height(h)
        , cursor_x(0)
        , cursor_y(0)
        , textcolor(WHITE)
        , textbgcolor(BLACK)
        , textsize(1)
        , rotation(0)
        , wrap(true)
        {};

    /// Paint one BLACK or WHITE pixel in the display buffer
    // this must be defined by the subclass
    virtual void drawPixel(int16_t x, int16_t y, uint16_t color) = 0;
    // this is optional
    virtual void invertDisplay(bool i) {};
    
    // Stream implementation - provides printf() interface
    // You would otherwise be forced to use writeChar()
    virtual int _putc(int value) { return writeChar(value); };
    virtual int _getc() { return -1; };




    /// Draw a text character at a specified pixel location
    void drawChar(int16_t x, int16_t y, unsigned char c, uint16_t color, uint16_t bg, uint8_t size);
    /// Draw a text character at the text cursor location
    size_t writeChar(uint8_t);

    /// Get the width of the display in pixels
    inline int16_t width(void) { return _width; };
    /// Get the height of the display in pixels
    inline int16_t height(void) { return _height; };

    /// Set the text cursor location, based on the size of the text
    inline void setTextCursor(int16_t x, int16_t y) { cursor_x = x; cursor_y = y; };
#if defined(GFX_WANT_ABSTRACTS) || defined(GFX_SIZEABLE_TEXT)
    /** Set the size of the text to be drawn
     * @note Make sure to enable either GFX_SIZEABLE_TEXT or GFX_WANT_ABSTRACTS
     */
    inline void setTextSize(uint8_t s) { textsize = (s > 0) ? s : 1; };
#endif
    /// Set the text foreground and background colors to be the same
    inline void setTextColor(uint16_t c) { textcolor = c; textbgcolor = c; }
    /// Set the text foreground and background colors independantly
    inline void setTextColor(uint16_t c, uint16_t b) { textcolor = c; textbgcolor = b; };
    /// Set text wraping mode true or false
    inline void setTextWrap(bool w) { wrap = w; };


protected:
    int16_t  _rawWidth, _rawHeight;   // this is the 'raw' display w/h - never changes
    int16_t  _width, _height; // dependent on rotation
    int16_t  cursor_x, cursor_y;
    uint16_t textcolor, textbgcolor;
    uint8_t  textsize;
    uint8_t  rotation;
    bool  wrap; // If set, 'wrap' text at right edge of display
};

#endif
