/** mDot_TTN_DS18B20 - Simple mDot temperature sensor using Dallas Semiconductors DS18B20 OneWire temperature sensor.
 * It used the MANUAL join mode with parameters for The Things Network.
 *
 *
 * Uses MultiTech mDot developer board http://www.multitech.com/models/94558010LF
 * Requires a MultiTech MultiConnect Conduit http://www.multitech.com/models/94557203LF
 * http://www.multitech.net/developer/software/lora/conduit-mlinux-convert-to-basic-packet-forwarder/
 * http://forum.thethingsnetwork.org/t/setting-up-multitech-conduit-gateway-for-ttn/216/35
 *
 */

#include "mbed.h"
#include "DHT.h"
#include "mDot.h"
#include "MTSLog.h"
#include "MTSText.h"
#include <string>
#include <vector>

using namespace mts;

#define MIN(a,b) (((a)<(b))?(a):(b))
#define MAX(a,b) (((a)>(b))?(a):(b))

// Values as used by The Things Network
//const uint8_t AppKey[16]={0x2B, 0x7E, 0x15, 0x16, 0x28, 0xAE, 0xD2, 0xA6, 0xAB, 0xF7, 0x15, 0x88, 0x09, 0xCF, 0x4F, 0x3C};
// Application session key
uint8_t AppSKey[16]= {0x2B, 0x7E, 0x15, 0x16, 0x28, 0xAE, 0xD2, 0xA6, 0xAB, 0xF7, 0x15, 0x88, 0x09, 0xCF, 0x4F, 0x3C};
// Network session key
uint8_t NwkSKey[16]= {0x2B, 0x7E, 0x15, 0x16, 0x28, 0xAE, 0xD2, 0xA6, 0xAB, 0xF7, 0x15, 0x88, 0x09, 0xCF, 0x4F, 0x3C};

// Network Address - Get your own address range at http://thethingsnetwork.org/wiki/AddressSpace
uint8_t NetworkAddr[4]= {0x02,0x01,0x2D,0x00};      // Our Network address or Node ID


// Some defines for the LoRa configuration
#define LORA_SF mDot::SF_9
#define LORA_ACK 0
#define LORA_TXPOWER 14

// Ignoring sub band for EU modules.
static uint8_t config_frequency_sub_band = 7;

// DS18B20 OneWire pin
// D13 on Dev Board, pin 18 on mDot, Compatible with Oxford Flood Network PCB temperature sensor.
#define DATA_PIN     PA_5
// A0 on Dev Board, pin 20 on mDot
//#define DATA_PIN     PB_1

// Temperature sensor object
#define DHTTYPE DHT22

DHT sensor(DATA_PIN, DHTTYPE);

// Serial via USB for debugging only
Serial pc(USBTX,USBRX);

int main()
{
    int initStatus;
    int32_t ret;
    mDot* dot;
    std::vector<uint8_t> send_data;
    std::vector<uint8_t> recv_data;
    std::vector<uint8_t> nwkSKey;
    std::vector<uint8_t> nodeAddr;
    std::vector<uint8_t> networkAddr;

    float temperature = 0.0;

    pc.baud(115200);
    pc.printf("TTN mDot LoRa Temperature sensor\n\r");

    // get a mDot handle
    dot = mDot::getInstance();

    dot->setLogLevel(MTSLog::WARNING_LEVEL);
//    dot->setLogLevel(MTSLog::TRACE_LEVEL);

    logInfo("Checking Config");

    // Test if we've already saved the config
    std::string configNetworkName = dot->getNetworkName();

    uint8_t *it = NwkSKey;
    for (uint8_t i = 0; i<16; i++)
        nwkSKey.push_back((uint8_t) *it++);

    it = NetworkAddr;
    for (uint8_t i = 0; i<4; i++)
        networkAddr.push_back((uint8_t) *it++);

    logInfo("Resetting Config");
    // reset to default config so we know what state we're in
    dot->resetConfig();

    // Set byte order - AEP less than 1.0.30
//    dot->setJoinByteOrder(mDot::LSB);
    dot->setJoinByteOrder(mDot::MSB);       // This is default for > 1.0.30 Conduit

    // Set Spreading Factor, higher is lower data rate, smaller packets but longer range
    // Lower is higher data rate, larger packets and shorter range.
    logInfo("Set SF");
    if((ret = dot->setTxDataRate( LORA_SF )) != mDot::MDOT_OK) {
        logError("Failed to set SF %d:%s", ret, mDot::getReturnCodeString(ret).c_str());
    }

    logInfo("Set TxPower");
    if((ret = dot->setTxPower( LORA_TXPOWER )) != mDot::MDOT_OK) {
        logError("Failed to set Tx Power %d:%s", ret, mDot::getReturnCodeString(ret).c_str());
    }

    logInfo("Set Public mode");
    if((ret = dot->setPublicNetwork(true)) != mDot::MDOT_OK) {
        logError("failed to set Public Mode %d:%s", ret, mDot::getReturnCodeString(ret).c_str());
    }

    logInfo("Set MANUAL Join mode");
    if((ret = dot->setJoinMode(mDot::MANUAL)) != mDot::MDOT_OK) {
        logError("Failed to set MANUAL Join Mode %d:%s", ret, mDot::getReturnCodeString(ret).c_str());
    }

    logInfo("Set Ack");
    // 1 retries on Ack, 0 to disable
    if((ret = dot->setAck( LORA_ACK)) != mDot::MDOT_OK) {
        logError("Failed to set Ack %d:%s", ret, mDot::getReturnCodeString(ret).c_str());
    }

//    Not applicable for 868MHz in EU
    if ((ret = dot->setFrequencySubBand(config_frequency_sub_band)) != mDot::MDOT_OK) {
        initStatus = false;
        logError("failed to set frequency sub band", ret, mDot::getReturnCodeString(ret).c_str());
    }

    logInfo("Set Network Address");
    if ((ret = dot->setNetworkAddress(networkAddr)) != mDot::MDOT_OK) {
        logError("Failed to set Network Address %d:%s", ret, mDot::getReturnCodeString(ret).c_str());
    }

    logInfo("Set Data Session Key");
    if ((ret = dot->setDataSessionKey(nwkSKey)) != mDot::MDOT_OK) {
        logError("Failed to set Data Session Key %d:%s", ret, mDot::getReturnCodeString(ret).c_str());
    }

    logInfo("Set Network Session Key");
    if ((ret = dot->setNetworkSessionKey(nwkSKey)) != mDot::MDOT_OK) {
        logError("Failed to set Network Session Key %d:%s", ret, mDot::getReturnCodeString(ret).c_str());
    }

    logInfo("Saving Config");
    // Save config
    if (! dot->saveConfig()) {
        logError("failed to save configuration");
    }

    // Display what is set
    std::vector<uint8_t> tmp = dot->getNetworkSessionKey();
    pc.printf("Network Session Key: ");
    pc.printf("%s\r\n", mts::Text::bin2hexString(tmp, " ").c_str());

    tmp = dot->getDataSessionKey();
    pc.printf("Data Session Key: ");
    pc.printf("%s\r\n", mts::Text::bin2hexString(tmp, " ").c_str());

    pc.printf("Device ID ");
    std::vector<uint8_t> deviceId;
    deviceId = dot->getDeviceId();
    for (std::vector<uint8_t>::iterator it = deviceId.begin() ; it != deviceId.end(); ++it)
        pc.printf("%2.2x",*it );
    pc.printf("\r\n");

    std::vector<uint8_t> netAddress;

    pc.printf("Network Address ");
    netAddress = dot->getNetworkAddress();
    for (std::vector<uint8_t>::iterator it = netAddress.begin() ; it != netAddress.end(); ++it)
        pc.printf("%2.2x",*it );

    pc.printf("\r\n");

    // Display LoRa parameters
    // Display label and values in different colours, show pretty values not numeric values where applicable
    pc.printf("Public Network: %s\r\n", (char*)(dot->getPublicNetwork() ? "Yes" : "No") );
    pc.printf("Frequency: %s\r\n", (char*)mDot::FrequencyBandStr(dot->getFrequencyBand()).c_str() );
    pc.printf("Sub Band: %s\r\n", (char*)mDot::FrequencySubBandStr(dot->getFrequencySubBand()).c_str() );
    pc.printf("Join Mode: %s\r\n", (char*)mDot::JoinModeStr(dot->getJoinMode()).c_str() );
    pc.printf("Join Retries: %d\r\n", dot->getJoinRetries() );
    pc.printf("Join Byte Order: %s\r\n", (char*)(dot->getJoinByteOrder() == 0 ? "LSB" : "MSB") );
    pc.printf("Link Check Count: %d\r\n", dot->getLinkCheckCount() );
    pc.printf("Link Check Thold: %d\r\n", dot->getLinkCheckThreshold() );
    pc.printf("Tx Data Rate: %s\r\n", (char*)mDot::DataRateStr(dot->getTxDataRate()).c_str() );
    pc.printf("Tx Power: %d\r\n", dot->getTxPower() );
    pc.printf("TxWait: %s, ", (dot->getTxWait() ? "Y" : "N" ));
    pc.printf("CRC: %s, ", (dot->getCrc() ? "Y" : "N") );
    pc.printf("Ack: %s\r\n", (dot->getAck() ? "Y" : "N")  );

    logInfo("Joining Network");

    while ((ret = dot->joinNetwork()) != mDot::MDOT_OK) {
        logError("failed to join network [%d][%s]", ret, mDot::getReturnCodeString(ret).c_str());
        wait_ms(dot->getNextTxMs() + 1);
    }

    logInfo("Joined Network");

    // Set the Temperature sesnor resolution, 9 bits is enough and makes it faster to provide a reading.



    char dataBuf[50];
    while( 1 ) {
    int err;
        err = sensor.readData();        
 //       if (err == 0) {
            // Output data as JSON e.g. {"t":21.3}
            temperature = sensor.ReadTemperature(CELCIUS);        
            sprintf(dataBuf, "{\"t\":%4.2f C}", temperature );
            send_data.clear();
            // probably not the most efficent way to do this
            for( int i=0; i< strlen(dataBuf); i++ )
                send_data.push_back( dataBuf[i] );
    
            if ((ret = dot->send(send_data)) != mDot::MDOT_OK) {
                logError("failed to send: [%d][%s]", ret, mDot::getReturnCodeString(ret).c_str());
            } else {
                logInfo("send data: %s", Text::bin2hexString(send_data).c_str());
                logError("send data: %d %s", err, Text::bin2hexString(send_data).c_str());   
            }
//        } else {
//                logError("failed to read Temp: [%d][%s]", err, mDot::getReturnCodeString(err).c_str());
       
 //       }       



        // Should  sleep here and wakeup after a set 5 minute interval.
        uint32_t sleep_time = MAX((dot->getNextTxMs() / 1000), 60);
        logInfo("going to sleep for %d seconds", sleep_time);

        // go to sleep and wake up automatically sleep_time seconds later
        dot->sleep(sleep_time, mDot::RTC_ALARM);
    }

    return 0;
}
