/* Senior Project Bluetooth bicycle speedometer
Author: Michael Galis
This is the main file, it implements both Acceleration Service and Reed Switch
Service to send all the necessary data over bluetooth in organized packages. 
Also there is code to reed from the accelerometer over an I2C connection.
*/

#include "mbed.h"
#include "ble/BLE.h"
#include "MMA8452Q.h"
#include "AcclerationService.h"
#include "ReedSwitchService.h"

#define REED_SWITCH     P0_5

MMA8452Q accel(P0_10, P0_8, 0x1D);  //Declare accel object at I2C pins(P0_0 -> SDA, P0_1 -> SCL, 0x1D -> Accelerometer Address )
Ticker ticker;
float x,y,z;                        //variables assigned values to read x, y, and z accelerometer numbers
//uint16_t x,y,z;
InterruptIn button(REED_SWITCH);

const static char     DEVICE_NAME[] = "BLE_Bike";                                       //Name of BLE Device
static const uint16_t uuid16_list[] = {ReedSwitchService::REED_SWITCH_SERVICE_UUID,     //UUID's of Services
                                       AccelerationService::ACCELERATION_SERVICE_UUID};

enum {                                                  //Different states of the reed switch
    RELEASED = 0,
    PRESSED,
    IDLE
};
static uint8_t reedSwitchState = IDLE;                  //Reed switch is initially idle
static ReedSwitchService *reedSwitchServicePtr;
static AccelerationService *accelerationServicePtr;
static volatile bool  triggerSensorPolling = false;


void reedSwitchPressedCallback(void)            //Change the Reed switch state to Pressed
{
    reedSwitchState = PRESSED;  
}

void reedSwitchReleasedCallback(void)           //Change the Reed switch state to Released
{
    reedSwitchState = RELEASED;
}

void disconnectionCallback(const Gap::DisconnectionCallbackParams_t *params)
{
    BLE::Instance().gap().startAdvertising();
}

/**
 * This function is called when the ble initialization process has failed
 */
void onBleInitError(BLE &ble, ble_error_t error)
{
}

/**
 * Callback triggered when the ble initialization process has finished
 */
void bleInitComplete(BLE::InitializationCompleteCallbackContext *params)
{
    BLE&        ble   = params->ble;
    ble_error_t error = params->error;

    if (error != BLE_ERROR_NONE) {
        /* In case of error, forward the error handling to onBleInitError */
        onBleInitError(ble, error);
        return;
    }

    /* Ensure that it is the default instance of BLE */
    if(ble.getInstanceID() != BLE::DEFAULT_INSTANCE) {
        return;
    }

    ble.gap().onDisconnection(disconnectionCallback);

    /* Setup primary service */
    reedSwitchServicePtr = new ReedSwitchService(ble, false /* initial value for button pressed */);
    accelerationServicePtr = new AccelerationService(ble);

    /* setup advertising */
    ble.gap().accumulateAdvertisingPayload(GapAdvertisingData::BREDR_NOT_SUPPORTED | GapAdvertisingData::LE_GENERAL_DISCOVERABLE);
    ble.gap().accumulateAdvertisingPayload(GapAdvertisingData::COMPLETE_LIST_16BIT_SERVICE_IDS, (uint8_t *)uuid16_list, sizeof(uuid16_list));
    ble.gap().accumulateAdvertisingPayload(GapAdvertisingData::COMPLETE_LOCAL_NAME, (uint8_t *)DEVICE_NAME, sizeof(DEVICE_NAME));
    ble.gap().setAdvertisingType(GapAdvertisingParams::ADV_CONNECTABLE_UNDIRECTED);
    ble.gap().setAdvertisingInterval(100); /* 1000ms. */
    ble.gap().startAdvertising();

}

void periodicCallback(void)
{

    /* Note that the periodicCallback() executes in interrupt context, so it is safer to do
     * heavy-weight sensor polling from the main thread. */
    triggerSensorPolling = true;
}

int main(void)
{
    button.fall(reedSwitchPressedCallback);
    button.rise(reedSwitchReleasedCallback);
    
    BLE &ble = BLE::Instance();
    ble.init(bleInitComplete);
    
    accel.init();
    ticker.attach(periodicCallback,0.2);
    
    
    /* SpinWait for initialization to complete. This is necessary because the
     * BLE object is used in the main loop below. */
    while (ble.hasInitialized()  == false) { /* spin loop */ }
    
    
    while (true) {
        if (reedSwitchState != IDLE) {                                      //When the Reed Switch changes states
            reedSwitchServicePtr->updateReedSwitchState(reedSwitchState);   //update the value over ble
            reedSwitchState = IDLE;
        }
        if (triggerSensorPolling && ble.getGapState().connected)
        {
            triggerSensorPolling = false;
            x=accel.readX(); 
            y=accel.readY();
            z=accel.readZ();
     
            accelerationServicePtr->updateXData(x);
            accelerationServicePtr->updateYData(y);
            accelerationServicePtr->updateZData(z);
            //accelerationServicePtr->updateALLState(x,y,z);
        }
        ble.waitForEvent();
    }
}
