#include "IM315.h"

IM315::IM315(PinName tx, PinName rx) :device(tx, rx)
{   
    is_axis_initialized = false;

    buffer_head = -1;
    device.baud(38400);
    device.attach(this, &IM315::data_receive_isr, Serial::RxIrq);

    error_count = 0;
    error_id = NoDataError;
    error_timer.start();
}

void IM315::data_receive_isr()
{
    char data, *ptr;
    int i;

    while (device.readable())
    {
        data = device.getc();

        if (data == 0x3A)
        {
            buffer_head = 0;
        }

        if (buffer_head < 17 && buffer_head != -1)
        {
            if (data != ',')
            {
                if (buffer_head == 0)
                {
                    received_buffer[buffer_head] = 0x3A;
                }   
                else
                {
                    if (buffer_head % 2 == 1)
                    {
                        received_buffer[buffer_head / 2 + 1] = strtol(&data, &ptr, 16) << 4;
                    }
                    else if (buffer_head % 2 == 0)
                    {
                        received_buffer[buffer_head / 2] |= strtol(&data, &ptr, 16);
                    }
                }
                buffer_head++;
            }
        }
        
        if (buffer_head == 17)
        {
            if (received_buffer[0] == 0x3A)
            {
                for (i = 0; i < 8; i++)
                {
                    received_data[i] = received_buffer[i];
                }
                
                if (!is_axis_initialized)
                {
                    initialize_axis();
                    is_axis_initialized = true;
                }

                error_count = 0;
                error_id = NoError;
                error_timer.reset();
            }
            else
            {
                buffer_head = -1;
                error_id = InvalidDataError;
            }
        }
    }
}

uint8_t IM315::get_all_switch()
{
    return received_data[1];
}

bool IM315::get_switch(int num)
{
    bool status = false;
    if (num < 8 && num >= 0)
    {
        if (get_all_switch() & (1 << num))
        {
            status = true;
        }
        else
        {
            status = false;
        }
    }
    else
    {
        status = false;
    }

    return status;
}

int IM315::get_raw_axis(AxisId id)
{
    uint8_t *data;
    int value;

    if((int)id < 3 && (int)id >= 0)
    {
        data = &received_data[(int) id * 2 + 2];
        value = (data[0] << 8) | (data[1]);
    }
    else
    {
        value = -1;
    }

    return value;
}

double IM315::get_axis(AxisId id)
{
    double value = 0;
    int raw;
    raw = get_raw_axis(id);

    if(raw != -1)
    {
        value = (double) (raw - axis_center[(int) id]) / 512.0;

        if (value > -axis_threshold && value < axis_threshold)
        {
            value = 0.0;
        }
        else if (value < -1.0)
        {
            value = -1.0;
        }
        else if (value > 1.0)
        {
            value = 1;
        }
    }
    else
    {
        value = 0.0;
    }

    return value;
}

void IM315::initialize_axis(void)
{
    int i;
    for (i = 0; i < 3; i++)
    {
        initialize_axis((AxisId) i);
    }
}

void IM315::initialize_axis(AxisId id)
{
    int raw;
    raw = get_raw_axis(id);

    if(raw != -1)
    {
        axis_center[(int) id] = raw;
    }
}

IM315::ErrorFactor IM315::get_error()
{   
    if (error_timer >= timeout_time)
    {
        error_id = TimeoutError;
    }
    
    return error_id;
}