/** Serial port handler.
 * Copyright (c) 2015 Gabriel Rivas
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */


#ifndef _SERIAL_PORT_HANDLER_H_
#define _SERIAL_PORT_HANDLER_H_

#include "mbed.h"
#include "MessageQueue.h"

/**
 * Serial port handler class that allows sharing data from the serial ports
 * via message queues objects. Each instance of this class is intended to handle
 * a serial port in the system and is suitable for slow serial protocols.
 */
class SerialPortHandler
{
public:
    /**
     * Creates a serial port handler object.
     * @param ps Pointer to a serial port instance object.
     * @param txQueue Message queue pointer from which to get data bytes to send.
     * @param rxQueue Message queue pointer to write bytes received.
     */
    SerialPortHandler(Serial* ps, MessageQueue<uint8_t>* txQueue, MessageQueue<uint8_t>*rxQueue);

    /**
     * Destroys a serial port handler object.
     */
    ~SerialPortHandler();

public:
    /**
     * Transmits data bytes stored in the transmit queue through the serial port.
     */
    void transmitPacket();

    /**
     * Receive bytes from the serial port and store it into the receive queue.
     */
    void receivePacket();

private:
    /**
     * Receive bytes from the serial port and store it into the receive queue.
     */
    Serial* m_ps;

    /**
     * Message queue from which to get data bytes to send.
     */
    MessageQueue<uint8_t>* m_txQueue;

    /**
     * Message queue pointer to write bytes received.
     */
    MessageQueue<uint8_t>* m_rxQueue;
};

#endif
