/* Modbus RTU Slave Library
 * Copyright (c) 2015 Gabriel Rivas
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#include "ModbusSlaveRTU.h"
#include "MessageQueue.h"

#define MODBUS_FRAME_OVERHEAD_BYTES (5)
#define MODBUS_ID_FUNCTION_SIZE (3)
#define CRC_POSITION1 (m_frame.mByteCount + MODBUS_ID_FUNCTION_SIZE)
#define CRC_POSITION2 (m_frame.mByteCount + MODBUS_ID_FUNCTION_SIZE + 1)

ModbusSlaveRTU::ModbusSlaveRTU( uint8_t id,
                                MessageQueue<uint8_t>* txQueue,
                                MessageQueue<uint8_t>* rxQueue,
                                ThreadSafeArray_t *coilRegisters,
                                ThreadSafeArray_t *inputRegisters,
                                ThreadSafeArray_t *holdingRegisters)
{
//   assert(txQueue != NULL);
    m_bTxQueue = txQueue;

//    assert(rxQueue != NULL);
    m_bRxQueue = rxQueue;

    m_state = WAIT_REQUEST_MESSAGE;

    m_address = id;
    m_status = LISTENING;

//   assert(coilRegisters != NULL);
    m_memoryMap.coilRegisters = coilRegisters;

//   assert(inputRegisters != NULL);
    m_memoryMap.inputRegisters = inputRegisters;

//   assert(holdingRegisters != NULL);
    m_memoryMap.holdingRegisters = holdingRegisters;
}

void ModbusSlaveRTU::trigger(void)
{
    m_state = GET_RECEIVED_BYTES;
}

ModbusSlaveRTU::MbStatusTypes_t ModbusSlaveRTU::getStatus(void)
{
    return m_status;
}

void ModbusSlaveRTU::FSM(void)
{
    uint32_t count;

    switch(m_state) {
        case WAIT_REQUEST_MESSAGE:
            m_status = LISTENING;

            break;
        case GET_RECEIVED_BYTES:
            count = m_bRxQueue->getWriteIndex();
            m_status = PROCESSING;
            if ((count > 0) && (count < MODBUS_MAX_LEN)) {
                for (uint32_t i = 0; i < count; i++) {
                    uint8_t data = m_bRxQueue->read();
                    m_receivedMessage[i] = data;
                }
                m_state = BUILD_RESPONSE_MESSAGE;
                m_bRxQueue->reset();
            } else {
                m_state = WAIT_REQUEST_MESSAGE;
            }
            break;
        case BUILD_RESPONSE_MESSAGE:
            if (buildResponse() == NO_ERROR) {
                m_status = SENDING;
                m_state = TRANSMIT_RESPONSE_MESSAGE;
            } else {
                m_status = LISTENING;
                m_state = WAIT_REQUEST_MESSAGE;
            }
            break;
        case TRANSMIT_RESPONSE_MESSAGE:
            for (uint32_t i = 0; i < m_frame.mSize; i++) {
                m_bTxQueue->write(m_responseMessage[i]);
            }

            m_status = LISTENING;
            m_state = WAIT_REQUEST_MESSAGE;
            break;
        default:
            break;
    }

}

ModbusSlaveRTU::MbReceivedMessageErrors_t ModbusSlaveRTU::buildResponse(void)
{
    MbReceivedMessageErrors_t result = NO_ERROR;

    switch(m_receivedMessage[FUNCTION]) {
        case READ_COIL_STATUS:
            result = readCoilRegistersHandler();
            break;
        case READ_HOLDING_REGISTERS:
            result = readHoldingRegistersHandler();
            break;
        case FORCE_SINGLE_COIL:
            break;
        default:
            break;
    }

    return result;
}

uint8_t ModbusSlaveRTU::checkMessageCRC(uint8_t *  data)
{
    uint8_t function = data[FUNCTION];
    uint8_t byteCount;
    uint16_t crcVal;
    uint16_t crcCalc;
    uint8_t result;

    if (function != READ_EXCEPTION_STATUS) {
        byteCount = 8;
        crcVal = data[byteCount-1];
        crcVal = (crcVal << 8) | data[byteCount - 2];
        crcCalc  = getCRC(data, byteCount - 2);

        if (crcVal == crcCalc) {
            result = 1;
        } else {
            result = 0;
        }
    } else {
        result = 0;
    }

    return result;
}

uint16_t ModbusSlaveRTU::getCRC(uint8_t *  data,uint8_t len)
{
    uint16_t crc = 0xFFFF;
    uint8_t i;

    while (len--) {
        i = *data++;
        *(uint8_t *)&crc  ^= i;

        for (i = 0; i != 8; i++) {
            if (crc & 0x0001) {
                crc = (crc >> 1);
                crc ^= 0xA001;
            } else {
                crc = (crc >> 1);
            }
        }
    }
    return crc;
}

ModbusSlaveRTU::MbReceivedMessageErrors_t ModbusSlaveRTU::readHoldingRegistersHandler(void)
{
    uint8_t i;
    MbReceivedMessageErrors_t result = NO_ERROR;

    m_frame.mAddress = m_receivedMessage[ADDRESS];
    m_frame.mFunction  =  static_cast<MbFunctionCodes_t>(m_receivedMessage[FUNCTION]);

    result = checkReceivedMessageErrors(m_memoryMap.holdingRegisters);

    if (result < ADDRESS_ERROR) {
        m_frame.mByteCount = ((m_receivedMessage[REGISTER_COUNT] << 8) | (m_receivedMessage[REGISTER_COUNT + 1] & 0xFF));
        m_frame.mByteCount <<= 1;


        if ((m_frame.mByteCount + m_frame.mOffset)  > m_memoryMap.holdingRegisters->length)
            m_frame.mByteCount = m_memoryMap.holdingRegisters->length - m_frame.mOffset;

        (m_memoryMap.holdingRegisters)->mutex.lock();

        m_frame.mSize      = m_frame.mByteCount + MODBUS_FRAME_OVERHEAD_BYTES;

        for(i = 0; i < m_frame.mByteCount; i++) {
            m_responseMessage[i + DATA_START] = *((m_memoryMap.holdingRegisters)->data + i + m_frame.mOffset);
        }

        appendHeaderAndTailToMessage();

        (m_memoryMap.holdingRegisters)->mutex.unlock();
    }

    return result;
}

ModbusSlaveRTU::MbReceivedMessageErrors_t ModbusSlaveRTU::readCoilRegistersHandler(void)
{
    MbReceivedMessageErrors_t result = NO_ERROR;
    uint16_t initRegister;
    uint16_t initCoil;
    uint16_t endRegister;
    uint16_t i = 0;
    uint16_t j = 0;
    uint8_t mask;
    uint8_t residual_bits;

    m_frame.mAddress = m_receivedMessage[ADDRESS];
    m_frame.mFunction  =  static_cast<MbFunctionCodes_t>(m_receivedMessage[FUNCTION]);

    result = checkReceivedMessageErrors(m_memoryMap.coilRegisters);

    if (result < ADDRESS_ERROR) {
        m_frame.mByteCount = ((m_receivedMessage[REGISTER_COUNT] << 8) | (m_receivedMessage[REGISTER_COUNT + 1] & 0xFF));

        residual_bits = (m_frame.mByteCount) % 8;
        m_frame.mByteCount >>= 3;

        if (residual_bits > 0)
            m_frame.mByteCount += 1;

        initRegister = m_frame.mOffset >> 3;
        initCoil = m_frame.mOffset % 8;
        endRegister = initRegister  + (m_frame.mByteCount);

        if (m_frame.mByteCount == 0)
            endRegister -= 1;

        mask = (1 << residual_bits) - 1;

        if ((m_frame.mByteCount + m_frame.mOffset)  > m_memoryMap.holdingRegisters->length)
            m_frame.mByteCount = m_memoryMap.holdingRegisters->length - m_frame.mOffset;

        m_frame.mSize      = m_frame.mByteCount + MODBUS_FRAME_OVERHEAD_BYTES;

        m_memoryMap.coilRegisters->mutex.lock();

        //Process coil data to generate the response message.
        for (i = initRegister; i < endRegister; i++) {
            m_responseMessage[j++ + DATA_START] = (m_memoryMap.coilRegisters->data[i] >> initCoil) |
                                                  (m_memoryMap.coilRegisters->data[i+1] << (8 - initCoil));
        }

        m_responseMessage[j + DATA_START] = (m_memoryMap.coilRegisters->data[i] >> initCoil) |
                                            (m_memoryMap.coilRegisters->data[i+1] << (8 - initCoil));
        m_responseMessage[j + DATA_START] &= mask;

        appendHeaderAndTailToMessage();

        m_memoryMap.coilRegisters->mutex.unlock();
    }

    return result;
}

void ModbusSlaveRTU::appendHeaderAndTailToMessage(void)
{
    uint16_t mCRC;

    m_responseMessage[ADDRESS]     = m_frame.mAddress;
    m_responseMessage[FUNCTION]    = m_frame.mFunction;
    m_responseMessage[BYTE_COUNT]  = m_frame.mByteCount;

    mCRC = getCRC(m_responseMessage,m_frame.mByteCount + DATA_START);

    m_responseMessage[CRC_POSITION1] = (uint8_t) mCRC & 0xFF;
    m_responseMessage[CRC_POSITION2] = (uint8_t) (mCRC >> 8) & 0xFF;
}

ModbusSlaveRTU::MbReceivedMessageErrors_t ModbusSlaveRTU::checkReceivedMessageErrors(ThreadSafeArray_t *array)
{
    MbReceivedMessageErrors_t result = NO_ERROR;

    //Check if Modbus Address is correct
    if (m_frame.mAddress != m_address) {
        result = ADDRESS_ERROR;
    }

    //Verify message errors
    if (checkMessageCRC(m_receivedMessage) == 0) {
        result = CRC_ERROR;
    }

    //Verify offset
    m_frame.mOffset    = ((m_receivedMessage[BYTE_OFFSET] << 8) | (m_receivedMessage[BYTE_OFFSET + 1] & 0xFF));
    if ((m_frame.mOffset > array->length - 1) ||
            (m_frame.mOffset > MODBUS_MAX_LEN-1)) {
        result = OFFSET_ERROR;
    }

    return result;
}
